/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

// Derived from forum post http://mbed.org/forum/helloworld/topic/2303

#ifndef AJK_SIMPLESERVOCONTROL_H
#define AJK_SIMPLESERVOCONTROL_H

#include "mbed.h"
#include "SimpleRCservos.h"

namespace AjK {

/** SimpleServoControl
 *
 * Very simple servo controller class.
 * 
 * @see http://mbed.org/forum/helloworld/topic/2303
 */
class SimpleServoControl {

protected:
    double _current_position;
    double _desired_position;
    double _step;
    int    _poll_interval;
    SimpleRCservos::Servo _motor;
    SimpleRCservos *_servos;
    Ticker _servo_poll;
    
public:

    // Constructor.
    SimpleServoControl(SimpleRCservos *servos, SimpleRCservos::Servo motor, double min = -90.0, double max = +90.0) 
    { 
        _servos = servos;
        _motor = motor; 
        _servos->setLimits(_motor, min, max); // define logical limits.
        _servos->enable(_motor); // Enable the PWM outout.
        _current_position = 0.0;
        _desired_position = 0.0;
        _step = 1.0;
        _poll_interval = 10000; // 100ms.
        _servo_poll.attach_us(this, &SimpleServoControl::poll, _poll_interval); 
    }
    
    void poll(void) 
    {
        if (_desired_position > _current_position) {
            _current_position += _step;
            // Don't allow the servo to oscillate around _desired_position.
            if (_desired_position < _current_position) { 
                _current_position = _desired_position; 
            }
            _servos->position(_motor, _current_position);                        
        }
        else if (_desired_position < _current_position) {
            _current_position -= _step;
            // Don't allow the servo to oscillate around _desired_position.
            if (_desired_position > _current_position) {
                _current_position = _desired_position; 
            }
            _servos->position(_motor, _current_position);            
        }
    }
    
    void position(double position = 90.0)  // spins the servo 90º to the left
    {
        _desired_position = position;
    }
    
    void setStep(double d) 
    { 
        _step = d; 
    }
    
    double getStep(void) 
    { 
        return _step; 
    } 
    
    void setPollInterval(int i) 
    { 
        _poll_interval = i;
        _servo_poll.detach();
        _servo_poll.attach_us(this, &SimpleServoControl::poll, _poll_interval); 
    }
    
    int getPollInterval(void) 
    { 
        return _poll_interval; 
    }
    
};

}; // namespace AjK ends.

using namespace AjK;

#endif
