/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifndef AJK_RICKER_H
#define AJK_RICKER_H

#include "mbed.h"
#include <list>

namespace AjK {

class Ricker; // forward ref.

// Ricker controller.
/** RickerSys
 *
 * Base system, only a single instance of this is allowed. It's used to manage
 * the RIT IRQ and the chain of "Rickers" attached.
 *
 * Note, unlike an Mbed Ticker the Ricker uses a 1ms resolution for it's timer.
 * Therefore teh smallest unit of time that a Ricker can measure is 1ms. If 
 * you need sub-millisecond then use an Mbed Ticker.
 */
class RickerSys {
protected:
    list<Ricker *> rickers;    
public:
    void init(void);
    RickerSys();
    void isr(void);    
    void addTicker(Ricker *t);    
    void delTicker(Ricker *t);    
};

extern RickerSys _rickerSys;

// Ricker instance object.
/** Ricker
 *
 * A class used to create multiple Rickers. These attach themselves to the RickerSys
 * main controller. 
 *
 * <b>Note</b>, unlike an Mbed Ticker, a Ricker can only make callbacks down to 1ms.
 *
 * @code
 * #include "mbed.h"
 * #include "Ricker.h"
 * 
 * DigitalOut led1(LED1);
 *
 * Ricker r1;
 *  
 * void cb1(void) { led1 = !led1; }
 * 
 * int main() {
 * 
 *     r1.attach(&cb1, 0.25);
 *     
 *     while(1) { }
 * }
 * @endcode
 */
class Ricker {
protected:
    FunctionPointer callback;
    uint32_t counter;
    uint32_t reload;
    
    /** tick
     * @internal
     * Called by the RickerSys controller every 1ms.
     */
    void tick(void) {
        if (counter) {
            counter--;
            if (counter == 0) {
                counter = reload;
                callback.call();
            }        
        }
    }
    
public:
    friend class RickerSys;
    
    /** Constructor
     */
    Ricker() { 
        counter = 0;
        _rickerSys.addTicker(this);
    }
    
    /** Destructor
     */
    ~Ricker() { 
        _rickerSys.delTicker(this);
    }
    
    /** detach
     *
     * Remove the callback from the Ricker.
     */
    void detach(void) {        
        callback.attach();
    }
    
    /** attach_ms
     *
     * Attach a C style function pointer callback.
     *
     * @param fptr A C style function pointer.
     * @param uint32_t u The number of milliseconds to call at.
     */
    void attach_ms(void (*fptr)(void), uint32_t u) {
        counter = reload = u;
        callback.attach(fptr);
    }
    
    /** attach_ms
     *
     * Attach a C++ style functor object/method callback.
     *
     * @param tptr A C++ style object pointer.
     * @param mptr A C++ style method pointer.
     * @param uint32_t u The number of milliseconds to call at.
     */
    template<typename T>
    void attach_ms(T* tptr, void (T::*mptr)(void), uint32_t u) {  
        if((mptr != NULL) && (tptr != NULL)) {
            counter = reload = u;
            callback.attach(tptr, mptr);         
        }        
    }
    
    /** attach
     *
     * Attach a C style function pointer callback.
     *
     * @param fptr A C style function pointer.
     * @param double d The number of seconds to call at.
     */
    void attach(void (*fptr)(void), double d) {
        counter = reload = (uint32_t)( d * 1000.0 );
        callback.attach(fptr);
    }
    
    /** attach
     *
     * Attach a C++ style functor object/method callback.
     *
     * @param tptr A C++ style object pointer.
     * @param mptr A C++ style method pointer.
     * @param double d The number of seconds to call at.
     */
    template<typename T>
    void attach(T* tptr, void (T::*mptr)(void), double d) {  
        if((mptr != NULL) && (tptr != NULL)) {
            counter = reload = (uint32_t)( d * 1000.0 );
            callback.attach(tptr, mptr);         
        }        
    }
    
};

}; // namespace AjK ends.

using namespace AjK;

#endif
