/*
    Copyright (c) 2010 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifndef MODADF4360_H
#define MODADF4360_H

#include "mbed.h"

#define GET_CNTL_LATCH(pos,mask)    return(control_latch>>pos)&mask
#define SET_CNTL_LATCH(pos,mask,n)  control_latch&=~(mask<<pos);control_latch|=((n&mask)<<pos)
#define GET_N_LATCH(pos,mask)       return(counterNlatch>>pos)&mask
#define SET_N_LATCH(pos,mask,n)     counterNlatch&=~(mask<<pos);counterNlatch|=((n&mask)<<pos)
#define GET_R_LATCH(pos,mask)       return(counterRlatch>>pos)&mask
#define SET_R_LATCH(pos,mask,n)     counterRlatch&=~(mask<<pos);counterRlatch|=((n&mask)<<pos)

/** @defgroup Control_Latch The Control Latch */
/** @defgroup Counter_N_Latch The Counter N Latch */
/** @defgroup Counter_R_Latch The Counter R Latch */

/**
 * @author Andy Kirkham
 * @see Control_Latch
 * @see Counter_N_Latch
 * @see Counter_R_Latch
 *
 * <b>MODADF4360</b> Test library
 *
 * Standard example:
 * @code
 * #include "mbed.h"
 * #include "MODADF4360.h"
 *
 * DigitalOut led1(LED1);
 * ADF4360 adf(p5, p6, p7, p8);
 *
 * int main() {
 *
 *     // Setup all the parts of the Control Latch
 *     adf.prescalerValue(1);
 *     adf.cpGain(1);
 *     //... set all the parts you need, then...
 *     
 *     // Once all the parts ate set, write them to the device.
 *     adf.controlLatchWrite();
 *
 *     // repeat for N and R latches.
 *
 *     while(1) {
 *         led1 = !led1;
 *         wait(1);
 *     }
 * }
 * @endcode
 */
class ADF4360 {
public:

    ADF4360(PinName mosi, PinName miso, PinName sclk, PinName le) {
        _le = new DigitalOut(le);
        _le->write(1);    
        _ssp = new SPI(mosi, miso, sclk);
        _ssp->format(8,0);
        _ssp->frequency(1000000);    
    };

    ~ADF4360() { delete(_le); delete(_ssp); };

    /** 
     * @ingroup Control_Latch 
     */
    int  prescalerValue(void)       { GET_CNTL_LATCH(22,3); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void prescalerValue(int i)      { SET_CNTL_LATCH(22,3UL,i); }

    /** 
     * @ingroup Control_Latch 
     */
    int  powerDown2(void)           { GET_CNTL_LATCH(21,1); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void powerDown2(int i)          { SET_CNTL_LATCH(21,1UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  powerDown1(void)           { GET_CNTL_LATCH(20,1); } 
    
    /** 
     * @ingroup Control_Latch 
     */
    void powerDown1(int i)          { SET_CNTL_LATCH(20,1UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  currentSetting2(void)      { GET_CNTL_LATCH(17,7); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void currentSetting2(int i)     { SET_CNTL_LATCH(17,7UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  currentSetting1(void)      { GET_CNTL_LATCH(14,7); } 
    
    /** 
     * @ingroup Control_Latch 
     */
    void currentSetting1(int i)     { SET_CNTL_LATCH(14,7UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  outputPowerLevel(void)     { GET_CNTL_LATCH(12,3); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void outputPowerLevel(int i)    { SET_CNTL_LATCH(12,3UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  muteTillLockDetect(void)   { GET_CNTL_LATCH(11,1); } 
    
    /** 
     * @ingroup Control_Latch 
     */
    void muteTillLockDetect(int i)  { SET_CNTL_LATCH(11,1UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  cpGain(void)               { GET_CNTL_LATCH(10,1); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void cpGain(int i)              { SET_CNTL_LATCH(10,1UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  cpOutput(void)             { GET_CNTL_LATCH(9,1); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void cpOutput(int i)            { SET_CNTL_LATCH(9,1UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  phaseDetectPol(void)       { GET_CNTL_LATCH(8,1); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void phaseDetectPol(int i)      { SET_CNTL_LATCH(8,1UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  muxControl(void)           { GET_CNTL_LATCH(5,7); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void muxControl(int i)          { SET_CNTL_LATCH(5,7UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  counterReset(void)         { GET_CNTL_LATCH(4,1); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void counterReset(int i)        { SET_CNTL_LATCH(4,1UL,i); }
    
    /** 
     * @ingroup Control_Latch 
     */
    int  corePowerLevel(void)       { GET_CNTL_LATCH(2,3); }
    
    /** 
     * @ingroup Control_Latch 
     */
    void corePowerLevel(int i)      { SET_CNTL_LATCH(2,3UL,i); }
    
    void controlLatchWrite(void) {
        _le->write(0);
        _ssp->write((control_latch >> 16) & 0xFF);
        _ssp->write((control_latch >> 8) & 0xFF);
        _ssp->write((control_latch & 0xFF) & ~(3UL << 0));
        _le->write(1);
    }

    /** 
     * @ingroup Counter_N_Latch 
     */
    int  divideBy2Select(void)      { GET_N_LATCH(23,1); }
    
    /** 
     * @ingroup Counter_N_Latch 
     */
    void divideBy2Select(int i)     { SET_N_LATCH(23,1UL,i); }

    /** 
     * @ingroup Counter_N_Latch 
     */
    int  divideBy2(void)            { GET_N_LATCH(22,1); }
    
    /** 
     * @ingroup Counter_N_Latch 
     */
    void divideBy2(int i)           { SET_N_LATCH(22,1UL,i); }
    
    /** 
     * @ingroup Counter_N_Latch 
     */
    int  nCPGain(void)              { GET_N_LATCH(21,1); }
    
    /** 
     * @ingroup Counter_N_Latch 
     */
    void nCPGain(int i)             { SET_N_LATCH(21,1UL,i); }
    
    /** 
     * @ingroup Counter_N_Latch 
     */
    int  counterB(void)             { GET_N_LATCH(8,0x1FFF); }
    
    /** 
     * @ingroup Counter_N_Latch 
     */
    void counterB(int i)            { SET_N_LATCH(8,0x1FFFUL,i); }
    
    /** 
     * @ingroup Counter_N_Latch 
     */
    int  counterA(void)             { GET_N_LATCH(2,0x1F); }
    
    /** 
     * @ingroup Counter_N_Latch 
     */
    void counterA(int i)            { SET_N_LATCH(2,0x1FUL,i); }
    
    void counterNWrite(void) {
        _le->write(0);
        _ssp->write((counterNlatch >> 16) & 0xFF);
        _ssp->write((counterNlatch >> 8) & 0xFF);
        _ssp->write(((counterNlatch & 0xFF) & ~(0x3CUL << 0)) | (2UL << 0));
        _le->write(1);
    }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    int  bandSelectClock(void)      { GET_R_LATCH(20,3); }
    /** 
     * @ingroup Counter_R_Latch 
     */
    void bandSelectClock(int i)     { SET_R_LATCH(20,3UL,i); }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    int  testModeBit(void)          { GET_R_LATCH(19,1); }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    void testModeBit(int i)         { SET_R_LATCH(19,1UL,i); }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    int  lockDetect(void)           { GET_R_LATCH(18,1); }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    void lockDetect(int i)          { SET_R_LATCH(18,1UL,i); }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    int  antiBacklash(void)         { GET_R_LATCH(16,3); }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    void antiBacklash(int i)        { SET_R_LATCH(16,3UL,i); }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    int  counterRef(void)           { GET_R_LATCH(2,0x3FFF); }
    
    /** 
     * @ingroup Counter_R_Latch 
     */
    void counterRef(int i)          { SET_R_LATCH(2,0x3FFFUL,i); }
    
    void counterRWrite(void) {
        _le->write(0);
        _ssp->write((counterRlatch >> 16) & 0xFF);
        _ssp->write((counterRlatch >> 8) & 0xFF);
        _ssp->write(((counterRlatch & 0xFF) & ~(0x3CUL << 0)) | (1UL << 0));
        _le->write(1);
    }
    
protected:

    uint32_t control_latch;
    uint32_t counterNlatch;
    uint32_t counterRlatch;
    
    SPI         *_ssp;
    DigitalOut  *_le;
};
   

#endif
