/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifdef MAX3100_EXAMPLE4_COMPILE

/*
 * This is the same as example1.h but shows how to attach a usre defined callback
 * function that is called by the ISR.
 *
 * Connecting up the MAX3100 for this test program. Note, to form a "loopback"
 * the MAX3100 TX pin (13) is connected to the RX pin (12). Don't forget thwe Xtal
 * and power pins that are not shown here. Although I do PullUp mode on the IRQ pin
 * I still needed a real external pull up resistor on the IRQ line. You may need one
 * also.
 *                                    ____________
 *                                   /            \
 * Mbed MOSI p5 |-----------------> 1| Din     TX | 13 ------\
 *      MISO p6 |-----------------> 2| Dout       |          |
 *      SCLK p7 |-----------------> 3| Sclk    RX | 12 ------/
 *           p8 |-----------------> 4| CS         |
 *           p9 |-----------------> 5| IRQ        | MAX3100
 *                      +5v ------> 6| SHTD       | Xtal and PWR not shown.
 *                                   \____________/
 */

#include "mbed.h"
#include "MAX3100.h"

Serial  pc(USBTX, USBRX);
MAX3100 max(p5, p6, p7, p8, p9);

// Used to count RX interrupts. Must be declared as
// volatile as it's used in MyIsrCallback() which is 
// in ISR context and main() which is user context.
// See http://mbed.org/users/AjK/notebook/regarding-interrupts-use-and-blocking/
volatile int isrCounter;

// User function called when an IRQ from the MAX3100 is activated.
void MyIsrCallback( int isrType ) {
    // Only count RX IRQs.
    if ( isrType & MAX3100::ISR_RX ) {
        isrCounter++;
    }
}

int main() {
    
    isrCounter = 0;
    
    // Set the PC USB serial baud rate.
    pc.baud(115200);
    
    max.attach_isr_user( MyIsrCallback );
    
    max.enableRxIrq();
    max.enableTxIrq();
    
    max.printf("\nHello World.\n");
    
    // Any byte received on the "USB serial port" is sent to the MAX3100 device.
    // Any byte received by the MAX3100 device is sent to the "USB serial port".
    while (1) {
        if (pc.readable()) {
            int c = pc.getc();
            max.putc(c);            
        }
        
        if ( isrCounter > 0 ) {
            pc.putc( max.getc() );
            isrCounter--;
        }
    }    
}

#endif
