/****************************************************************************
 * Product: LIB17 Open Source Library
 *
 *                    Steller Technologies Limited
 *                    ----------------------------
 *
 * Copyright (C) 2002-2011 Steller Technologies Limited. All rights reserved.
 *
 * This software may be distributed and modified under the terms of the GNU
 * General Public License version 2 (GPL) as published by the Free Software
 * Foundation and appearing in the file GPL.TXT included in the packaging of
 * this file. Please note that GPL Section 2[b] requires that all works based
 * on this software must also be made publicly available under the terms of
 * the GPL ("Copyleft").
 *
 * Alternatively, this software may be distributed and modified under the
 * terms of Steller Technologies Limited commercial licenses, which expressly
 * supersede the GPL and are specifically designed for licensees interested in
 * retaining the proprietary status of their code.
 *
 ***************************************************************************/

/**
 * @see http://cornflakes.wikidot.com/lib17:core
 * @defgroup API The Lib17 Core API 
 * @defgroup Lib17_Callback Lib17_Callback functions
 */

#ifndef AJK_Lib17_Callback_H
#define AJK_Lib17_Callback_H

#include "LPC17xx.h"

#include <stdint.h>

namespace AjK {

class Lib17_CallbackDummy;

/** Lib17_Callback - Adds callbacks that take and return a 32bit uint32_t data type.
 *
 * The Mbed library supplies a callback using the FunctionPointer object as
 * defined in FunctionPointer.h  However, this callback system does not allow
 * the caller to pass a value to the callback. Likewise, the callback itself
 * cannot return a value.
 *
 * Lib17_Callback allows library objects to implement callbacks using either C
 * style function pointers or C++ style object/method functors.
 *
 * Note, when passing pointers to variables to the callback, if the callback
 * function/method changes that variable's value then it will also change the
 * value the caller sees. If C pointers are new to you, you are strongly
 * advised to read up on the subject. It's pointers that often get beginners
 * into trouble when mis-used.
 *
 */
class Lib17_Callback {

protected:

    //! C callback function pointer.
    uint32_t (*_c_callback)(uint32_t);

    //! C++ callback object/method pointer (the object part).
    Lib17_CallbackDummy  *_obj_callback;

    //! C++ callback object/method pointer (the method part).
    uint32_t (Lib17_CallbackDummy::*_method_callback)(uint32_t);

public:

    /**
     * Used to create callback chains if requirted.
     */
    Lib17_Callback *_next;

    /** Constructor
     */
    Lib17_Callback() { detach(); _next = 0; }

    /** attach - Overloaded attachment function.
     *
     * Attach a C type function pointer as the callback.
     *
     * Note, the callback function prototype must be:-
     * @code
     * uint32_t myCallbackFunction(uint32_t);
     * @endcode
     *
     * @see http://cornflakes.wikidot.com/lib17:core:lib17-callback
     * @ingroup Lib17_Callback
     * @ingroup API 
     *
     * @param A C function pointer to call.
     */
    void attach(uint32_t (*function)(uint32_t) = 0) {
        _c_callback = function;
    }
    
    /** attach - Overloaded attachment function.
     *
     * Attach a C++ type object/method pointer as the callback.
     *
     * Note, the callback method prototype must be:-
     * @code
     * public: uint32_t myCallbackFunction(uint32_t);
     * @endcode
     *
     * @see http://cornflakes.wikidot.com/lib17:core:lib17-callback
     * @ingroup Lib17_Callback
     * @ingroup API 
     *
     * @param A C++ object pointer.
     * @param A C++ method within the object to call.
     */
    template<class T>
    void attach(T* item = 0, uint32_t (T::*method)(uint32_t) = 0) {
        _obj_callback    = (Lib17_CallbackDummy *)item;
        _method_callback = (uint32_t (Lib17_CallbackDummy::*)(uint32_t))method;        
    }

    /** detach - Remove callbacks
     *
     * Detach the pointers to teh callback.
     *
     * @see http://cornflakes.wikidot.com/lib17:core:lib17-callback
     * @ingroup Lib17_Callback
     * @ingroup API 
     *
     */
    void detach(void) {
        _c_callback      = 0;
        _obj_callback    = 0;
        _method_callback = 0;        
    }


    /** call - Overloaded callback initiator.
     *
     * call the callback function.
     *
     * @see http://cornflakes.wikidot.com/lib17:core:lib17-callback
     * @ingroup Lib17_Callback
     * @ingroup API 
     *
     * @param uint32_t The value to pass to the callback.
     * @return uint32_t The value the callback returns.
     */
    uint32_t call(uint32_t arg) {
        if (_c_callback != 0) {
            return (*_c_callback)(arg);
        }
        else {
            if (_obj_callback != 0 && _method_callback != 0) {
                return (_obj_callback->*_method_callback)(arg);
            }
        }
        return (uint32_t)0;
    }

    /** call - Overloaded callback initiator.
     *
     * Call the callback function without passing an argument.
     * The callback itself is passed NULL. Note, the callback
     * prototype should still be <b>uint32_t callback(uint32_t)</b>.
     *
     * @see http://cornflakes.wikidot.com/lib17:core:lib17-callback
     * @ingroup Lib17_Callback
     * @ingroup API 
     *
     * @return uint32_t The value the callback returns.
     */
    uint32_t call(void) {
        if (_c_callback != 0) {
            return (*_c_callback)((uint32_t)0);
        }
        else {
            if (_obj_callback  != 0 && _method_callback != 0) {
                return (_obj_callback->*_method_callback)(0);
            }
        }
        return (uint32_t)0;
    }

};

}; // namespace cf ends

using namespace AjK;

#endif /* AJK_Lib17_Callback_H */

