/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#include "AD9951DSPSERIAL.h"
#include "iomacros.h"

extern Serial pc;

namespace AjK {

class AD9951DSPSERIAL *ad9951dspread_inst = NULL;

extern "C" void _ssp_isr(void) __irq {
    LED3_ON;
    if (ad9951dspread_inst) ad9951dspread_inst->ssp_isr();
}

AD9951DSPSERIAL::AD9951DSPSERIAL()
{
    error("No pin definition supplied\n");
}

AD9951DSPSERIAL::AD9951DSPSERIAL(PinName mosi, PinName miso, PinName sclk, PinName ssel)
{
    ad9951dspread_inst = this;
    LED3_USE;
    LED4_USE;
       
    ssp = NULL;
    
    if (mosi == p5 && miso == p6 && sclk == p7 && ssel == p8) {
        ssp = LPC_SSP1;        
        pc.printf("  Setting SSP1\n");
    }
    else if (mosi == p11 && miso == p12 && sclk == p13 && ssel == p14) {
        ssp = LPC_SSP0;        
        pc.printf("  Setting SSP0\n");
    }
    else {
        error("Incorrect pin definition supplied\n");
    }
 
    init();   
}

void
AD9951DSPSERIAL::init(void)
{
    _dspread_function = NULL;
    _dspread_obj      = NULL;
    _dspread_method   = NULL;
    
    if (ssp == LPC_SSP0) {
        p12_AS_INPUT;
        LPC_SC->PCONP        |=  (1UL << 21);   // Enable the SSP0 peripheral.
        LPC_SC->PCLKSEL1     &= ~(3UL << 10);   // Select the clock required for SSP0...
        LPC_SC->PCLKSEL1     |=  (3UL << 10);   // ... must be at least 12x faster then slave clk coming in.
        LPC_PINCON->PINSEL0  &= ~(3UL << 30);   // Connect pin SCK0
        LPC_PINCON->PINSEL0  |=  (2UL << 30);               
        LPC_PINCON->PINSEL1  &= ~(3UL << 2);    // Connect pin MISO0
        LPC_PINCON->PINSEL1  |=  (2UL << 2);    
        LPC_PINCON->PINSEL1  &= ~(3UL << 0);    // Connect pin SSEL0
        LPC_PINCON->PINSEL1  |=  (2UL << 0);
        LPC_PINCON->PINSEL1  &= ~(3UL << 4);    // Connect pin MOSI0
        LPC_PINCON->PINSEL1  |=  (2UL << 4);
    }
    
    if (ssp == LPC_SSP1) {
        p6_AS_INPUT;
        LPC_SC->PCONP        |=  (1UL << 10);   // Enable the SSP1 peripheral.
        LPC_SC->PCLKSEL0     &= ~(3UL << 20);   // Select the clock required for SSP1...
        LPC_SC->PCLKSEL0     |=  (1UL << 20);   // ... must be at least 12x faster then slave clk coming in.
        LPC_PINCON->PINSEL0  &= ~(3UL << 14);   // Connect pin SCK1
        LPC_PINCON->PINSEL0  |=  (2UL << 14);
        LPC_PINCON->PINSEL0  &= ~(3UL << 16);   // Connect pin MISO1
        LPC_PINCON->PINSEL0  |=  (2UL << 16);
        LPC_PINCON->PINSEL0  &= ~(3UL << 12);   // Connect pin SSEL1
        LPC_PINCON->PINSEL0  |=  (2UL << 12);        
        LPC_PINCON->PINSEL0  &= ~(3UL << 18);   // Connect pin MOSI1 
        LPC_PINCON->PINSEL0  |=  (2UL << 18);
    }
    
    // Common setup.
    //            CPHA=0       CPOL=1    16bit
    ssp->CR0  = (0UL << 7) | (1UL << 6) | 0xF;
    ssp->CPSR = 0;
    ssp->CR1  = (0x3 << 1); 
    
    // Enable receiver interrupt.
    ssp->IMSC = (1UL << 2);
    
    if (ssp == LPC_SSP0) {
        NVIC_SetVector(SSP0_IRQn, (uint32_t)&_ssp_isr);    
        NVIC_EnableIRQ(SSP0_IRQn);   
    }
    
    if (ssp == LPC_SSP1) {
        NVIC_SetVector(SSP1_IRQn, (uint32_t)&_ssp_isr);    
        NVIC_EnableIRQ(SSP1_IRQn);   
    }       
}

void
AD9951DSPSERIAL::ssp_isr(void)
{
    uint16_t value;
    
    LED4_ON;
    
    while (ssp->SR & (1UL << 2)) {
        value = (uint16_t)(ssp->DR & 0xFFFF);
        if (_dspread_function != NULL) (*_dspread_function)(value);
        else {
            if (_dspread_obj && _dspread_method) (_dspread_obj->*_dspread_method)(value);
        }
    }  
}

}; // namespace AjK ends.
