/* Copyright 2011 Adam Green (http://mbed.org/users/AdamGreen/)

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
/* The purpose of this sample program is to provide an example of how to
   create a simple server based application using the raw callback based
   APIs provided by the lwIP network stack.
   
   It implements a very simple HTTP server that is able to source a single
   HTML file which is embedded in the source code itself.

   NOTE: This code contains no error handling code to make the process flow
         easier to see.  As such, it is not suitable to be used as the basis
         for production code.
            
   More detailed notes about this sample can be found at:
    http://mbed.org/users/AdamGreen/notebook/inside_lwipsample1/
*/
#include "mbed.h"
#include "EthernetNetIf.h"
#include "lwip/init.h"
#include "lwip/dhcp.h"
#include "lwip/tcp_impl.h"
#include "lwip/dns.h"
#include "netif/etharp.h"
#include "drv/eth/eth_drv.h"


// String used to identify this sample HTTP server to the client.
#define HTTP_SERVER_NAME "Server: lwIPSample1\r\n"


// Structure which stores information about the network stack.
struct SNetwork
{
    // Listening port for HTTP Server.
    tcp_pcb*        pHTTPListenPCB;
    // Timers used to determine when various network maintenance tasks should
    // be executed.
    Timer           ARP;
    Timer           DHCPCoarse;
    Timer           DHCPFine;
    Timer           TCP;
    Timer           DNS;
    // Object used to track the DHCP state.
    dhcp            DHCP;
    // The structure used to represent the LPC17xx polled Ethernet interface.
    netif           EthernetInterface;
};


// Structure which stored information about each HTTP client connection.
struct SHTTPContext
{
    // Pointer to response text being sent back to client.
    const char*     pResponseCurr;
    // Number of bytes in response left to be sent back to client.
    size_t          ResponseBytesLeft;
    // Have we starting receiving the request yet?
    int             RequestStarted;
    // Have we finished receiving the request yet?
    int             RequestCompleted;
    // Number of outbound bytes outstanding, awaiting acknowledgement.
    int             UnacknowledgedBytes;
    // Number of tcp_write() retries attempted so far.
    int             RetryCount;
    // Buffer used to track the last four characters read when searching for
    // request terminator.
    char            LastFourChars[4];
};


// HTTPParseRequest returns a bitmask composed of the followings bits to
// indicate the result of the parsing.
// ----------------------------------------------------------------------------
// A GET request was received and ppData will point to NULL termianted URI.
#define HTTP_PARSE_GET              1
// An unsupported request was received.
#define HTTP_PARSE_NOT_IMPLEMENTED  2
// An unknown request was received.
#define HTTP_PARSE_BAD_REQUEST      4


// Function Prototypes
static void        NetworkInit(SNetwork* pNetwork);
static void        NetworkPoll(SNetwork* pNetwork);
static void        NetworkPrintAddress(ip_addr_t* pAddress);
static void        HTTPInit(SNetwork* pNetwork);
static err_t       HTTPAccept(void* pvArg, tcp_pcb* pNewPCB, err_t err);
static err_t       HTTPRecv(void* pvArg, tcp_pcb* pPCB, pbuf* pBuf, err_t err);
static int         HTTPContainsRequestTerminator(pbuf* pBuf, SHTTPContext* pHTTPContext);
static int         HTTPParseRequest(pbuf* pBuf, const char** ppData);
static const char* HTTPOpenFile(const char* pURI, size_t* pFileSize);
static void        HTTPSendResponse(tcp_pcb* pPCB, SHTTPContext* pHTTPContext);
static void        HTTPCloseConnection(tcp_pcb* pPCB, SHTTPContext* pHTTPContext);
static err_t       HTTPSent(void* pvArg, tcp_pcb* pPCB, u16_t SendCount);
static void        HTTPError(void* pvArg, err_t Error);
static err_t       HTTPPoll(void* pvArg, tcp_pcb* pPCB);


int main() 
{
    DigitalOut      ProgressLED(LED1);
    Timer           BlinkTimer;
    SNetwork        Network;
    
    printf("\r\nlwIPSample1\r\n");
    
    NetworkInit(&Network);
    HTTPInit(&Network);
                
    BlinkTimer.start();
    while(1) 
    {
        NetworkPoll(&Network);

        if (BlinkTimer.read_ms() > 500)
        {
            BlinkTimer.reset();
            ProgressLED = !ProgressLED;
        }
    }
}

static void NetworkInit(SNetwork* pNetwork)
{
    ip_addr_t   IpAddress;
    ip_addr_t   NetMask;
    ip_addr_t   GatewayAddress;

    // Clear out the network object and fill it in during the initialization
    // process.
    memset(pNetwork, 0, sizeof(*pNetwork));
    
    // Initialize the lwIP network stack.
    lwip_init();
    
    // Clear the IP addresses and use DHCP instead
    ip_addr_set_zero(&IpAddress);
    ip_addr_set_zero(&NetMask);
    ip_addr_set_zero(&GatewayAddress);
    
    // Obtain the MAC address for the Ethernet device.
    // NOTE: This should really be done in eth_init()
    pNetwork->EthernetInterface.hwaddr_len = ETHARP_HWADDR_LEN;
    eth_address((char *)pNetwork->EthernetInterface.hwaddr);    
    
    // Connect the LPC17xx ethernet driver into the lwIP stack.
    netif_add(&pNetwork->EthernetInterface, 
              &IpAddress, 
              &NetMask, 
              &GatewayAddress, 
              NULL, 
              eth_init, 
              ethernet_input);
    netif_set_default(&pNetwork->EthernetInterface);
    
    // Start the required timers.
    pNetwork->ARP.start();
    pNetwork->DHCPCoarse.start();
    pNetwork->DHCPFine.start();
    pNetwork->TCP.start();
    pNetwork->DNS.start();
    
    // Start the DHCP request.
    pNetwork->EthernetInterface.hostname = "MBED";
    dhcp_set_struct(&pNetwork->EthernetInterface, &pNetwork->DHCP);
    dhcp_start(&pNetwork->EthernetInterface);

    // Wait for DHCP request to complete.
    Timer timeout;
    timeout.start();
    printf("Waiting for DHCP address...\r\n");

    // Wait until interface is up
    while (!netif_is_up(&pNetwork->EthernetInterface)) 
    {
        NetworkPoll(pNetwork);

        // Stop program if we get a timeout on DHCP attempt.
        if (timeout.read_ms() > 10000) 
        {
            error("DHCP Timeout.\r\n");
        }
    }

    // Print out the DHCP provided IP addresses.
    printf("IP     : "); 
    NetworkPrintAddress(&pNetwork->EthernetInterface.ip_addr);
    printf("\r\n");

    printf("Gateway: "); 
    NetworkPrintAddress(&pNetwork->EthernetInterface.gw);
    printf("\r\n");

    printf("Mask   : "); 
    NetworkPrintAddress(&pNetwork->EthernetInterface.netmask);
    printf("\r\n");
}

static void NetworkPoll(SNetwork* pNetwork)
{
    if (pNetwork->DHCPFine.read_ms() >= DHCP_FINE_TIMER_MSECS)
    {
        pNetwork->DHCPFine.reset();
        dhcp_fine_tmr();
    }
    if (pNetwork->DHCPCoarse.read() >= DHCP_COARSE_TIMER_SECS)
    {
        pNetwork->DHCPCoarse.reset();
        dhcp_coarse_tmr();
    }
    if (pNetwork->ARP.read_ms() >= ARP_TMR_INTERVAL)
    {
        pNetwork->ARP.reset();
        etharp_tmr();
    }
    if (pNetwork->TCP.read_ms() >= TCP_TMR_INTERVAL)
    {
        pNetwork->TCP.reset();
        tcp_tmr();
    }
    if (pNetwork->DNS.read_ms() >= DNS_TMR_INTERVAL)
    {
        pNetwork->DNS.reset();
        dns_tmr();
    }

    // Poll the ethernet driver to let it pull packets in and push new packets
    // into the lwIP network stack.
    eth_poll();
}

static void NetworkPrintAddress(ip_addr_t* pAddress)
{
    printf("%d.%d.%d.%d", 
            ip4_addr1(pAddress),
            ip4_addr2(pAddress),
            ip4_addr3(pAddress),
            ip4_addr4(pAddress));
}

static void HTTPInit(SNetwork* pNetwork)
{
    tcp_pcb*    pcb = NULL;
    
    pcb = tcp_new();
    tcp_bind(pcb, IP_ADDR_ANY, 80);
    pNetwork->pHTTPListenPCB = tcp_listen(pcb);
    
    tcp_arg(pNetwork->pHTTPListenPCB, pNetwork);
    tcp_accept(pNetwork->pHTTPListenPCB, HTTPAccept);
}

static err_t HTTPAccept(void* pvArg, tcp_pcb* pNewPCB, err_t err)
{
    SNetwork*       pNetwork = (SNetwork*)pvArg;
    SHTTPContext*   pHTTPContext = NULL;
    
    printf("HTTP: Accepting client connection from: ");
    NetworkPrintAddress(&pNewPCB->remote_ip);
    printf("\r\n");
    
    tcp_accepted(pNetwork->pHTTPListenPCB);

    pHTTPContext = (SHTTPContext*)malloc(sizeof(*pHTTPContext));
    memset(pHTTPContext, 0, sizeof(*pHTTPContext));
    tcp_arg(pNewPCB, pHTTPContext);
    
    tcp_recv(pNewPCB, HTTPRecv);
    tcp_sent(pNewPCB, HTTPSent);
    tcp_err(pNewPCB, HTTPError);
    tcp_poll(pNewPCB, HTTPPoll, 4);
    
    return ERR_OK;
}

static err_t HTTPRecv(void* pvArg, tcp_pcb* pPCB, pbuf* pBuf, err_t err)
{
    SHTTPContext* pHTTPContext = (SHTTPContext*)pvArg;
    
    if (!pBuf)
    {
        return ERR_OK;
    }
    
    tcp_recved(pPCB, pBuf->tot_len);
    
    if (pHTTPContext->RequestStarted)
    {
        // Check for the blank line terminator if we haven't seen it yet.
        if (!pHTTPContext->RequestCompleted)
        {
            pHTTPContext->RequestCompleted = 
                            HTTPContainsRequestTerminator(pBuf, pHTTPContext);
        }
    }
    else
    {
        // Start parsing the request.
        const char* pURI = NULL;
        int         ParseResult = 0;
        
        ParseResult = HTTPParseRequest(pBuf, &pURI);
        if (ParseResult & HTTP_PARSE_GET)
        {
            printf("HTTP: GET '%s'\r\n", pURI);
            pHTTPContext->pResponseCurr = 
                            HTTPOpenFile(pURI, 
                                         &pHTTPContext->ResponseBytesLeft);
            pHTTPContext->RequestCompleted = 
                            HTTPContainsRequestTerminator(pBuf, pHTTPContext);
        }
        else if (ParseResult & HTTP_PARSE_NOT_IMPLEMENTED)
        {
            static const char UnknownResponse[] = 
                "HTTP/1.0 501 Not Implemented\r\n"
                 HTTP_SERVER_NAME
                 "Content-type: text/html\r\n"
                 "\r\n"
                 "<html><body><h2>501: Not Implemented.</h2></body></html>"
                 "\r\n";
            
            printf("HTTP: Command not implemented.\r\n");
            pHTTPContext->pResponseCurr = UnknownResponse;
            pHTTPContext->ResponseBytesLeft = sizeof(UnknownResponse) - 1;
            pHTTPContext->RequestCompleted = 
                            HTTPContainsRequestTerminator(pBuf, pHTTPContext);
        }
        else if (ParseResult & HTTP_PARSE_BAD_REQUEST)
        {
            printf("HTTP: Bad Request.\r\n");
            HTTPCloseConnection(pPCB, pHTTPContext);
        }

        pHTTPContext->RequestStarted = 1;        
    }

    HTTPSendResponse(pPCB, pHTTPContext);

    pbuf_free(pBuf);
    
    return ERR_OK;
}

static int HTTPContainsRequestTerminator(pbuf* pBuf, SHTTPContext* pHTTPContext)
{
    static const char   RequestTerminator[4] = { '\r', '\n', '\r', '\n' };
    
    // Loop through all chunks in the pbuf list.
    while (pBuf)
    {
        char* pCurr = (char*)pBuf->payload;
        u16_t len = pBuf->len;
        
        // Loop through the characters in this chunk.
        while (len)
        {
            // Shift the current byte into the buffer.
            memcpy(pHTTPContext->LastFourChars, 
                   pHTTPContext->LastFourChars+1, 
                   3);
            pHTTPContext->LastFourChars[3] = *pCurr;
            
            // Have we encountered the blank line request terminator.
            if (0 == memcmp(pHTTPContext->LastFourChars, 
                            RequestTerminator, 
                            sizeof(RequestTerminator)))
            {
                return 1;
            }
            
            // Advance to the next character in this chunk.
            pCurr++;
            len--;
        }
        
        // Advance to the next chunk in the pbuf list.
        pBuf = pBuf->next;
    }
    
    // If we get here then we didn't find the blank line terminator.
    return 0;
}

static int HTTPParseRequest(pbuf* pBuf, const char** ppData)
{
    int         Return = HTTP_PARSE_BAD_REQUEST;
    u16_t       CharsLeft = pBuf->len;
    const char* pURI = NULL;
    
    if (CharsLeft >= 4)
    {
        // pbuf is large enough to contain command so attempt to parse it.
        if (0 == memcmp(pBuf->payload, "GET ", 4))
        {
            // This is GET request so search for next whitespace character as it
            // will terminate the URI.
            char* pCurr = (char*)pBuf->payload + 4;
            CharsLeft -= 4;
            pURI = pCurr;
                   
            while(CharsLeft > 0 &&
                  ' '  != *pCurr &&
                  '\r' != *pCurr &&
                  '\n' != *pCurr)
            {
                pCurr++;
                CharsLeft--;
            }
            
            // Have a valid URI if whitespace was found
            if (CharsLeft > 0)
            {
                if (' ' == *pCurr)
                {
                    // Protocol string follows URI in request so must be
                    // HTTP/1.0 or HTTP/1.1
                    Return = HTTP_PARSE_GET;
                }
                else
                {
                    // Protocol string didn't follow URI so request must be
                    // from older v0.9 client which this sample doesn't
                    // support.
                    Return = HTTP_PARSE_BAD_REQUEST;
                }
                
                // Null terminate the URI at the whitespace separator
                *pCurr = '\0';
                
                // Return a pointer to where the URI begins
                *ppData = pURI;
            }
        }
        else
        {
            Return = HTTP_PARSE_NOT_IMPLEMENTED;
        }
    }
        
    return Return;
}

static const char* HTTPOpenFile(const char* pURI, size_t* pFileSize)
{
    static const char HelloWorldPage[] = 
        "HTTP/1.0 200 OK\r\n"
        HTTP_SERVER_NAME
        "Content-Type: text/html\r\n"
        "\r\n"
        "<html>"
        "<head><title>Hello World</title></head>"
        "<body>"
        "<h1>mbed + lwIP = Internet of Things</h1>"
        "</body>"
        "</html>\r\n";
    static const char FileNotFoundPage[] = 
        "HTTP/1.0 404 File not found\r\n"
        HTTP_SERVER_NAME
        "Content-Type: text/html\r\n"
        "\r\n"
        "<html>"
        "<head><title>404 - File not found</title></head>"
        "<body><h2>404: File not found.</h2></body>"
        "</html>\r\n";
                                          
    // Either return the main page if clients asks for "/" or "/index.html"
    // Otherwise return the 404 File Not Found page.
    if (0 == strcmp(pURI, "/") ||
        0 == strcmp(pURI, "/index.html"))
    {
        *pFileSize = sizeof(HelloWorldPage) - 1;
        return HelloWorldPage;
    }
    else
    {
        printf("HTTP: File not found.\r\n");
        *pFileSize = sizeof(FileNotFoundPage) - 1;
        return FileNotFoundPage;
    }
}

static void HTTPSendResponse(tcp_pcb* pPCB, SHTTPContext* pHTTPContext)
{
    size_t BytesToSend = pHTTPContext->ResponseBytesLeft;
    err_t  WriteResult = ERR_MEM;
    
    if (BytesToSend > tcp_sndbuf(pPCB))
    {
        BytesToSend = (size_t)tcp_sndbuf(pPCB);
    }
    
    while (BytesToSend > 0)
    {
        WriteResult = tcp_write(pPCB, 
                                pHTTPContext->pResponseCurr, BytesToSend, 
                                TCP_WRITE_FLAG_COPY);
        if (ERR_OK == WriteResult)
        {
            pHTTPContext->pResponseCurr += BytesToSend;
            pHTTPContext->ResponseBytesLeft -= BytesToSend;
            pHTTPContext->UnacknowledgedBytes += BytesToSend;
            break;
        }
        else if (ERR_MEM == WriteResult)
        {
            if (0 == tcp_sndqueuelen(pPCB))
            {
                break;
            }
            BytesToSend >>= 1;
        }
        else
        {
            // Received unexpected error.
            break;
        }
    }
    
    if (0 == pHTTPContext->ResponseBytesLeft && pHTTPContext->RequestCompleted)
    {
        printf("HTTP: Response send completed.\r\n");
        HTTPCloseConnection(pPCB, pHTTPContext);
    }
}

static void HTTPCloseConnection(tcp_pcb* pPCB, SHTTPContext* pHTTPContext)
{
    err_t CloseResult = ERR_MEM;
    
    printf("HTTP: Closing connection.\r\n");
    
    if (pHTTPContext)
    {
        free(pHTTPContext);
    }
    
    tcp_arg(pPCB, NULL);
    tcp_accept(pPCB, NULL);
    tcp_recv(pPCB, NULL);
    tcp_sent(pPCB, NULL);
    tcp_poll(pPCB, NULL, 0);
    tcp_err(pPCB, NULL);
    
    CloseResult = tcp_close(pPCB);
    if (ERR_OK != CloseResult)
    {
        tcp_poll(pPCB, HTTPPoll, 4);
    }
}

static err_t HTTPSent(void* pvArg, tcp_pcb* pPCB, u16_t AcknowledgeCount)
{
    SHTTPContext* pHTTPContext = (SHTTPContext*)pvArg;
    
    pHTTPContext->UnacknowledgedBytes -= AcknowledgeCount;
    
    HTTPSendResponse(pPCB, pHTTPContext);
    
    return ERR_OK;
}

static err_t HTTPPoll(void* pvArg, tcp_pcb* pPCB)
{
    SHTTPContext* pHTTPContext = (SHTTPContext*)pvArg;
    
    if (!pHTTPContext)
    {
        err_t CloseResult;
        
        CloseResult = tcp_close(pPCB);
        if (ERR_OK != CloseResult)
        {
            tcp_abort(pPCB);
            return ERR_ABRT;
        }
        else
        {
            return ERR_OK;
        }
    }
    
    if (0 == pHTTPContext->UnacknowledgedBytes && pHTTPContext->ResponseBytesLeft > 0)
    {
        if (pHTTPContext->RetryCount++ > 8)
        {
            HTTPCloseConnection(pPCB, pHTTPContext);
        }
        else
        {
            HTTPSendResponse(pPCB, pHTTPContext);
        }
    }
    
    return ERR_OK;
}

static void HTTPError(void* pvArg, err_t Error)
{
    SHTTPContext* pHTTPContext = (SHTTPContext*)pvArg;
    
    if (pHTTPContext)
    {
        free(pHTTPContext);
    }
}
