/* Copyright 2011 Adam Green (http://mbed.org/users/AdamGreen/)

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
/* Classes to apply acceleration to input from Sparkfun's Blackberry Trackball
   Breakout board.
*/
#ifndef _ACCELERATION_H_
#define _ACCELERATION_H_

#include <mbed.h>
#include <limits.h>
#include "BBTrackball.h"



/* Class which can apply acceleration to each of the roller inputs from the
   Sparkfun Blackberry Trackball device.
*/
template<size_t SampleCount,
         int    DecelerateThreshold,
         int    AccelerateThreshold,
         int    MaxAcceleration>
class CAcceleratedRoller
{
public:
    CAcceleratedRoller(void)
    {
        m_Acceleration = 1;
        m_VelocitySum = 0;
        m_SampleIndex = 0;
        memset(m_Samples, 0, sizeof(m_Samples));
    }

    void AccumulateSample(short CurrVelocity);
    int  Acceleration(void) const
    {
        return m_Acceleration;
    }
    
protected:
    unsigned char   ClampToUChar(short Value);
    int             m_Acceleration;
    int             m_VelocitySum;
    size_t          m_SampleIndex;
    unsigned char   m_Samples[SampleCount];
};


class CAcceleratedTrackball : public CBBTrackball
{
public:
    CAcceleratedTrackball(PinName BluePin,
                          PinName RedPin,
                          PinName GreenPin,
                          PinName WhitePin,
                          PinName UpPin,
                          PinName DownPin,
                          PinName LeftPin,
                          PinName RightPin,
                          PinName ButtonPin) : CBBTrackball(BluePin,
                                                            RedPin,
                                                            GreenPin,
                                                            WhitePin,
                                                            UpPin,
                                                            DownPin,
                                                            LeftPin,
                                                            RightPin,
                                                            ButtonPin),
                                               m_IterationsSinceLastMotion(INT_MAX)
    {
    }                                                            
    void GetState(int& DeltaX, int& DeltaY, int& ButtonPressed);

protected:
    void    UpdateLEDColourBasedOnMotion(int VelocityX, 
                                         int VelocityY, 
                                         int ButtonPressed);
    
    int                                 m_IterationsSinceLastMotion;
    CAcceleratedRoller<250, 4, 4, 40>   m_UpAcceleration;
    CAcceleratedRoller<250, 4, 4, 40>   m_DownAcceleration;
    CAcceleratedRoller<250, 4, 4, 30>   m_LeftAcceleration;
    CAcceleratedRoller<250, 4, 4, 30>   m_RightAcceleration;
};

#endif // _ACCELERATION_H_
