/* Copyright 2011 Adam Green (http://mbed.org/users/AdamGreen/)

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
/* Implementation of class to control Sparkfun's Blackberry Trackball:
     http://www.sparkfun.com/products/9320
*/
#include <limits.h>
#include "BBTrackball.h"



// Public methods.
void CBBTrackball::GetState(SState* pState)
{
    pState->ButtonPressed = !m_ButtonState;
    pState->Up = UpCount();
    pState->Down = DownCount();
    pState->Left = LeftCount();
    pState->Right = RightCount();
}


void CBBTrackball::SetColour(const SColour* pColour)
{
    m_BluePWM.write((float)pColour->Blue / 255.0f);
    m_RedPWM.write((float)pColour->Red / 255.0f);
    m_GreenPWM.write((float)pColour->Green / 255.0f);
    m_WhitePWM.write((float)pColour->White / 255.0f);
}        



// Protected methods.
void CBBTrackball::UpISR(void)
{
    m_UpCount++;
}

void CBBTrackball::DownISR(void)
{
    m_DownCount++;
}

void CBBTrackball::LeftISR(void)
{
    m_LeftCount++;
}

void CBBTrackball::RightISR(void)
{
    m_RightCount++;
}

void CBBTrackball::ButtonSampleISR(void)
{
    int ButtonSample = m_Button.read();
    
    if (ButtonSample != m_ButtonState)
    {
        // The button appears to be transition to a new state. Don't actually
        // switch to the new state until enough samples have been seen to
        // reduce button bounce.
        if (m_ButtonNewStateCount++ > 5)
        {
            m_ButtonState = ButtonSample;
            m_ButtonNewStateCount = 0;
        }
    }
    else
    {
        m_ButtonNewStateCount = 0;
    }
}

short CBBTrackball::ThresholdToShort(unsigned int Value)
{
    if (Value > SHRT_MAX)
    {
        Value = SHRT_MAX;
    }
    
    return (short)Value;
}

short CBBTrackball::UpCount(void)
{
    unsigned int Count;
    
    Count = m_UpCount;
    m_UpCount = 0;

    return ThresholdToShort(Count);
}

short CBBTrackball::DownCount(void)
{
    unsigned int Count;
    
    Count = m_DownCount;
    m_DownCount = 0;

    return ThresholdToShort(Count);
}

short CBBTrackball::LeftCount(void)
{
    unsigned int Count;
    
    Count = m_LeftCount;
    m_LeftCount = 0;

    return ThresholdToShort(Count);
}

short CBBTrackball::RightCount(void)
{
    unsigned int Count;
    
    Count = m_RightCount;
    m_RightCount = 0;

    return ThresholdToShort(Count);
}