/* Copyright 2011 Adam Green (http://mbed.org/users/AdamGreen/)

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
/* Header file for class to control Sparkfun's Blackberry Trackball:
     http://www.sparkfun.com/products/9320
*/
#ifndef _BBTRACKBALL_H_
#define _BBTRACKBALL_H_
#include <mbed.h>

namespace AFP 
{

/** A class to interface with the Sparkfun Blackberry Trackball Breakout Board.
 *  http://www.sparkfun.com/products/9320
 *
 *  This breakout board uses a hall effect for each direction of motion (up, 
 *  down, left, right) and generates pulses as the trackball is rotated in each
 *  of these directions.  This class counts both the rising and falling edges
 *  of these pulses in an interrupt handler.  The main program can query for
 *  the number of pulses that have been generated since the previous query was
 *  made.
 *
 *  The breakout board also has a push button placed beneath the trackball so
 *  that user presses can be detected as well.  This class will provide the
 *  state of this button as well after filtering it for the purpose of 
 *  debouncing.
 *
 *  The last feature of this breakout board that is supported by the class
 *  includes the 4 LEDs that have been placed beneath the trackball.  This
 *  class allows the caller to specify an individual brightness value for each
 *  of the LEDs (red, blue, green, white). 
 *
 * Example:
 * @code
#include <mbed.h>
#include <USBMouse.h>
#include "BBTrackball.h"

int main() 
{
    static const CBBTrackball::SColour GreenColour = { 0, 255, 0, 0 };
    static CBBTrackball Trackball(p20,  // BLU
                                  p25,  // RED
                                  p26,  // GRN
                                  p10,  // WHT
                                  p5,   // UP
                                  p6,   // DWN
                                  p7,   // LFT
                                  p8,   // RHT
                                  p9);  // BTN
    static USBMouse     Mouse;
    
    // Turn the green LED on.
    Trackball.SetColour(&GreenColour);
    
    for(;;)
    {
        CBBTrackball::SState    TrackballState;
        int                     DeltaX;
        int                     DeltaY;
        
        Trackball.GetState(&TrackballState);

        // NOTE: The breakout board is rotated 90 degrees on my breadboard.
        DeltaX = TrackballState.Up - TrackballState.Down;
        DeltaY = TrackballState.Right - TrackballState.Left;

        Mouse.update(DeltaX, 
                     DeltaY, 
                     TrackballState.ButtonPressed ? MOUSE_LEFT : 0,
                     0);
    }
}
 * @endcode
 */
class CBBTrackball
{
public:
     /** A structure in which the current state of the trackball is returned
         from the GetState() method.
         
         @see GetState()
     */
     struct SState
     {
        int     ButtonPressed;  /**< 1 if the button is currently pressed and 0 otherwise. */
        short   Up;             /**< Number of upward pulses since last call. */
        short   Down;           /**< Number of downward pulses since last call. */
        short   Left;           /**< Number of leftward pulses since last call. */
        short   Right;          /**< Number of rightward pulses since last call. */
     };
     
     /** A structure used to represent the colour of the illumination for the
         trackball.
         
         @see SetColour()
     */
     struct SColour
     {
        unsigned char   Red;    /**< Red colour component (0 - 255). */
        unsigned char   Green;  /**< Green colour component (0 - 255). */
        unsigned char   Blue;   /**< Blue colour component (0 - 255). */
        unsigned char   White;  /**< Additional white colour component (0 - 255). */
     };
     
     /**
     * Create a CBBTrackball object, binds it to the specified input/output
     * pins, and initializes the required interrupt handling.
     *
     * @param BluePin The mbed pin which is connected to the BLU pin of the breakout board.
     *                Must be a pin on which the mbed supports PwmOut.
     * @param RedPin The mbed pin which is connected to the RED pin of the breakout board.
     *                Must be a pin on which the mbed supports PwmOut.
     * @param GreenPin The mbed pin which is connected to the GRN pin of the breakout board.
     *                Must be a pin on which the mbed supports PwmOut.
     * @param WhitePin The mbed pin which is connected to the WHT pin of the breakout board.
     *                Must be a pin on which the mbed supports PwmOut.
     * @param UpPin The mbed pin which is connected to the UP pin of the breakout board.
     *              Must be a pin on which the mbed supports InterruptIn.
     * @param DownPin The mbed pin which is connected to the DWN pin of the breakout board.
     *              Must be a pin on which the mbed supports InterruptIn.
     * @param LeftPin The mbed pin which is connected to the LFT pin of the breakout board.
     *              Must be a pin on which the mbed supports InterruptIn.
     * @param RightPin The mbed pin which is connected to the RHT pin of the breakout board.
     *              Must be a pin on which the mbed supports InterruptIn.
     * @param ButtonPin The mbed pin which is connected to the BTN pin of the breakout board.
     */
    CBBTrackball(PinName BluePin,
                 PinName RedPin,
                 PinName GreenPin,
                 PinName WhitePin,
                 PinName UpPin,
                 PinName DownPin,
                 PinName LeftPin,
                 PinName RightPin,
                 PinName ButtonPin) : m_BluePWM(BluePin),
                                      m_RedPWM(RedPin),
                                      m_GreenPWM(GreenPin),
                                      m_WhitePWM(WhitePin),
                                      m_UpInterrupt(UpPin),
                                      m_DownInterrupt(DownPin),
                                      m_LeftInterrupt(LeftPin),
                                      m_RightInterrupt(RightPin),
                                      m_Button(ButtonPin),
                                      m_UpCount(0),
                                      m_DownCount(0),
                                      m_LeftCount(0),
                                      m_RightCount(0),
                                      m_ButtonState(1),
                                      m_ButtonNewStateCount(0)
    {
        static const SColour AllLEDsOff = { 0, 0, 0, 0 };
        
        m_UpInterrupt.rise<CBBTrackball>(this, &CBBTrackball::UpISR);
        m_DownInterrupt.rise<CBBTrackball>(this, &CBBTrackball::DownISR);
        m_LeftInterrupt.rise<CBBTrackball>(this, &CBBTrackball::LeftISR);
        m_RightInterrupt.rise<CBBTrackball>(this, &CBBTrackball::RightISR);
        m_ButtonSampleTicker.attach_us<CBBTrackball>(this, &CBBTrackball::ButtonSampleISR, 100);
        SetColour(&AllLEDsOff);
    }
    
    
    /** Gets current state of the trackball.
    *
    * Returns state indicating how much the trackball has moved in each 
    * direction since the last call to GetState() and the current state of the
    * push button.
    *
    * @param pState points to the state structure to be filled in with the 
    *        current state.
    */
    void GetState(SState* pState);
    
    /** Sets the colour of the trackball illumination.
    *
    * @param pColour points to the colour structure used to determine the
    *        illumination contribution from each of the 4 LEDs found beneath
    *        the trackball.
    */
    void SetColour(const SColour* pColour);
    
                     
protected:
    void UpISR(void);
    void DownISR(void);
    void LeftISR(void);
    void RightISR(void);
    void ButtonSampleISR(void);
    short ThresholdToShort(unsigned int Value);
    short UpCount(void);
    short DownCount(void);
    short LeftCount(void);
    short RightCount(void);
    
    PwmOut          m_BluePWM;
    PwmOut          m_RedPWM;
    PwmOut          m_GreenPWM;
    PwmOut          m_WhitePWM;
    
    InterruptIn     m_UpInterrupt;
    InterruptIn     m_DownInterrupt;
    InterruptIn     m_LeftInterrupt;
    InterruptIn     m_RightInterrupt;
    
    DigitalIn       m_Button;
    Ticker          m_ButtonSampleTicker;  
    
    unsigned int    m_UpCount;
    unsigned int    m_DownCount;
    unsigned int    m_LeftCount;
    unsigned int    m_RightCount;
    
    int             m_ButtonState;
    unsigned int    m_ButtonNewStateCount;        
};

} // namespace AFP
using namespace AFP;

#endif /* _BBTRACKBALL_H__BBTRACKBALL_H_ */
