
/**
 * Low level infrared transmission
 * This library provides a low level interface to send IR commands. in principle
 * other higher level IR type remotes could be build on top of this. I wrote
 * this code to send IR commands to a Nikon camera, which it works rather well
 * for. 
 *
 * Copyright (C) 2010 Ali Saidi
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
 #ifndef __TXIR_HPP__
 #define __TXIR_HPP__
 
 #include <mbed.h>
 
 /** A low-level ifrared transmit library */
 class TxIR {
    private:
        /** Function that does the work */
        void doAction();
        
        /** A timeout that is used to do the next action, so we don't spin
          * for the entire IR sequence
          */
        Timeout delay;
        
        /** If the stack is currently busy transmitting something */
        bool _inUse;
        
        /** A local copy of the data so the caller can muck with their own */
        unsigned *data;
        
        /** length of the array */
        unsigned len;
        
        /** Current location in the array */
        unsigned pos;
        
        /** The pin used to send the IR commands */
        PwmOut txPin;
        
        /** If we're currenting sending data or waiting */
        bool high;
        
    public:
        /** Initialize the TxIR library
         * @param pin a pin on the mbed that is PWM capable
         */
        TxIR(PinName pin)
            : _inUse(false), txPin(pin)
        {}
        
        /** If something is currently being sent 
          * @return is something being tranmitted currently 
          * @todo is there a lock library available, this certainly isn't safe
          */
        bool inUse() { return _inUse; }
        
        /** Transmit an IR sequence
          * @param freq the modulation frequency in us normally ~40kHz -> 26us
          * @param len the number of modulation changes to send (length of data) 
          * @param data an array of on/off times in us that will be transmitted
          * @return transmission has begun
          */
        bool txSeq(unsigned freq, unsigned len, const unsigned *data);
};

#endif // __TXIR_HPP__
          