#include "mbed.h"
#include "Adxl355Drv.h"

/* --- Definitions --- */
#define MAX_SEND_SIZE   (128)

/* AXIS */
#define AXIS_X  (0)
#define AXIS_Y  (1)
#define AXIS_Z  (2)
#define NUM_OF_AXIS (3)

/* REGISTERS */
#define ADXL355_REG_DEVID_AD    (0x00)
#define ADXL355_REG_DEVID_MST   (0x01)
#define ADXL355_REG_PARTID      (0x02)
#define ADXL355_REG_REVID       (0x03)

#define ADXL355_REG_XDATA_H     (0x08)
#define ADXL355_REG_XDATA_L     (0x09)
#define ADXL355_REG_YDATA_H     (0x0A)
#define ADXL355_REG_YDATA_L     (0x0B)
#define ADXL355_REG_ZDATA_H     (0x0C)
#define ADXL355_REG_ZDATA_L     (0x0D)


#define ADXL355_REG_FIFO_CTL    (0x3A)
#define ADXL355_REG_MESURE      (0x3E)
#define ADXL355_REG_POWER_CTL   (0x3F)

/* --- Static Variables --- */
static int dummyreg = 0;
static int devSpiId = 0;
static int senseUid = 0;
static float a[3] = {0.0};

static Ticker mesureTick;

/* --- Prototypes --- */
static void tickHanlderForSensorRead(void);

/* DeviceDriverBody*/
static void initConnectionToAdxl355(void);
static void updateDeviceInfoAdxl355(void);
static void startMesumentAdxl355(void);
static void stopMesumentAdxl355(void);
static int readAccelAdxl355(int selAxis);
static void spiWriteAdxl355(int reg, int val);
static int spiReadAdxl355(int reg);
static int spiReadDualAdxl355(int reg);

extern Serial pc;

static SPI myspi(SPI_MOSI, SPI_MISO, SPI_SCK); /* D13, D12, D11 */
static DigitalOut mycs(SPI_CS);   /* D10 */
static DigitalIn int1(D7);      /* D7/D6 */
static DigitalIn int2(D5);      /* D5/D4 */

#define SPI_RD  (1)
#define SPI_WR  (0)
#define SPI_EMPTY_SEND  (0x00)


/***************************************************************/
/* Device Driver Interface */
/***************************************************************/
/** This handler is called 
 *  in SRS command */
static bool adxl355_simple_reset(void)
{
    /* TODO */
    dbgprintf("debug:adxl355_simple_reset\n");
    mesureTick.detach();
    return true;
}

/** This handler is called 
 *  in the first time of mesuments */
static int adxl355_simple_init(void)
{
    /* TODO */
    dbgprintf("debug:adxl355_simple_init\n");
    mesureTick.detach();
    
    initConnectionToAdxl355();

    return 0;   
}

/** This handler is called 
 *  in the repling mesurement data to Host (ASAP) */
static bool adxl355_simple_exec(int deviceID, Serial *pOut, FILE *pSave)
{
    char sendBuf[MAX_SEND_SIZE] = {0};
    int sendCharLen;
    
    dbgprintf("debug:adxl355_simple_exec\n");
    
    updateDeviceInfoAdxl355();
 
    a[AXIS_X] = ((double)readAccelAdxl355(AXIS_X)) * 9.8 * 0.1; /* only use 8bit/MSB in 12bit */
    a[AXIS_Y] = ((double)readAccelAdxl355(AXIS_Y)) * 9.8 * 0.1; /* only use 8bit/MSB in 12bit */
    a[AXIS_Z] = ((double)readAccelAdxl355(AXIS_Z)) * 9.8 * 0.1; /* only use 8bit/MSB in 12bit */
    
    /* TODO */
    uprintf(":%d XDS ", deviceID);
    uprintf("0029 REG=%04x,AXIS=%d,ID=%08x\n", 
        dummyreg, NUM_OF_AXIS, devSpiId);
    uprintf(":%d XFD ", deviceID);
    sprintf(sendBuf, "%4.4f %4.4f %4.4f", a[AXIS_X], a[AXIS_Y], a[AXIS_Z]);
    sendCharLen = strlen(sendBuf);
    uprintf("%04d %s\n", sendCharLen, sendBuf);

    if (pSave != NULL) {        
        dbgprintf("debug:adxl355_simple_exec [Write SD card]\n");
        /* TODO Sd Card save */
        fprintf(pSave, "Device:%08d,%d,%f,%f,%f\n", 
            devSpiId, senseUid, a[AXIS_X], a[AXIS_Y], a[AXIS_Z]);
    }
    
    return true;   
}

/** This handler is called 
 *  in CRS command */
static bool adxl355_simple_ready2run(void)
{
    /* TODO */
    dbgprintf("debug:adxl355_simple_ready2run\n");
    updateDeviceInfoAdxl355();
    startMesumentAdxl355();
    mesureTick.attach(tickHanlderForSensorRead, 0.5);
    return true;   
}

/** This handler is called 
 *  in End of Mesument */
static bool adxl355_simple_run2ready(void)
{
    /* TODO */
    dbgprintf("debug:adxl355_simple_ready2run\n");
    stopMesumentAdxl355();
    mesureTick.detach();
    return true;   
}

/** This handler is called 
 *  in CFW after CID */
static bool adxl355_simple_get_config(int id_pos, int*get)
{
    /* TODO */
    dbgprintf("debug:adxl355_simple_get_config(ID=%x)\n", id_pos);
    *get = dummyreg;
    return true;
}

/** This handler is called 
 *  in CFW after CID */
static bool adxl355_simple_set_config(int id_pos, int set)
{
    /* TODO */
    dbgprintf("debug:adxl355_simple_set_config(ID=%x)\n", id_pos);
    dummyreg = set;
    return true;
}

/****************************************************/
/* for Mesurement */
/****************************************************/
static void tickHanlderForSensorRead(void)
{
    /* TODO */
    senseUid++;
    return;
}

/***************************************************************/
/* Device Driver Body */
/***************************************************************/
static void initConnectionToAdxl355(void)
{
    /** Init SPI for ADXL355 
     * CPHA = CPOL = 0, 1MHz(max.10MHz))
     * CS = inactive 
     */
    myspi.format(8,0);
    myspi.frequency(1000000);
    mycs = 1;
}

static void startMesumentAdxl355(void)
{
    spiWriteAdxl355(ADXL355_REG_FIFO_CTL, 0); /* all data will be bipassed FIFO. */
    spiWriteAdxl355(ADXL355_REG_POWER_CTL, 1); /* WAKEUP mode */
}

static void stopMesumentAdxl355(void)
{
    spiWriteAdxl355(ADXL355_REG_FIFO_CTL, 0); /* all data will be bipassed FIFO. */
    spiWriteAdxl355(ADXL355_REG_POWER_CTL, 0); /* STANDBY mode */
}

static int readAccelAdxl355(int selAxis)
{
    int ret;
    switch(selAxis) {
        case AXIS_X:
            ret = spiReadDualAdxl355(ADXL355_REG_XDATA_H);
            break;
        case AXIS_Y:
            ret = spiReadDualAdxl355(ADXL355_REG_YDATA_H);
            break;
        case AXIS_Z:
            ret = spiReadDualAdxl355(ADXL355_REG_ZDATA_H);
            break;
        default:
            ret = 0xFFFFFFFF;
    }
    ret = ret >> 4;
    if ((ret & 0x00000800) != 0) {
        ret |= 0xFFFFF800;
    }
    dbgprintf("debug:readAccelAdxl355(AXIS=%d)=0x%08x\n", selAxis, ret);
    return ret;
}

static void updateDeviceInfoAdxl355(void)
{

    int devid_ad;
    int devid_mst;
    int devid_part;
    int devid_rev;
    
    devid_ad = spiReadAdxl355(ADXL355_REG_DEVID_AD);
    devid_mst = spiReadAdxl355(ADXL355_REG_DEVID_MST);
    devid_part = spiReadAdxl355(ADXL355_REG_PARTID);
    devid_rev = spiReadAdxl355(ADXL355_REG_REVID);
    devSpiId = (devid_ad    << 0) 
            | (devid_mst    << 8) 
            | (devid_part   << 16)
            | (devid_rev    << 24);
}

static int spiReadAdxl355(int reg)
{
    int val;
    mycs = 0;
    myspi.write(reg << 1 | SPI_RD);
    val = myspi.write(SPI_EMPTY_SEND);
    mycs = 1;  
    dbgprintf("debug:spiReadAdxl355(reg=0x%08x)=0x%08x\n", reg, val);
    return val;
}

static int spiReadDualAdxl355(int reg)
{
    int valh, vall, val;
    mycs = 0;
    myspi.write(reg << 1 | SPI_RD);
    valh = myspi.write(SPI_EMPTY_SEND);
    vall = myspi.write(SPI_EMPTY_SEND);
    mycs = 1;  
    val = valh << 8 | vall;    
    dbgprintf("debug:spiReadDualAdxl355(reg=0x%08x)=0x%08x\n", reg, val);
    return val;
}

static void spiWriteAdxl355(int reg, int val)
{
    mycs = 0;
    myspi.write(reg << 1 | SPI_WR);
    myspi.write(val);
    mycs = 1;  
    dbgprintf("debug:spiWriteAdxl355(reg=0x%08x)=0x%08x\n", reg, val);
}

/****************************************************/
/* for Registration of Device Driver  */
/****************************************************/
DeviceDriver Adxl355Simple = {
    .init           = adxl355_simple_init,
    .set_config     = adxl355_simple_set_config,
    .get_config     = adxl355_simple_get_config,
    .reset          = adxl355_simple_reset,
    .exec           = adxl355_simple_exec,
    .ready2run      = adxl355_simple_ready2run,
    .run2ready      = adxl355_simple_run2ready,
    };