/* mbed Microcontroller Library
 * Copyright (c) 2018 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */

#include "mbed.h"
#include "stats_report.h"
#include "ADXL362.h"
#include "math.h"

#define G_2 1.0
#define G_4 2.0
#define G_8 4.0



#define MODE_NORMAL 0
#define MODE_LOW_POWER 1
#define MODE_ULTRA_LOW_POWER 2

DigitalOut led1(LED1);
Serial PC(USBTX, USBRX);

char Buf[128];
char command[8];
char mode[8];
char val[8];
int f_val;

int set_mode(int m);
int set_freq(int v);
int set_grav(int v);
void set_start(void);
void set_stop(void);
static void clearDisplay(void);
static void drawCLI(ADXL362 *adxlCtrl, AccelTemp *getData);
static void draw_Raw(ADXL362 *adxlCtrl, AccelTemp *getData);

volatile int f_run = 0;
volatile int f_wake = 0;
SPI         *spi1;
ADXL362     *adxlCtrl;
AccelTemp GetData;
DigitalIn ADXL_WAKE(WAKE2);
 
void help(void);
        
// main() runs in its own thread in the OS
int main()
{
    PC.baud(115200);
    PC.printf("EV-COG-AD4050LZ Demo\n");

    spi1 = new SPI(SPI1_MOSI, SPI1_MISO, SPI1_SCLK);
    adxlCtrl = new ADXL362(&PC, spi1); 

    clearDisplay();
    
    help();
    
    while (true) {
        // Blink LED and wait 0.5 seconds
        led1 = !led1;
        wait_ms(500);

        PC.scanf("%s", command);
        
        if(strncmp(command, "set", 3) == 0)
        {
            PC.scanf("%s",mode);
            
            if(strncmp(mode, "freq", 4) == 0)
            {
                PC.scanf("%s", val);
                f_val = atoi(val);
                set_freq(f_val);
            }
            else if(strncmp(mode, "mode", 4) == 0)
            {
                PC.scanf("%s", val);
                f_val = atoi(val);
                set_mode(f_val);
            }
            else if(strncmp(mode, "grav", 4) == 0)
            {
                PC.scanf("%s", val);
                f_val = atoi(val);
                set_grav(f_val);
            }
            else if(strncmp(mode, "wakeup", 6) == 0)
            {
                PC.printf("ADXL362 wakeup mode\n");
                adxlCtrl->set_wakeupmode();
                f_wake = 1;
                PC.printf("ADXL362 wakeup waiting......\n");
                while(f_wake)
                {
                    if(ADXL_WAKE.read() == 1)
                    {
                        f_wake = 0;
                        PC.printf("ADXL362 Wake-up\n");
                        led1 = 1;
                        PC.printf("ADXL362 Condition %d\n", ADXL_WAKE.read());
                        adxlCtrl->GetStatus();
                    }
                    else
                    {
                        PC.printf(".");
                    }
                    wait_ms(100);
                }
                
                
            }else
            {
                PC.printf("Command Invalid\n");
            }
        }
        else if(strncmp(command, "start", 5) == 0)
        {
            set_start();
            while(f_run)
            {
                adxlCtrl->SensorRead(&GetData);
                draw_Raw(adxlCtrl, &GetData);
                wait(0.2);
            }
        }
        //else if(strncmp(command, "stop", 4) == 0)
        //{
        //    set_stop();
        //}
        else
        {
            help();
        }
        // Following the main thread wait, report on the current system status
        //sys_state.report_state();
    }
}

void help(void)
{
    PC.printf("Command Usage\n");
    PC.printf("Command : only 'set'\n");
    PC.printf("Mode    : 'freq' is ODR configuration\n");
    PC.printf("        : ODR frequency from 12, 25, 50, 100, 200, 400Hz\n");
    PC.printf("Mode    : 'grav' is select gravity\n");
    PC.printf("        : 0:2g 1:4g 2:8g\n");
    PC.printf("Mode    : 'mode' is power mode\n");
    PC.printf("        : 0:Normal 1:Low Power 2:Ultra Low Power(freq is fixed at 100Hz)\n");
    PC.printf("i.e.  'set mode 0'\n");
    PC.printf("i.e.  'set freq 1000'\n");
    PC.printf("i.e.  'start'\n");    
}

int set_freq(int v)
{
    if((v < 12) || (v > 400)) 
    {
        PC.printf("ODR Freqency Range is over/under. Please configure for 12-400 Hz.\n");
        return 1;
    }
    else
    {
        PC.printf("Set Frequency %d Hz\n", v);
        return 0;
    }
}

int set_grav(int v)
{
    switch(v)
    {
        case 0:
            PC.printf("Set gravity 2g\n");
            adxlCtrl->set_gravity(GRAVITY_2G);
            break;
        case 1:
            PC.printf("Set gravity 4g\n");
            adxlCtrl->set_gravity(GRAVITY_4G);
            break;
        case 2:
            PC.printf("Set gravity 8g\n");
            adxlCtrl->set_gravity(GRAVITY_8G);
            break;
        default:
            PC.printf("Unknown Command\n");
            break;
    };
    
    return 0;
}
int set_mode(int m)
{
    switch(m)
    {
        case MODE_NORMAL:
            PC.printf("Set Normal Mode\n");
            adxlCtrl->set_powermode(POWER_CTL_PARAM_LOWNOISE_NORM);
            break;
        case MODE_LOW_POWER:
            PC.printf("Set Low Power Mode\n");
            adxlCtrl->set_powermode(POWER_CTL_PARAM_LOWNOISE_LOW);
            break;
        case MODE_ULTRA_LOW_POWER:
            PC.printf("Set Ultra Low Power Mode\n");
            adxlCtrl->set_powermode(POWER_CTL_PARAM_LOWNOISE_ULTRA);
            break;
        default:
            PC.printf("Unknown Command\n");
            break;
    };
    
    return 0;
}

void set_start(void)
{
    f_run = 1;
    adxlCtrl->start();
    PC.printf("Start Accel\n");
}

void set_stop(void)
{
    f_run = 0;
    adxlCtrl->stop();
    PC.printf("Stop Accel\n");
}

static void clearDisplay(void)
{
    PC.printf("\033[2J");
    PC.printf("\033[0;0H");
    PC.printf("\033[0m\033[37m");    
}

static void drawCLI(ADXL362 *adxlCtrl, AccelTemp *getData)
{
    AccelTemp *min = adxlCtrl->GetMinInfo();
    AccelTemp *max = adxlCtrl->GetMaxInfo();
    AccelTemp *p;
    float x, y, z, t;
    
    PC.printf("\033[2J");
    
    /* BLANK LINE */
    
    PC.printf("\033[0m\033[33m\033[1m");
    PC.printf("\033[2;1H");
    PC.printf("\033[K");
    PC.printf("ACCELEROMETERs");
    
    PC.printf("\033[0m\033[37m");
    PC.printf("\033[3;3H");
    PC.printf("\033[K");
    PC.printf("\033[3;10H-X---\033[3;20H-Y---\033[3;30H-Z---");
    
    PC.printf("\033[0m\033[35m");
    PC.printf("\033[4;3H");
    PC.printf("\033[K");
    p = min;
    x = sin(adxlCtrl->ConvAccel(p->ax));
    y = adxlCtrl->ConvAccel(p->ay);
    z = adxlCtrl->ConvAccel(p->az);
    PC.printf("min\033[4;10H%04.2f\033[4;20H%04.2f\033[4;30H%04.2f", x, y, z);
    
    PC.printf("\033[0m\033[37m\033[1m");
    PC.printf("\033[5;3H");
    PC.printf("\033[K");
    p = getData;
    x = adxlCtrl->ConvAccel(p->ax);
    y = adxlCtrl->ConvAccel(p->ay);
    z = adxlCtrl->ConvAccel(p->az);
    PC.printf("-->\033[5;10H%04.2f\033[5;20H%04.2f\033[5;30H%04.2f", x, y, z);
    
    PC.printf("\033[0m\033[36m");
    PC.printf("\033[6;3H");
    PC.printf("\033[K");
    p = max;
    x = adxlCtrl->ConvAccel(p->ax);
    y = adxlCtrl->ConvAccel(p->ay);
    z = adxlCtrl->ConvAccel(p->az);
    PC.printf("max\033[6;10H%04.2f\033[6;20H%04.2f\033[6;30H%04.2f", x, y, z);
    
    /* BLANK LINE */
        
    PC.printf("\033[0m\033[31m\033[1m");
    PC.printf("\033[8;1H");
    PC.printf("\033[K");
    PC.printf("Temperature");

    /* BLANK LINE */

    PC.printf("\033[0m\033[35m");
    PC.printf("\033[10;3H");
    PC.printf("\033[K");
    p = min;
    t = adxlCtrl->ConvThermal(p->tm);
    PC.printf("min\033[10;10H%04.2f", t);

    PC.printf("\033[0m\033[37m\033[1m");
    PC.printf("\033[11;3H");
    PC.printf("\033[K");
    p = getData;
    t = adxlCtrl->ConvThermal(p->tm);
    PC.printf("-->\033[11;10H%04.2f", t);
    
    PC.printf("\033[0m\033[36m");
    PC.printf("\033[12;3H");
    PC.printf("\033[K");
    p = max;
    t = adxlCtrl->ConvThermal(p->tm);
    PC.printf("max\033[12;10H%04.2f", t);
}

static void draw_Raw(ADXL362 *adxlCtrl, AccelTemp *getData)
{
    AccelTemp *p;
    float x, y, z, t;
    
    /* BLANK LINE */
    p = getData;
    t = adxlCtrl->ConvThermal(p->tm);
    x = adxlCtrl->ConvAccel(p->ax);
    y = adxlCtrl->ConvAccel(p->ay);
    z = adxlCtrl->ConvAccel(p->az);
    PC.printf("%04.2f, %04.2f, %04.2f, %04.2f\n", x, y, z, t);
}


    