/**
 ******************************************************************************
 * @file    MB85RSxx_SPI.h
 * @author  APS Lab
 * @version V1.0.0
 * @date    26 June 2017
 * @brief   This file contains the class of an MB85RSxx FRAM library with SPI interface
 ******************************************************************************
 * @attention
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef MBED_MB85RSxx_SPI_H
#define MBED_MB85RSxx_SPI_H

#include "mbed.h"
 
#define WREN    0x06
#define WRDI    0x04
#define RDSR    0x05
#define WRSR    0x01
#define READ    0x03
#define WRITE   0x02

#define SR_WPEN 0x80
#define SR_BP0  0x08
#define SR_BP1  0x04
#define SR_WEL  0x02

/**  Interface for accessing Fujitsu MB85RSxx FRAM
 *
 * @code
 * #include "mbed.h"
 * #include "MB85RSxx_SPI.h"
 * 
 * Serial pc(USBTX, USBRX);
 * 
 * #if defined(TARGET_STM32F401RE)
 * MB85RSxx_SPI fram(D11, D12, D13, D10);
 * #else
 * MB85RSxx_SPI fram(D5, D6, D7, D8);
 * #endif
 * 
 * int main()
 * {
 *     char buf[16];
 *     uint32_t address;
 *     
 *     fram.fill(0, 0, 256);
 * 
 *     for (int i = 0; i < 16; i++) {
 *         buf[i] = i;
 *     }
 *     fram.write(0, buf, 16);
 * 
 *     for (address = 0; address < 0x80; address += 16) {
 *         fram.read(address, buf, 16);
 *         pc.printf("%08X : ", address);
 *         for (int i = 0; i < 16; i++) {
 *             pc.printf("%02X ", buf[i]);    
 *         }
 *         pc.printf("\n");
 *     }
 * }
 * 
 * @endcode
 */

/** MB85RSxx_SPI class
 *
 *  MB85RSxx_SPI: A library to access Fujitsu MB85RSxx_SPI FRAM
 *
 */ 
class MB85RSxx_SPI
{
public:

    /** Create an MB85RSxx_SPI instance
     *  which is connected to specified I2C pins with specified address
     *
     * @param mosi SPI master out pin
     * @param miso SPI master in pin
     * @param sclk SPI clock pin
     * @param nss  SPI node select pin
     */
    MB85RSxx_SPI(PinName mosi, PinName miso, PinName sclk, PinName nss);

    /** Destructor of MB85RCxx_I2C
     */
    virtual ~MB85RSxx_SPI();

    
    /** Initialize FRAM **/
    void Init(void);

    /** Write data to memory address
     *
     * @param address Memory address
     * @param data data to write out to memory
     *
     */
    void write(uint32_t address, char data);
    
    /** Read data from memory address 
     *
     * @param FRAM address
     * return data
     */
    char read(uint32_t addr);

    /** Write data to memory address
     *
     * @param address Memory address
     * @param data Pointer to the byte-array data to write
     * @param length Number of bytes to write
     *
     */
    void write(uint32_t address, char *data, uint32_t length);
    
    /** Read data from memory address 
     *
     * @param FRAM address
     * @param FRAM data Pointer to the byte-array to read data in to
     * @prama length Number of bytes to read
     */
    void read(uint32_t address, char *data, uint32_t length);


private:
    SPI         *_spi_p;
    SPI         &_spi;
    DigitalOut  *_di_p;
    DigitalOut  &_di;
};

#endif
