/*
SSD1353Test2 - main.cpp

Copyright 2018 ADay (or "ikubaku")

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "mbed.h"
#include "stdio.h"

#include "main.h"

#define CD_PIN P4_29
#define NRES_PIN P4_28
#define MOSI_PIN P0_18
#define MISO_PIN P0_17
#define SCK_PIN P0_15
#define CS_PIN P0_4

#define LED_PIN P1_18

// 1 bit framebuffer
#define FB_WIDTH 160
#define FB_HEIGHT 128
#define FB_SIZE FB_WIDTH*FB_HEIGHT

#define FPS 30

int b_x;
int b_y;

byte fillR = 0xFF;
byte fillG = 0xFF;
byte fillB = 0xFF;

const uint32_t tiles[] = {0x00000000, 0x747F1880, 0xE4BD1F00, 0x7C210780, 0xF4631F00, 0xFC3F0F80, 0xFC3F0800, 0x7C2F1780, 0x8C7F1880, 0x71084700, 0x71094400, 0x8DB14980, 0x84210F80, 0xDD6B1880, 0x8E6B3880, 0x74631700, 0xF47D0800, 0x64A56780, 0xF47D8B80, 0x7C1C1F00, 0xF9084200, 0x8C631700, 0x8C54A200, 0x8C6B5D80, 0xDA88AD80, 0x8A884200, 0xF8888F80, 0x21080200, 0x744C0200, 0x57D5F500, 0x00004400, 0x00000400, 0x01004000, 0x746B1700, 0x23084700, 0x744C8F80, 0x744D1700, 0x32A5F100, 0xF43C1700, 0x741D1700, 0xFC444200, 0x745D1700, 0x745C1700};

bool framebuf[FB_SIZE];
byte color = 0xFF;
bool fill_en = false;


// I/O
DigitalOut cs_pin(CS_PIN);
DigitalOut cd_pin(CD_PIN);
DigitalOut nres_pin(NRES_PIN);

DigitalOut led_pin(LED_PIN);

SPI spi(MOSI_PIN, MISO_PIN, SCK_PIN);

void sendCommand(byte com) {
    cd_pin = 0;
    spi.write(com);
}
void sendData(byte dat) {
    cd_pin = 1;
    spi.write(dat);
}

void beginSend() {
    cs_pin = 0;
}
void endSend() {
    cs_pin = 1;
    wait_us(10);
}

void displayInit() {
    beginSend();

    // Remap & Color depth setting
    sendCommand(0xA0);
    sendData(0x30);

    // Dim mode
    sendCommand(0xAB);
    sendData(0x00);    // reserved(0x00)
    sendData(0x7F);    // R
    sendData(0x7F);    // G
    sendData(0x7F);    // B
    sendData(0x10);    // precharge
    
    // Enable linear grayscale table
    sendCommand(0xB9);

    // Display On
    sendCommand(0xAF);

    endSend();
}

void fill(byte R, byte G, byte B) {
    beginSend();
    sendCommand(0x26);
    sendData(0x01);
    endSend();
    fill_en = true;
    fillR = R;
    fillG = G;
    fillB = B;
    color = fillB & 0b11100000 | (fillG >> 3) & 0b00011100 | (fillR >> 6) & 0b00000011;
}
void nofill() {
    beginSend();
    sendCommand(0x26);
    sendData(0x00);
    endSend();
    fill_en = false;
}

void drawLine(byte x1, byte y1, byte x2, byte y2, byte R, byte G, byte B) {
    beginSend();
    sendCommand(0x21);
    sendData(x1);  //x1
    sendData(y1 + 0x04);  //y1
    sendData(x2);  //x2
    sendData(y2 + 0x04);  //y2 83
    sendData(R);  //foregroundColor3
    sendData(G);  //2
    sendData(B);  //1
    endSend();
}

void commitFramebuffer(bool * fb) {
    // Set column addr.
    beginSend();
    sendCommand(0x15);
    sendData(0x00);   // start
    sendData(0x9F);   // end

    // Set row addr.
    sendCommand(0x75);
    sendData(0x04);   // start
    sendData(0x83);   // end

    // Write to GDDRAM
    sendCommand(0x5C);
    for(int i=0; i<FB_WIDTH*FB_HEIGHT; i++) {
        if(fb[i]) {
            sendData(color);
        } else {
            sendData(0x00);
        }
    }
  
    endSend();
}

void clearFramebuffer(bool * fb) {
    for(int i=0; i<FB_WIDTH*FB_HEIGHT; i++) {
        fb[i] = false;
    }
}

void blitPoint(bool * fb, byte x, byte y) {
    fb[x + FB_WIDTH*y] = true;
}

void blitLine(bool * fb, byte x1, byte y1, byte x2, byte y2) {
    // stub
}

void blitRect(bool * fb, byte x1, byte y1, byte x2, byte y2) {
    int ix, iy;
    for(ix=x1; ix<=x2; ix++) {
        fb[ix + FB_WIDTH*y1] = true;
        fb[ix + FB_WIDTH*y2] = true;
    }
    
    if(fill_en) {
        for(iy=y1+1; iy<=y2-1; iy++) {
            for(ix=x1; ix<=x2; ix++) {
                fb[ix + FB_WIDTH*iy] = true;
            }
        }
    } else {
        for(iy=y1+1; iy<=y2-1; iy++) {
            fb[x1 + FB_WIDTH*iy] = true;
            fb[x2 + FB_WIDTH*iy] = true;
        }
    }
}

void blitTile(bool * fb, uint32_t x, uint32_t y, uint32_t t_id) {
    int ix, iy;
    uint32_t t = tiles[t_id];
    for(iy = 0; iy < 5; iy++) {
        if(y+iy >= FB_HEIGHT || y+iy < 0) {
            continue;
        }
        for(ix = 0; ix < 5; ix++) {
            if(x+ix >= FB_WIDTH || x+ix < 0) {
                continue;
            }
            if(t & (0b10000000000000000000000000000000 >> (ix + 5*iy))) {
                fb[x+ix + FB_WIDTH*(y+iy)] = true;
            } else {
                fb[x+ix + FB_WIDTH*(y+iy)] = false;
            }
        }
    }
}

void clearScreen() {
    clearFramebuffer(framebuf);
    commitFramebuffer(framebuf);
}

uint32_t char2tid(char c) {
    if('a' <= c && c <= 'z') {
        return (uint32_t)(c - 'a' + 0x01);
    }
    if('A' <= c && c <= 'Z') {
        return (uint32_t)(c - 'A' + 0x01);
    }
    if('0' <= c && c <= '9') {
        return (uint32_t)(c - '0' + 0x21);
    }
    
    switch(c) {
        case '!':
            return 0x1B;
            break;
        case '?':
            return 0x1C;
            break;
        case '#':
            return 0x1D;
            break;
        case ',':
            return 0x1E;
            break;
        case '.':
            return 0x1F;
            break;
        case ':':
            return 0x20;
            break;
        case ' ':
            return 0x00;
            break;
        default:
            return 0x1C;
            break;
    }
}

void blitString(bool * fb, uint32_t x, uint32_t y, const char * str) {
    uint32_t buf[33];
    int d_x = x;
    int i;
    for(i = 0; str[i] != '\0' && i < 32; i++) {
        buf[i] = char2tid(str[i]);
    }
    buf[i] = 0xFF;
    for(i = 0; buf[i] != 0xFF; i++) {
        blitTile(framebuf, d_x, y, buf[i]);
        d_x += 6;
    }
}


int main() {
    // initialize
    spi.format(8, 3);
    spi.frequency(24000000);
    
    cs_pin = 1;
    
    // force reset
    nres_pin = 1;
    wait_us(100);
    nres_pin = 0;
    wait_us(100);
    nres_pin = 1;
    wait(1);
    
    // initialize the display
    displayInit();
    
    // clear screen
    clearScreen();
    
    
    // Set color and fill
    fill(0, 255, 255);
    
    Timer frame_timer;
    frame_timer.reset();
    int value = 0;
    int v_diff = 1;
    while(1) {
        frame_timer.start();
        value += v_diff;
        if(value > 64) {
            v_diff = -1;
        } else if(value <= 0) {
            v_diff = 1;
        }
        
        clearFramebuffer(framebuf);
        
        blitString(framebuf, 32, 32, "Hello, world!");
        
        blitRect(framebuf, 32, 64, 32 + value+1, 72);
        
        commitFramebuffer(framebuf);
        frame_timer.stop();
        int f_busy_time = frame_timer.read_ms();
        int frame_wait_time = 1000 / FPS - f_busy_time;
        if(frame_wait_time > 0) {
            wait_ms(frame_wait_time);
        }
        
        frame_timer.reset();
    }
}
