#include "mbed.h"
#include "SDHCFileSystem.h"

Serial pc(USBTX, USBRX);
SDFileSystem sd(p5, p6, p7, p8, "sd");

DigitalOut CS0(p21);
DigitalOut CD(p22);
DigitalOut RD(p23);
DigitalOut WR(p24);
DigitalOut RSTB(p25);

BusOut Bout(p17, p16, p15, p14, p13, p12, p11, p10);

Timer t;

void LCD_Write_CMD(unsigned char cmd)
{
    CS0=0;
    RD=1;
    CD=0;
    Bout=cmd;
    WR=0;
    WR=1;
    CS0=1;
}

void LCD_Write_Data(unsigned char d)
{
    CS0=0;
    RD=1;
    CD=1;
    Bout=d;
    WR=0;
    WR=1;
    CS0=1;
}

void LCD_Reset(void)
{
    RSTB=1;
    wait_ms(1);
    RSTB=0;
    wait_ms(1);
    RSTB=1;
    wait_ms(120);
    LCD_Write_CMD(0x11); // Sleep Out and Booster On
    wait_ms(120);
}

void LCD_Init(void)
{
    LCD_Reset();

    LCD_Write_CMD(0xB1); //In Normal Mode(Full Colors) Frame rate 80Hz
    LCD_Write_Data(0x02);
    LCD_Write_Data(0x35);
    LCD_Write_Data(0x36);

    LCD_Write_CMD(0xB2); //In Idle Mode(8-colors)
    LCD_Write_Data(0x02);
    LCD_Write_Data(0x35);
    LCD_Write_Data(0x36);

    LCD_Write_CMD(0xB3); //In Partial Mode + Full Colors
    LCD_Write_Data(0x02);
    LCD_Write_Data(0x35);
    LCD_Write_Data(0x36);
    LCD_Write_Data(0x02);
    LCD_Write_Data(0x35);
    LCD_Write_Data(0x36);

    LCD_Write_CMD(0xB4); //Dot Inversion Control
    LCD_Write_Data(0x03); // Inversion setting in Idle mode and full Colors partial mode

    LCD_Write_CMD(0xC0); // Power Control Setting (GVDD Voltage)
    LCD_Write_Data(0xA2); // AVDD=5, GVDD=4.6
    LCD_Write_Data(0x02); // VGCL=-4.6
    LCD_Write_Data(0x84); // MODE=AUTO

    LCD_Write_CMD(0xC1); // Power Control Setting (VGH/VGL Voltage)
    LCD_Write_Data(0xC5); // VGH and VGL supply power level = 2.4, VGL=-10, VGH=3*AVDD-0.5

    LCD_Write_CMD(0xC2); // In Normal Mode(Full Colors) APA/DCA
    LCD_Write_Data(0x0D); // SAP=Small, AP=Large
    LCD_Write_Data(0x00); // Clock frequency for Booster circuit/1,/3,/1,/1,/1

    LCD_Write_CMD(0xC3); // In Idle Mode(8-colors) APA/DCA
    LCD_Write_Data(0x8D);
    LCD_Write_Data(0xEA);

    LCD_Write_CMD(0xC4); // In Partial Mode(Full Colors) APA/DCA
    LCD_Write_Data(0x8D);
    LCD_Write_Data(0xEE);

    LCD_Write_CMD(0xC5); // VCOM
    LCD_Write_Data(0x05); // -0.55

    LCD_Write_CMD(0x36); // Memory Data Access Control
    LCD_Write_Data(0x48); // MX, RGB mode (Row Address Order, RGB color filter panel)

    LCD_Write_CMD(0xE0); // Gamma Adjustment (+Polarity)
    LCD_Write_Data(0x03);
    LCD_Write_Data(0x1B);
    LCD_Write_Data(0x09);
    LCD_Write_Data(0x0E);
    LCD_Write_Data(0x32);
    LCD_Write_Data(0x2D);
    LCD_Write_Data(0x28);
    LCD_Write_Data(0x2C);
    LCD_Write_Data(0x2B);
    LCD_Write_Data(0x29);
    LCD_Write_Data(0x30);
    LCD_Write_Data(0x3B);
    LCD_Write_Data(0x00);
    LCD_Write_Data(0x01);
    LCD_Write_Data(0x02);
    LCD_Write_Data(0x10);

    LCD_Write_CMD(0xE1); // Gamma Adjustment (-Polarity)
    LCD_Write_Data(0x03);
    LCD_Write_Data(0x1B);
    LCD_Write_Data(0x09);
    LCD_Write_Data(0x0E);
    LCD_Write_Data(0x32);
    LCD_Write_Data(0x2E);
    LCD_Write_Data(0x28);
    LCD_Write_Data(0x2C);
    LCD_Write_Data(0x2B);
    LCD_Write_Data(0x28);
    LCD_Write_Data(0x31);
    LCD_Write_Data(0x3C);
    LCD_Write_Data(0x00);
    LCD_Write_Data(0x00);
    LCD_Write_Data(0x02);
    LCD_Write_Data(0x10);

    LCD_Write_CMD(0x3A); // Interface Pixel Format
    LCD_Write_Data(0x05); // 16-bit/pixel

    LCD_Write_CMD(0x2A); // Column Address Set
    LCD_Write_Data(0x00);
    LCD_Write_Data(0x02);
    LCD_Write_Data(0x00);
    LCD_Write_Data(0x81);

    LCD_Write_CMD(0x2B); // Row Address Set
    LCD_Write_Data(0x00);
    LCD_Write_Data(0x01);
    LCD_Write_Data(0x00);
    LCD_Write_Data(0x80);

    wait_ms(100);
    LCD_Write_CMD(0x2C); // Memory Write

}

void LCD_Enter_Sleep(void)
{
    LCD_Write_CMD(0x10); //Sleep in
    wait_ms(120);
}
void LCD_Exit_Sleep(void)
{
    LCD_Write_CMD(0x11); //Sleep out
    wait_ms(120);
}

void LCD_Fill_Black(void)
{
    t.start();
    LCD_Write_CMD(0x2C); // Memory Write
    for (int i=0; i<128; i++) {
        for (int j=0; j<128*2; j++) {
            LCD_Write_Data(0x00);
        }
    }
    LCD_Write_CMD(0x29); // Display On
    t.stop();
    pc.printf("Fill Black %dms\r\n",t.read_ms());
    t.reset();
}

void LCD_Fill_White(void)
{
    t.start();
    LCD_Write_CMD(0x2C); // Memory Write
    for (int i=0; i<128; i++) {
        for (int j=0; j<128*2; j++) {
            LCD_Write_Data(0xFF);
        }
    }
    LCD_Write_CMD(0x29); // Display On
    t.stop();
    pc.printf("Fill White %dms\r\n",t.read_ms());
    t.reset();
}

void LCD_Print_Bitmap(char path[32])
{
    t.start();
    FILE *fp;
    int i,j;
    unsigned char offset;
    unsigned short pixel[128];
    
    if ((fp=fopen(path, "rb")) == NULL) {
        LCD_Fill_Black();
        exit(1);
    }
    
    //LCD_Write_CMD(0x28); // Display Off
    
    fseek(fp, 0x0a, SEEK_SET);
    fread(&offset, 1, 1, fp);
    fseek(fp, offset, SEEK_SET);
    
    LCD_Write_CMD(0x2C); // Memory Write
    
    for (i=0; i<128; i++) {
        fread(&pixel, 2, 128, fp);
        for(j=0;j<128;j++){
            LCD_Write_Data((unsigned char)((pixel[j] & 0xFFFF) >> 8));
            LCD_Write_Data((unsigned char)pixel[j]);
        }
    }
    
    LCD_Write_CMD(0x29); // Display On
    
    fclose(fp);
    t.stop();
    pc.printf("Print Bitmap %dms\r\n",t.read_ms());
    t.reset();
}

int main()
{
    LCD_Init();
    pc.baud(115200);

    if (sd.disk_initialize()){
         pc.printf("error\r\n");
         LCD_Fill_Black();
         exit(1);
    }

    while(1) {
        LCD_Print_Bitmap("/sd/sample.bmp");
        wait(5);
        LCD_Print_Bitmap("/sd/sample2.bmp");
        wait(5);
        LCD_Print_Bitmap("/sd/sample3.bmp");
        wait(5);
    }
}