#include "mbed.h"
#include "Gamepad.h"
#include "N5110.h"


// number of pixels on display
#define WIDTH 84
#define HEIGHT 48
#define BANKS 6

/** Bird Class
@brief Library for the function used in the Game
@brief Under the foundation of the LCD(N5110) and Gamepad
@brief Display welcome, mode choosing and playing screen
@brief Refresh for the speed at 1.0/fps

@brief Revision 1.0

@author Junyao. Yang
@date   6th May 2019

@code
#include "Bird.h"




// Objects
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad pad;
Bird bird;

// Prototypes
void init();
void welcome();
void mode1();
void start(int fps);
void play(int fps);
void failure();



int main()
{
    // set the frame per second
    int fps = 5;
    // first need to initialise the display
    init();
    // these are default settings so not strictly needed
    lcd.normalMode();      // normal colour mode
    lcd.setBrightness(0.6); // put LED backlight on 60%
    // change set contrast in range 0.0 to 1.0
    // 0.5 appears to be a good starting point
    lcd.setContrast(0.5);      
    lcd.clear();

    // welcome screen that waiting the player to  start
    welcome();
    // Mode choosing, to determine the barrier speed
    mode1();

    // game start, draw the bird and the barrier
    start(fps);
    
    while(1){
        // playing procedure
        play(fps);
    
    // failure screen, to ask user to play again 
        failure();
    
    }
    
    
}

void init()
{
    // initialise lcd and gamepad
    lcd.init();
    pad.init();
    bird.init();
}

void welcome()
{
    // print welcome screen 
    bird.welcome(lcd);
    lcd.refresh();
    
    // move to game while start button is pressed
    while ( pad.check_event(Gamepad::START_PRESSED) == false) {
        pad.leds_on();
        wait(0.1);
        pad.leds_off();
        wait(0.1);
    }
}

void mode1()
{      
    // choosing difficulty
    while (pad.check_event(Gamepad::A_PRESSED) == false) {
    bird.mode(lcd, pad);
    }
}

void start(int fps)
{
    // draw the first bird and barrier, 
    // and giving the last 3 seconds to prepare
    bird.ready(lcd);

}

void play(int fps)
{
    lcd.clear(); 
    if(pad.check_event(Gamepad::A_PRESSED)){
    bird.flyup(lcd);    
    }
    else {bird.flydown(lcd);}
    bird.barrier(lcd);
    bird.score(lcd);
    bird.pause(lcd,pad); 
    lcd.refresh();
    wait(1.0/fps); 
}

void failure()
{
    bird.fail(lcd, pad);
}


@endcode
*/

class Bird
{

public:

/** Initialize all the parameter 
    *
    *   initialise each time the procedure run
    */ 
void init();
 
/** Welcome
    *
    *   Print welcome screen
    *   wait until start button is pressed
    */ 
void welcome(N5110 &lcd);

/** Difficulty choosing
    *
    *   Using N5110, Gamepad
    *   Choose difficulty
    *   determine the speed of barrier vb
    */ 
void mode(N5110 &lcd, Gamepad &pad);

/** Display score
    *
    *   Using N5110
    *   start at 0
    *   plus 1 each time pass the barrier
    */ 
void score(N5110 &lcd);

/** Pause screen
    *
    *   Using N5110, Gamepad
    *   run if button back is pressed
    *   return game if press again
    */ 
void pause(N5110 &lcd, Gamepad &pad);

/** Game start structure
    *
    *   Using N5110
    *   draw out the basic structure
    */ 
void stru(N5110 &lcd);

/** Random barrier
    *
    *   Using N5110
    *   random barrier produced
    *   move in the speed be given
    */ 
void barrier(N5110 &lcd);

/** Bird fly up
    *
    *   Using N5110
    *   run as button A is pressed
    *   bird fly up 8 pixel each time
    */ 
void flyup(N5110 &lcd);

/** Bird fly down
    *
    *   Using N5110
    *   run each time screen refresh
    *   drop 2 pixel each time
    */ 
void flydown(N5110 &lcd);

/** Random number 
    *
    *   random number between 1-4
    *   to determine different barrier type
    */ 
void check();

/** Failure screen 
    *
    *   Using N5110, Gamepad
    *   show failure as collided
    *   ask if play again
    */ 
void fail(N5110 &lcd, Gamepad &pad);

/** Ready screen 
    *
    *   Using N5110
    *   give three seconds to prepare
    *   play game later
    */ 
void ready(N5110 &lcd);

/** If collided 
    *
    *   return true if collided
    *   return false if not
    */ 
bool coll();


private:

int _vb; // mode, determine the barrier moving speed
int _birdy; // parameter that determine y-axis of bird
int _x1;
int _x2;
int _x3;
int _x4; // x-axis coordinate of four barrier
int _y1; 
int _y2; 
int _y3; 
int _y4; // y-axis coordinate of four barrie upper side
int _birdh; // actual y coordinate of the bird
int _score; // score display at the top-left corner while playing


};