/**
 ******************************************************************************
 * @file    HC12.h
 * @author  Ernst-Georg Schmid
 * @version 0.0.7
 * @date    04 September 2020
 * @brief   This file contains a HC12 433 MHz ISM-band transceiver library with
 * serial interface
 ******************************************************************************
 * @attention
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 *
 *  For more information about the HC12 433 MHz ISM-band transceiver:
 *  https://ecksteinimg.de/Datasheet/HC-12%20english%20datasheets.pdf
 */

#ifndef MBED_HC12_H
#define MBED_HC12_H

#include "mbed.h"
#include "platform/PlatformMutex.h"

/**  Interface for controlling the HC12 HC12 433 MHz ISM-band transceiver
 *
 * @code
 * #include "mbed.h"
 * #include "HC12.h"
 *
 * static HC12 hc12;
 *
 * int32_t main() {
 *
 * int32_t data = 627604;
 *
 *     while(true) {
 *          hc12.begin_command_mode();
 *          hc12.wakeup();
 *          hc12.end_command_mode();
 *          hc12.send(&data, sizeof(int32_t));
 *          hc12.begin_command_mode();
 *          hc12.sleep();
 *          hc12.end_command_mode();
 *     ThisThread::sleep_for(10s);
 *     }
 * }
 *
 * @endcode
 */

/** HC12 class
 *
 * HC12: A library to transmit and receive data using the HC12 433 MHz ISM-band
 * transceiver
 *
 */
class HC12 {
public:
  /** Create a HC12 instance
   *
   * @param p_tx USART TX Pin (default: D8)
   * @param p_rx USART RX Pin (default: D2)
   * @param p_set MODE SET DigitalOut Pin (default: D4)
   * @param baudrate USART / HC12 Baudrate (default: 9600)
   * @param data_bits Number of Data Bits (default: 8)
   * @param parity Parity (default: No Parity)
   * @param stop_bits Number of Stop Bits (default: 1)
   */
  HC12(PinName p_tx = D8, PinName p_rx = D2, PinName p_set = D4,
       uint32_t baudrate = 9600, uint32_t data_bits = 8,
       SerialBase::Parity parity = SerialBase::None, uint32_t stop_bits = 1);

  /** Destructor of HC12
   */
  virtual ~HC12();

  uint16_t htons(uint16_t data);
  uint16_t ntohs(uint16_t data);
  uint32_t htonl(uint32_t data);
  uint32_t ntohl(uint32_t data);

  /** Set HC12 to command mode
   *
   */
  void begin_command_mode();

  /** Set HC12 to transparent mode
   *
   */
  void end_command_mode();

  /** Set HC12 deep sleep
   *
   */
  bool cmd_sleep();

  /** Wake HC12 up
   *
   */
  bool cmd_wakeup();

  /** Set HC12 channel
   *
   */
  bool cmd_channel(uint32_t channel, bool unsafe_channels = false);

  /** Set HC12 UART TX mode
   *
   */
  bool cmd_uart_tx_mode(uint32_t mode);

  /** Set HC12 TX power in dBm
   *
   */
  bool cmd_tx_power(int32_t dBm);

  /** Set HC12 UART baud rate
   *
   */
  bool cmd_baudrate(uint32_t baud);

  /** Set HC12 UART comm format
   *
   */
  bool cmd_format(uint32_t data_bits = 8,
                  SerialBase::Parity parity = SerialBase::Parity::None,
                  uint32_t stop_bits = 1);

  /** Send data
   *
   */
  ssize_t send(const void *data, std::size_t length);

  /** Receive data
   *
   */
  ssize_t receive(void *data, std::size_t length);

private:
  PlatformMutex _mutex;
  bool _transparent_mode;
  DigitalOut _set;
  BufferedSerial _hc12;
};

#endif // MBED_HC12_H