#include "HC12.h"

HC12::HC12(PinName p_tx, PinName p_rx, PinName p_set, uint32_t baudrate,
           uint32_t data_bits, SerialBase::Parity parity, uint32_t stop_bits)
    : _set(p_set), _hc12(p_tx, p_rx) {
  _hc12.set_baud(baudrate);
  _hc12.set_format(
      /* bits */ data_bits,
      /* parity */ parity,
      /* stop bit */ stop_bits);
  _set = 1;
  _transparent_mode = true;
}

HC12::~HC12() {}

/**
 * Convert an unsigned short from host- to network byte order.
 *
 * @param n unsigned short in host byte order
 * @return n in network byte order
 */

uint16_t HC12::htons(uint16_t data) {
  return ((data & 0xff) << 8) | ((data & 0xff00) >> 8);
}

/**
 * Convert an unsigned short from network- to host byte order.
 *
 * @param n unsigned short in network byte order
 * @return n in host byte order
 */
uint16_t HC12::ntohs(uint16_t data) { return htons(data); }

/**
 * Convert an unsigned int from host- to network byte order.
 *
 * @param n unsigned int in host byte order
 * @return n in network byte order
 */
uint32_t HC12::htonl(uint32_t data) {
  return ((data & 0xff) << 24) | ((data & 0xff00) << 8) |
         ((data & 0xff0000UL) >> 8) | ((data & 0xff000000UL) >> 24);
}

/**
 * Convert an unsigned int from network- to host byte order.
 *
 * @param n unsigned int in network byte order
 * @return n in host byte order
 */
uint32_t HC12::ntohl(uint32_t data) { return htonl(data); }

void HC12::begin_command_mode() {
  if (_transparent_mode) {
    _mutex.lock();
    ThisThread::sleep_for(200ms);
    _set = 0;
    ThisThread::sleep_for(80ms);
    _transparent_mode = false;
  }
}

void HC12::end_command_mode() {
  if (!_transparent_mode) {
    _set = 1;
    ThisThread::sleep_for(80ms);
    _transparent_mode = true;
    _mutex.unlock();
  }
}

bool HC12::cmd_sleep() {
  if (_transparent_mode)
    return false;
  char buf[32] = {0x0};

  _hc12.write("AT+SLEEP", 8 * sizeof(char));
  ThisThread::sleep_for(50ms);
  _hc12.read(buf, sizeof(buf));

  if (NULL != strstr(buf, "OK+SLEEP")) {

    return true;
  }

  return false;
}

bool HC12::cmd_wakeup() {
  if (_transparent_mode)
    return false;
  char buf[32] = {0x0};

  _hc12.write("AT", 2 * sizeof(char));
  ThisThread::sleep_for(50ms);
  _hc12.read(buf, sizeof(buf));

  if (NULL != strstr(buf, "OK")) {

    return true;
  }

  return false;
}

bool HC12::cmd_channel(uint32_t channel, bool unsafe_channels) {
  if (_transparent_mode)
    return false;
  uint32_t max_channel = 100;
  if (unsafe_channels)
    max_channel = 127;
  if (channel < 1 || channel > max_channel) {
    return false;
  }
  char buf[32] = {0x0};
  sprintf(&buf[0], "AT+C%03u", channel);

  _hc12.write(&buf[0], 7 * sizeof(char));
  ThisThread::sleep_for(50ms);
  _hc12.read(buf, sizeof(buf));

  if (NULL != strstr(buf, "OK+C")) {

    return true;
  }

  return false;
}

bool HC12::cmd_uart_tx_mode(uint32_t mode) {
  if (_transparent_mode)
    return false;
  char buf[32] = {0x0};

  switch (mode) {
  case 1:

    _hc12.write("AT+FU1", 6 * sizeof(char));
    break;
  case 2:

    _hc12.write("AT+FU2", 6 * sizeof(char));
    break;
  case 3:

    _hc12.write("AT+FU3", 6 * sizeof(char));
    break;
  case 4:

    _hc12.write("AT+FU4", 6 * sizeof(char));
    break;
  default:

    return false;
  }
  ThisThread::sleep_for(50ms);
  _hc12.read(buf, sizeof(buf));

  if (NULL != strstr(buf, "OK+FU")) {

    return true;
  }

  return false;
}

bool HC12::cmd_tx_power(int32_t dBm) {
  if (_transparent_mode)
    return false;
  char buf[32] = {0x0};

  switch (dBm) {
  case -1:
    _hc12.write("AT+P1", 5 * sizeof(char));
    break;
  case 2:
    _hc12.write("AT+P2", 5 * sizeof(char));
    break;
  case 5:
    _hc12.write("AT+P3", 5 * sizeof(char));
    break;
  case 8:
    _hc12.write("AT+P4", 5 * sizeof(char));
    break;
  case 11:
    _hc12.write("AT+P5", 5 * sizeof(char));
    break;
  case 14:
    _hc12.write("AT+P6", 5 * sizeof(char));
    break;
  case 17:
    _hc12.write("AT+P7", 5 * sizeof(char));
    break;
  case 20:
    _hc12.write("AT+P8", 5 * sizeof(char));
    break;
  default:

    return false;
  }
  ThisThread::sleep_for(50ms);
  _hc12.read(buf, sizeof(buf));

  if (NULL != strstr(buf, "OK+P")) {

    return true;
  }

  return false;
}

bool HC12::cmd_baudrate(uint32_t baud) {
  if (_transparent_mode)
    return false;
  char buf[32] = {0x0};

  switch (baud) {
  case 1200:
    _hc12.write("AT+B1200", 8 * sizeof(char));
    break;
  case 2400:
    _hc12.write("AT+B2400", 8 * sizeof(char));
    break;
  case 4800:
    _hc12.write("AT+B4800", 8 * sizeof(char));
    break;
  case 9600:
    _hc12.write("AT+B9600", 8 * sizeof(char));
    break;
  case 19200:
    _hc12.write("AT+B19200", 9 * sizeof(char));
    break;
  case 38400:
    _hc12.write("AT+B38400", 9 * sizeof(char));
    break;
  case 57600:
    _hc12.write("AT+B57600", 9 * sizeof(char));
    break;
  case 115200:
    _hc12.write("AT+B115200", 10 * sizeof(char));
    break;
  default:

    return false;
  }
  ThisThread::sleep_for(50ms);
  _hc12.read(buf, sizeof(buf));

  if (NULL != strstr(buf, "OK+B")) {
    _hc12.set_baud(baud);

    return true;
  }

  return false;
}

bool HC12::cmd_format(uint32_t data_bits, SerialBase::Parity parity,
                      uint32_t stop_bits) {
  if (_transparent_mode)
    return false;
  if (data_bits < 7 || data_bits > 8)
    return false;
  if (stop_bits < 1 || stop_bits > 2)
    return false;
  if (parity == SerialBase::Parity::Forced0 ||
      parity == SerialBase::Parity::Forced0)
    return false;

  char buf[32] = {0x0};
  char _parity = 'N';

  if (parity == SerialBase::Parity::Even) {
    _parity = 'E';
  } else if (parity == SerialBase::Parity::Odd) {
    _parity = 'O';
  }

  sprintf(&buf[0], "AT+U%u%c%u", data_bits, _parity, stop_bits);

  _hc12.write(&buf[0], 7 * sizeof(char));
  ThisThread::sleep_for(50ms);
  _hc12.read(buf, sizeof(buf));

  if (NULL != strstr(buf, "OK+U")) {
    _hc12.set_format(data_bits, parity, stop_bits);

    return true;
  }

  return false;
}

ssize_t HC12::send(const void *data, std::size_t length) {
  if (!_transparent_mode)
    return EIO;

  return _hc12.write(data, length);
}

ssize_t HC12::receive(void *data, std::size_t length) {
  if (!_transparent_mode)
    return EIO;

  return _hc12.read(data, length);
}