/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"

#include "ble/services/UARTService.h"
#include "LSM6DS3/LSM6DS3.h"

#include "nrf51_rtc.h"

LSM6DS3 imu(p30, p7);
Serial pc(p9, p11);

#define NEED_CONSOLE_OUTPUT 1 /* Set this if you need debug messages on the console;
                               * it will have an impact on code-size and power consumption. */

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

BLEDevice  ble;
// Status lights
DigitalOut ledAlive(LED1);
DigitalOut ledIntrG(LED3);
DigitalOut ledIntrO(LED4);
// Interrupt pin connection
InterruptIn intrGX(p14);    //INT1 (Gyro & Accel Interrupt)
InterruptIn intrGXO(p13);   //INT2 (Gyro & Accel Data Ready)

UARTService *uartServicePtr;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    DEBUG("Disconnected!\n\r");
    DEBUG("Restarting the advertising process\n\r");
    ble.startAdvertising();
}

void onDataWritten(const GattWriteCallbackParams *params)
{
    if ((uartServicePtr != NULL) && (params->handle == uartServicePtr->getTXCharacteristicHandle())) {
        uint16_t bytesRead = params->len;
        uint8_t value = *(params->data);
        uint16_t status;
        DEBUG("received %u bytes\n\r", bytesRead);
        DEBUG("recevied %d data\n\r", value);
        
        switch(value) {
            case 49:
                DEBUG("Off power\n\r");
                status = imu.begin(imu.G_SCALE_245DPS, imu.A_SCALE_2G,
                                   imu.G_POWER_DOWN, imu.A_POWER_DOWN);
                break;
            case 50:
                DEBUG("Accelerometer Low, Gyro Off\n\r");
                status = imu.begin(imu.G_SCALE_245DPS, imu.A_SCALE_2G,
                                   imu.G_POWER_DOWN, imu.A_ODR_13);
                break;
            case 51:
                DEBUG("Accelerometer High, Gyro Off\n\r");
                status = imu.begin(imu.G_SCALE_245DPS, imu.A_SCALE_8G,
                                   imu.G_POWER_DOWN, imu.A_ODR_6660);
                break;
            case 52:
                DEBUG("Accelerometer High, Gyro High\n\r");
                status = imu.begin(imu.G_SCALE_2000DPS, imu.A_SCALE_8G,
                                   imu.G_ODR_1660, imu.A_ODR_6660);
                break;
            default:
                DEBUG("Default power\n\r");
                status = imu.begin();
                break;
        }
                
        pc.printf("LSM6DS3 WHO_AM_I's returned: 0x%X\r\n", status);
        pc.printf("Should be 0x69\r\n");
        
        ble.updateCharacteristicValue(uartServicePtr->getRXCharacteristicHandle(), params->data, bytesRead);
    }
}

void periodicCallback(void)
{
    ble.waitForEvent();
    ledAlive = !ledAlive;
}

// Interrupt signal
void flip(DigitalOut led) {
    led = !led;
}

void flip3() {
//    pc.printf(" ++++ Data Change Triggered!! ++++ \n\r");
    flip(ledIntrO);
}

void flip4() {
//    pc.printf(" ++++ G/X Triggered!! ++++ \n\r");
    flip(ledIntrG);
}

int main(void)
{
    time_t rawtime = 0;
    
    // turn all LED off
    ledAlive = 1;
    ledIntrG = 1;
    ledIntrO = 1;
    
    Ticker ticker;
    ticker.attach(periodicCallback, 1);

    DEBUG("Initialising the nRF51822\n\r");
    pc.printf("PC: Initialising the nRF51822\n\r");
    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(onDataWritten);

    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)"BLE UART", sizeof("BLE UART") - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));

    ble.setAdvertisingInterval(1000); /* 1000ms; in multiples of 0.625ms. */
    ble.startAdvertising();

    UARTService uartService(ble);
    uartServicePtr = &uartService;
    
    // setup baud rate
    pc.baud(115200);
    
    // setup Interrupt mode
    intrGXO.mode(PullUp);
    intrGX.mode(PullUp);

    // status light will be lit based on sensor/data change
    intrGX.rise(&flip4);
    intrGXO.rise(&flip3);

    while (true) {
        rawtime = rtc.time();
    
        imu.readAccel();
        imu.readGyro();
        // no magnetometer available for this sensor
        pc.printf("%d,%d,%d,%d,%d,%d,%d,0.000000,0.000000,0.000000\r\n", rawtime,imu.ax_raw,imu.ay_raw,imu.az_raw,imu.gx_raw,imu.gy_raw,imu.gz_raw);

        //imu.readIntr();
//        pc.printf("intr: %f\r\n", imu.intr);
//        if(imu.intr > 0) {
//            pc.printf(" == interrupted == \r\n");
//            flip(LED2);
//        }
    }
}
