/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"

#include "ble/services/UARTService.h"
#include "BNO055/BNO055.h"

#include "nrf51_rtc.h"

BNO055 imu(p30, p7);
Serial pc(p9, p11);

#define NEED_CONSOLE_OUTPUT 1 /* Set this if you need debug messages on the console;
                               * it will have an impact on code-size and power consumption. */

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

BLEDevice  ble;
// Status lights
DigitalOut ledAlive(LED1);
DigitalOut ledIntrG(LED3);
// Interrupt pin connection
InterruptIn intrGX(p14);    //INT (Gyro & Accel Interrupt)

UARTService *uartServicePtr;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    DEBUG("Disconnected!\n\r");
    DEBUG("Restarting the advertising process\n\r");
    ble.startAdvertising();
}

void onDataWritten(const GattWriteCallbackParams *params)
{   
    if ((uartServicePtr != NULL) && (params->handle == uartServicePtr->getTXCharacteristicHandle())) {
        uint16_t bytesRead = params->len;
        uint8_t value = *(params->data);
        DEBUG("received %u bytes\n\r", bytesRead);
        DEBUG("recevied %d data\n\r", value);

        imu.reset();
        
        // Set power mode, POWER_MODE_NORMAL(default), POWER_MODE_LOWPOWER, or POWER_MODE_SUSPEND(sleep)
        // By default, I2C will pull high when bus is free(pg90 of datasheet)
        switch(value) {
            case 49:
                DEBUG("Off power\n\r");
                imu.setpowermode(POWER_MODE_SUSPEND);
                break;
            case 50:
                DEBUG("Accelerometer Low, Gyro Off, Magnetometer Off\r");
                imu.setpowermode(POWER_MODE_LOWPOWER);
                break;
            case 51:
                DEBUG("Accelerometer High, Gyro Off\n\r");
//                imu.setpowermode(POWER_MODE_NORMAL);
                imu.setpowermode3();
                break;
            case 52:
                DEBUG("Accelerometer High, Gyro High\n\r");
                imu.setpowermode(POWER_MODE_NORMAL);
                break;
            default:
                DEBUG("Default power\n\r");
                imu.setpowermode(POWER_MODE_NORMAL);
                break;
        }
                
        if (imu.check())
        {
            pc.printf("BNO055 WHO_AM_I's returned: 0x%X\r\n", imu.ID.id);
            pc.printf("Should be 0xA0\r\n");
            imu.initIntr();
        }
        
        ble.updateCharacteristicValue(uartServicePtr->getRXCharacteristicHandle(), params->data, bytesRead);
    }
}

void periodicCallback(void)
{
    ble.waitForEvent();
    ledAlive = !ledAlive;
}

// Interrupt signal
void flip(DigitalOut led) {
    led = !led;
}

void flip4() {
//    pc.printf(" ++++ G/X Triggered!! ++++ \n\r");
    flip(ledIntrG);
    imu.resetIntr();
}

int main(void)
{
    time_t rawtime = 0;
    
    // turn all LED off
    ledAlive = 1;
    ledIntrG = 1;
    
    Ticker ticker;
    ticker.attach(periodicCallback, 1);

    DEBUG("Initialising the nRF51822\n\r");
    pc.printf("PC: Initialising the nRF51822\n\r");
    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(onDataWritten);

    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)"BLE UART", sizeof("BLE UART") - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));

    ble.setAdvertisingInterval(1000); /* 1000ms; in multiples of 0.625ms. */
    ble.startAdvertising();

    UARTService uartService(ble);
    uartServicePtr = &uartService;
    
    // setup baud rate and reset sensor
    pc.baud(115200);
    
    // setup Interrupt mode
    intrGX.mode(PullUp);

    // status light will be lit based on sensor/data change
    intrGX.rise(&flip4);

    while (true) {
        rawtime = rtc.time();
        
        imu.setmode(OPERATION_MODE_AMG);
        
        imu.get_accel();
        imu.get_gyro();
        imu.get_mag();
        pc.printf("%d,%d,%d,%d,%d,%d,%d,%d,%d,%d\r\n", rawtime,imu.accel.rawx,imu.accel.rawy,imu.accel.rawz,imu.gyro.rawx, imu.gyro.rawy, imu.gyro.rawz,imu.mag.rawx, imu.mag.rawy, imu.mag.rawz);
        
        // Interrupt stats: This is only enabled in Low Power Mode
        // 0 - no interrupt
        // 64 - interrupt
        // 128 - sleep
        //imu.get_intr();
//        pc.printf("intr: %d\r\n", imu.intr);
//        if(imu.intr == 64) {
//            pc.printf(" == interrupted == \r\n");
//            imu.resetIntr();
//            pc.printf("rx before: %d\r\n", imu.test3);
//            pc.printf("rx after: %d\r\n", imu.test4);
//        }
    }
}
