/*************************************************/
/*           Stealth "Radar" C API               */
/*************************************************/

#ifndef _Stealth_Radar_h_
#define _Stealth_Radar_h_

#include <time.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/********************************************************************/
/* API that the client calls to send requests to the server         */
/********************************************************************/

extern int RadarDefaultTimeoutMS;

/* Error/Status Codes and Return Type */
enum radar_result_values
{
   RADAR_OK                   = 0,        /* everything's awesome */
   RADAR_DISCONNECTED         = -100,     /* can't use API because transport appears disconnected */
   RADAR_SERVER_ERROR         = -101,     /* reached the server, but something went wrong there */
   RADAR_NETWORK_ERROR        = -102,     /* something went wrong with the socket/transport locally */
   RADAR_SEND_TIMEOUT         = -103,     /* sending data timed out */
   RADAR_RECV_TIMEOUT         = -104,     /* receiving data timed out */
   RADAR_RECV_CORRUPT         = -105,     /* receiving data got corrupted or out-of-sync */
   RADAR_BAD_PARAM_1          = -201,     /* radar call parameter 1 was invalid */
   /* ... */
   RADAR_BAD_PARAM_99         = -299      /* radar call parameter 99 was invalid */
};

typedef int radar_result_t;

/* Connection/Disconnection */
radar_result_t Radar_Hello(
   uint32_t firmwareBuild,
   time_t upTime,
   time_t currentTime,
   const char* serialNumber);


radar_result_t Radar_Goodbye(void);

/* Sending Samples */
enum radar_trigger_values
{
   RADAR_TRIGGER_INTERVAL          = 1000, /* Sample is published due to regular interval elapse */
   RADAR_TRIGGER_BATTERY_DISCHARGE = 1001, /* Sample is published due to usage (battery discharge) event */
   RADAR_TRIGGER_BATTERY_CHARGE    = 1002  /* Sample is published due to receiving power (battery charge) event */
};

typedef int radar_trigger_t;

/* Sending Sample Data */
radar_result_t Radar_PublishSample(
   radar_trigger_t trigger,          /* Why is this sample being sent (i.e. to which table) */
   time_t beginTime,                 /* First time_t second covered by this sample */
   time_t endTime,                   /* First time_t second *not* covered by this sample */
   int32_t VspAvg,
   int32_t VspMin,
   int32_t VspMax,
   int32_t VoemAvg,
   int32_t VoemMin,
   int32_t VoemMax,
   int32_t V3Avg,
   int32_t V3Min,
   int32_t V3Max,
   int32_t V4Avg,
   int32_t V4Min,
   int32_t V4Max,   
   int32_t A1Avg,
   int32_t A1Min,
   int32_t A1Max,
   int32_t A2Avg,
   int32_t A2Min,
   int32_t A2Max,
   int32_t temperatureAvg,              /* in oC */
   int32_t temperatureMin,              /* in oC */
   int32_t temperatureMax,              /* in oC */
   int32_t sampleCount,
   int32_t dischargeCount,/* how many seconds of the interval fromTime...toTime was battery discharging */
   int32_t chargeCount /* how many seconds of the interval fromTime...toTime was battery charging */
);

/* Sending Status Updates */

radar_result_t Radar_PublishStatus(
   time_t atTime,                 /* The time_t moment that this status took effect */
   uint8_t picState,
   uint8_t picK,
   uint8_t batteryState,
   int statusCode,                /* 0 for OK, or error code from radar_result_t or elsewhere */
   const char* message,           /* Optional (may be NULL) text description of status, up to 4096 bytes */
   int32_t Vsp,                   /* latest values... */
   int32_t Voem,
   int32_t V3,
   int32_t V4,
   int32_t A1,
   int32_t A2,
   int32_t temperature
);

/* Getting Settings */
typedef char hostname_t[128];
radar_result_t Radar_GetTelemetrySettings(
   int32_t* dischargeThreshold,     /* threshold above which the stealth battery current is considered 'discharging' */
   int32_t* chargeThreshold,        /* threshold below which the stealth battery current is considered 'charging' */
   int32_t* currentHysteresis,      /* amount to ease the threshold for staying in a charging/discharging state */
   int32_t* regularSampleInterval,  /* interval, in seconds, at which to send regularly-sampled data; 0 means "don't" */
   int32_t* settingsCheckInterval,  /* interval, in seconds, at which to check for new settings */
   /* other settings you can think of here ... ? */
   hostname_t primaryHost,
   uint32_t* primaryPort,
   hostname_t secondaryHost,
   uint32_t* secondaryPort,
   uint32_t* statusUpdateEvictionTime, /* eviction timer for pushing latest status if no change specifically triggered it */
   uint32_t* voltageFilterLengthInSamples, /* exponential moving average length in samples; alpha = 2 / (this + 1) */
   int32_t* vspUpdateThreshold, /* how many mV deviation before a filtered or jumped Vsp change causes a status update */
   int32_t* vspJumpThreshold, /* how many mV deviation before a step in the value bypasses the noise filter and jumps to latest value */
   int32_t* voemUpdateThreshold,
   int32_t* voemJumpThreshold,
   int32_t* v3UpdateThreshold,
   int32_t* v3JumpThreshold,
   int32_t* v4UpdateThreshold,
   int32_t* v4JumpThreshold,
   uint32_t* currentFilterLengthInSamples, /* exponential moving average length in samples; alpha = 2 / (this + 1) */
   int32_t* a1UpdateThreshold,
   int32_t* a1JumpThreshold,
   int32_t* a2UpdateThreshold,
   int32_t* a2JumpThreshold,
   uint32_t* temperatureFilterLengthInSamples, /* exponential moving average length in samples; alpha = 2 / (this + 1) */
   int32_t* temperatureUpdateThreshold,
   int32_t* temperatureJumpThreshold,
   int32_t* useFilteredA1ForBatteryState,
   int32_t* minumEventSampleDuration
);

/********************************************************************/
/* Callbacks that the client implements using the transport
  (MSTAS TCPSocket, SSL/TLS, etc.) of their choice */
/********************************************************************/

/* Radar_SendBytes
      Parameters
         bytes       : array of bytes to send
         size        : number of bytes in 'bytes' array; may be 0.
         timeoutMS   : number of milliseconds to wait/select() on
                       network. 0 = immediate, (uint32_t)-1 = infinite.
      Return Value
         Nonnegative on success -- Number of bytes actually sent, if
                                   successful.  0 is valid, e.g. if timeoutMS
                                   was 0 and there was no room in the xmit
                                   buffer.
         Negative on failure -- ideally one of the values of radar_result_t or
                                well-known socket error like -ESHUTDOWN.
         This value is returned by whichever Radar_*() call had invoked
         Radar_SendBytes().
*/
int Radar_SendBytes(
   const void* bytes,
   size_t size,
   uint32_t timeoutMS);

/* Radar_RecvBytes
      Parameters
         bytes       : array of bytes to recv into
         maxSize     : maximum number of bytes to recv into 'bytes' array
         timeoutMS   : number of milliseconds to wait/select() on
                       network. 0 = immediate, (uint32_t)-1 = infinite.
      Return Value
         Nonnegative on success -- Number of bytes actually received, if
                                   successful.  0 is valid, e.g. if timeoutMS
                                   was 0 and there was nothing in the recv
                                   buffer.
         Negative on failure -- ideally one of the values of radar_result_t or
                                well-known socket error like -ESHUTDOWN.
         This value is returned by whichever Radar_*() call had invoked
         Radar_SendBytes().

         NOTE: a normal berkeley sockets recv() call would indicate a closed
               socket by returning 0 bytes, but this function can return
               0 bytes without it meaning the socket has ended.  If this
               function does detect a lost/ended socket, and less than the
               requested bytes are available, it should return
               RADAR_DISCONNECTED.
*/
int Radar_RecvBytes(
   void* bytes,
   size_t maxSize,
   uint32_t timeoutMS);

/* Radar_Flush
      Flushes all bytes queued for transmit by Radar_SendBytes().

      Parameters
         timeoutMS : timeout for flushing any transmittable bytes
      Return Value
         Negative on failure -- ideally one of the values of radar_result_t or
                                well-known socket error like -ESHUTDOWN.
         Zero on success.
*/
int Radar_Flush(uint32_t timeoutMS);


/* Radar_RecvResetChecksum
      Returns, and internally resets, the per-byte-received checksum.
      Used for validating messages received.

      Return Value
        Current value of the 32-bit uint32_t checksum of every byte
        since the last call to this function.

      Side Effect
        Resets the (implementor's) internal checksum counter to default/0.
*/
uint32_t Radar_RecvResetChecksum(void);


#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* _Stealth_Radar_h_ */
