/*************************************************/
/*           Stealth "Radar" C API               */
/*************************************************/

#include "StealthRadarAPI.h"
#include <string.h>
#include <stdio.h>



/********************************************************************/
/* API that the client calls to send requests to the server         */
/********************************************************************/

int RadarDefaultTimeoutMS = 2000;
/* All messages start with a 0xaa hail byte and an 8-bit message ID */

/* Macros to convert from this processor to network order (big endian) */
#if defined(__BIG_ENDIAN) || (defined(__BYTE_ORDER__) && (__BYTE_ORDER__ == __ORDER_BIG_ENDIAN__))
   #define ENDIANIZE16(x)
   #define ENDIANIZE32(x)
#elif defined(__ARMCC_VERSION) || (defined(__BYTE_ORDER__) && (__BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__))
   #define ENDIANIZE16(x) { *(uint16_t*)&x = ( (( (*(uint16_t*)&x) << 8) & 0xff00) | (( (*(uint16_t*)&x) >> 8) & 0xff) ); }
   #define ENDIANIZE32(x) { *(uint32_t*)&x = ( (( *(uint32_t*)&x  << 24) & 0xff000000) | (( *(uint32_t*)&x << 8) & 0xff0000) | \
                            (( *(uint32_t*)&x >> 8) & 0xff00) | (( *(uint32_t*)&x >> 24) & 0xff) ); }
#else
   #error Needs to be big endian or little endian.
#endif

#define WORSE_ERROR(original, alternative) \
   ((radar_result_t)(((int)original < (int)RADAR_OK) ? ((int)alternative) : ((int)original)))
   
   
int Radar_Message_Sequence = 10000;

radar_result_t Radar_Hello(
   uint32_t firmwareBuild,
   time_t upTime,
   time_t currentTime,
   const char* serialNumber)
{
   int result = RADAR_OK;
   uint8_t hail = 0xaa;
   uint8_t id = 0x01;
   int32_t temp32;
   uint16_t temp16 = (uint16_t)strlen(serialNumber);
   uint32_t remoteCurrentTime = 0xFFFFFFFF;
   radar_result_t remoteResult = RADAR_OK;
   
   /* Send 64-bit start-of-stream marker to server (we shouldn't need this but modem seems to keep and resend
      stale packets on subsequent sockets...) */
   temp32 = (int32_t)0x859ABD33;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t)0x4589257B;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   /* Send message to server */
   if ((result = Radar_SendBytes(&hail, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   if ((result = Radar_SendBytes(&id, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   temp32 = (int32_t)Radar_Message_Sequence++;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t)firmwareBuild;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t)upTime;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t)currentTime;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   ENDIANIZE16(temp16);
   if ((result = Radar_SendBytes(&temp16, sizeof(temp16), RadarDefaultTimeoutMS)) != sizeof(temp16)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   if ((result = Radar_SendBytes(serialNumber, strlen(serialNumber), RadarDefaultTimeoutMS)) != strlen(serialNumber)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   /* Flush it all to server */
   result = Radar_Flush(RadarDefaultTimeoutMS);
   if (result != RADAR_OK) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   /* Get acknowledgement from server */
   if ((result = Radar_RecvBytes(&remoteCurrentTime, sizeof(remoteCurrentTime), RadarDefaultTimeoutMS)) != sizeof(remoteResult)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(remoteCurrentTime);
   if (remoteCurrentTime != currentTime)
   {
      // Server must repeat back exactly the current time we sent, so we know socket is in sync.
      return WORSE_ERROR(result, RADAR_RECV_CORRUPT);
   }
   
   if ((result = Radar_RecvBytes(&remoteResult, sizeof(remoteResult), RadarDefaultTimeoutMS)) != sizeof(remoteResult)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(remoteResult);

   return remoteResult;
}

radar_result_t Radar_Goodbye()
{
   int result = RADAR_OK;
   uint8_t hail = 0xaa;
   uint8_t id = 0x02;
   int32_t temp32 = 0;
   radar_result_t remoteResult = RADAR_OK;

   /* Send message to server */
   if ((result = Radar_SendBytes(&hail, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   if ((result = Radar_SendBytes(&id, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t)Radar_Message_Sequence++;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   /* Flush it all to server */
   result = Radar_Flush(RadarDefaultTimeoutMS);
   if (result != RADAR_OK) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   /* Get acknowledgement from server */
   if ((result = Radar_RecvBytes(&remoteResult, sizeof(remoteResult), RadarDefaultTimeoutMS)) != sizeof(remoteResult)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(remoteResult);

   /* Server should close the socket now, and so should the SoM on its end. */

   return remoteResult;
}

/* Sending Sample Data */
radar_result_t Radar_PublishSample(
   radar_trigger_t trigger,          /* Why is this sample being sent (i.e. to which table) */
   time_t beginTime,                 /* First time_t second covered by this sample */
   time_t endTime,                   /* First time_t second *not* covered by this sample */
   int32_t VspAvg,
   int32_t VspMin,
   int32_t VspMax,
   int32_t VoemAvg,
   int32_t VoemMin,
   int32_t VoemMax,
   int32_t V3Avg,
   int32_t V3Min,
   int32_t V3Max,
   int32_t V4Avg,
   int32_t V4Min,
   int32_t V4Max,   
   int32_t A1Avg,
   int32_t A1Min,
   int32_t A1Max,
   int32_t A2Avg,
   int32_t A2Min,
   int32_t A2Max,
   int32_t temperatureAvg,              /* in oC */
   int32_t temperatureMin,              /* in oC */
   int32_t temperatureMax,              /* in oC */
   int32_t sampleCount,
   int32_t dischargeCount,/* how many seconds of the interval fromTime...toTime was battery discharging */
   int32_t chargeCount /* how many seconds of the interval fromTime...toTime was battery charging */
)
{
   int result = RADAR_OK;
   uint8_t hail = 0xaa;
   uint8_t id = 0x10;
   int32_t temp32;
   radar_result_t remoteResult = RADAR_OK;

   /* Send message to server */
   if ((result = Radar_SendBytes(&hail, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   if ((result = Radar_SendBytes(&id, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t)Radar_Message_Sequence++;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) trigger;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) beginTime;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) endTime;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) VspAvg;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) VspMin;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) VspMax;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) VoemAvg;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) VoemMin;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) VoemMax;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) V3Avg;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) V3Min;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) V3Max;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) V4Avg;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) V4Min;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) V4Max;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) A1Avg;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) A1Min;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) A1Max;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) A2Avg;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) A2Min;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) A2Max;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) temperatureAvg;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   temp32 = (int32_t) temperatureMin;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   temp32 = (int32_t) temperatureMax;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) sampleCount;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   temp32 = (int32_t) dischargeCount;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) chargeCount;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   /* Flush it all to server */
   result = Radar_Flush(RadarDefaultTimeoutMS);
   if (result != RADAR_OK) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   /* Get acknowledgement from server */
   if ((result = Radar_RecvBytes(&remoteResult, sizeof(remoteResult), RadarDefaultTimeoutMS)) != sizeof(remoteResult)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(remoteResult);

   /* Server should close the socket now, and so should the SoM on its end. */
   return remoteResult;
}

radar_result_t Radar_PublishStatus(
   time_t atTime,                 /* The time_t moment that this status took effect */
   uint8_t picState,
   uint8_t picK,
   uint8_t batteryState,
   int statusCode,                /* 0 for OK, or error code from radar_result_t or elsewhere */
   const char* message,
   int32_t Vsp,                   /* latest values... */
   int32_t Voem,
   int32_t V3,
   int32_t V4,
   int32_t A1,
   int32_t A2,
   int32_t temperature)           /* Optional (may be NULL) text description of status, up to 4096 bytes */
{
   int result = RADAR_OK;
   uint8_t hail = 0xaa;
   uint8_t id = 0x11;
   uint16_t temp16 = 0;
   int32_t temp32;
   uint8_t temp8;
   radar_result_t remoteResult = RADAR_OK;

   /* Send message to server */
   if ((result = Radar_SendBytes(&hail, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   if ((result = Radar_SendBytes(&id, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t)Radar_Message_Sequence++;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) atTime;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp8 = (int32_t) picState;
   if ((result = Radar_SendBytes(&temp8, sizeof(temp8), RadarDefaultTimeoutMS)) != sizeof(temp8)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp8 = (int32_t) picK;
   if ((result = Radar_SendBytes(&temp8, sizeof(temp8), RadarDefaultTimeoutMS)) != sizeof(temp8)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp8 = (int32_t) batteryState;
   if ((result = Radar_SendBytes(&temp8, sizeof(temp8), RadarDefaultTimeoutMS)) != sizeof(temp8)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) statusCode;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   if (message != NULL) temp16 = (int16_t)strlen(message);
   ENDIANIZE16(temp16);
   if ((result = Radar_SendBytes(&temp16, sizeof(temp16), RadarDefaultTimeoutMS)) != sizeof(temp16)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   if (temp16 != 0)
   {
      if ((result = Radar_SendBytes(message, strlen(message), RadarDefaultTimeoutMS)) != strlen(message)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   }


   temp32 = (int32_t) Vsp;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) Voem;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) V3;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) V4;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   temp32 = (int32_t) A1;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   //printf("Sent A1 as %d\r\n", A1);

   temp32 = (int32_t) A2;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   temp32 = (int32_t) temperature;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   /* Flush it all to server */
   result = Radar_Flush(RadarDefaultTimeoutMS);
   if (result != RADAR_OK) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);

   /* Get acknowledgement from server */
   if ((result = Radar_RecvBytes(&remoteResult, sizeof(remoteResult), RadarDefaultTimeoutMS)) != sizeof(remoteResult)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(remoteResult);

   /* Server should close the socket now, and so should the SoM on its end. */
   return remoteResult;
}

radar_result_t Radar_GetTelemetrySettings(
   int32_t* dischargeThreshold,     /* threshold above which the stealth battery current is considered 'discharging' */
   int32_t* chargeThreshold,        /* threshold below which the stealth battery current is considered 'charging' */
   int32_t* currentHysteresis,      /* amount to ease the threshold for staying in a charging/discharging state */
   int32_t* regularSampleInterval,  /* interval, in seconds, at which to send regularly-sampled data; 0 means "don't" */
   int32_t* settingsCheckInterval,  /* interval, in seconds, at which to check for new settings */
   hostname_t primaryHost,
   uint32_t* primaryPort,
   hostname_t secondaryHost,
   uint32_t* secondaryPort,
   uint32_t* statusUpdateEvictionTime, /* eviction timer for pushing latest status if no change specifically triggered it */
   uint32_t* voltageFilterLengthInSamples, /* exponential moving average length in samples; alpha = 2 / (this + 1) */
   int32_t* vspUpdateThreshold, /* how many mV deviation before a filtered or jumped Vsp change causes a status update */
   int32_t* vspJumpThreshold, /* how many mV deviation before a step in the value bypasses the noise filter and jumps to latest value */
   int32_t* voemUpdateThreshold,
   int32_t* voemJumpThreshold,
   int32_t* v3UpdateThreshold,
   int32_t* v3JumpThreshold,
   int32_t* v4UpdateThreshold,
   int32_t* v4JumpThreshold,
   uint32_t* currentFilterLengthInSamples, /* exponential moving average length in samples; alpha = 2 / (this + 1) */
   int32_t* a1UpdateThreshold,
   int32_t* a1JumpThreshold,
   int32_t* a2UpdateThreshold,
   int32_t* a2JumpThreshold,
   uint32_t* temperatureFilterLengthInSamples, /* exponential moving average length in samples; alpha = 2 / (this + 1) */
   int32_t* temperatureUpdateThreshold,
   int32_t* temperatureJumpThreshold,
   int32_t* useFilteredA1ForBatteryState,
   int32_t* minumEventSampleDuration
   )
{
   int result = RADAR_OK;
   uint8_t hail = 0xaa;
   uint8_t id = 0x12;
   int32_t temp32 = 0;
   hostname_t tempHostnameString = { 0 };
   radar_result_t remoteResult = RADAR_OK;
   
   printf("GETTING TELEMETRY SETTINGS.\r\n");

   /* Send message to server */
   if ((result = Radar_SendBytes(&hail, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   if ((result = Radar_SendBytes(&id, 1, RadarDefaultTimeoutMS)) != 1) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   temp32 = (int32_t)Radar_Message_Sequence++;
   ENDIANIZE32(temp32);
   if ((result = Radar_SendBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   /* Flush it all to server */
   result = Radar_Flush(RadarDefaultTimeoutMS);
   if (result != RADAR_OK) return WORSE_ERROR(result, RADAR_SEND_TIMEOUT);
   
   wait_ms(50); // TODO: This seems to improve reliability--is it the radio?.  TODO: root cause and remove all these wait_ms's.

   /* Reset the checksum before recv'ing bytes */
   Radar_RecvResetChecksum();

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (dischargeThreshold) *dischargeThreshold = temp32;
   wait_ms(10);
 
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (chargeThreshold) *chargeThreshold = temp32;
   wait_ms(10);
   
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (currentHysteresis) *currentHysteresis = temp32;
   wait_ms(10);   

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (regularSampleInterval) *regularSampleInterval = temp32;
   wait_ms(10);
  
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (settingsCheckInterval) *settingsCheckInterval = temp32;
   wait_ms(10);
   
   if ((result = Radar_RecvBytes(&tempHostnameString, sizeof(tempHostnameString), RadarDefaultTimeoutMS)) != sizeof(tempHostnameString)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   if (primaryHost)
   {
      strncpy(primaryHost, tempHostnameString, sizeof(tempHostnameString));
   }
   wait_ms(10);
   
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (primaryPort) *primaryPort = temp32;
   wait_ms(10);
   
   if ((result = Radar_RecvBytes(&tempHostnameString, sizeof(tempHostnameString), RadarDefaultTimeoutMS)) != sizeof(tempHostnameString)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   if (secondaryHost)
   {
      strncpy(secondaryHost, tempHostnameString, sizeof(tempHostnameString));
   }
   wait_ms(10);
   
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (secondaryPort) *secondaryPort = temp32;
   wait_ms(10);
   
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (statusUpdateEvictionTime) *statusUpdateEvictionTime = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (voltageFilterLengthInSamples) *voltageFilterLengthInSamples = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (vspUpdateThreshold) *vspUpdateThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (vspJumpThreshold) *vspJumpThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (voemUpdateThreshold) *voemUpdateThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (voemJumpThreshold) *voemJumpThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (v3UpdateThreshold) *v3UpdateThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (v3JumpThreshold) *v3JumpThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (v4UpdateThreshold) *v4UpdateThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (v4JumpThreshold) *v4JumpThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (currentFilterLengthInSamples) *currentFilterLengthInSamples = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (a1UpdateThreshold) *a1UpdateThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (a1JumpThreshold) *a1JumpThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (a2UpdateThreshold) *a2UpdateThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (a2JumpThreshold) *a2JumpThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (temperatureFilterLengthInSamples) *temperatureFilterLengthInSamples = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (temperatureUpdateThreshold) *temperatureUpdateThreshold = temp32;
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (temperatureJumpThreshold) *temperatureJumpThreshold = temp32;
   wait_ms(10);
   
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (useFilteredA1ForBatteryState) *useFilteredA1ForBatteryState = temp32;   
   wait_ms(10);

   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   if (minumEventSampleDuration) *minumEventSampleDuration = temp32;   
   wait_ms(10);

//   printf("GETTING TELEMETRY SETTINGS GOT THRU TEMP USE FILT A1.\r\n");// TODO CLeanup
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   const uint32_t sentinelValue = temp32;
   if (sentinelValue != 0xdeadbeef) return RADAR_RECV_CORRUPT;
   wait_ms(10);
   
//   printf("GETTING TELEMETRY SETTINGS GOT THRU SENTINEL.\r\n");// TODO CLeanup

   /* Get acknowledgement status from server */
   if ((result = Radar_RecvBytes(&remoteResult, sizeof(remoteResult), RadarDefaultTimeoutMS)) != sizeof(remoteResult)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(remoteResult);
   wait_ms(10);
   
  // printf("GETTING TELEMETRY SETTINGS GOT THRU STATUS.\r\n");// TODO CLeanup
   
   uint32_t calculatedChecksum = Radar_RecvResetChecksum();
   if ((result = Radar_RecvBytes(&temp32, sizeof(temp32), RadarDefaultTimeoutMS)) != sizeof(temp32)) return WORSE_ERROR(result, RADAR_RECV_TIMEOUT);
   ENDIANIZE32(temp32);
   uint32_t receivedChecksum = temp32;
   if (receivedChecksum != calculatedChecksum)
   {
      printf("RECV CHECKSUM ERROR: Calculated %08x, told %08x\r\n", calculatedChecksum, receivedChecksum);
      return RADAR_RECV_CORRUPT;
   }

   // printf("GETTING TELEMETRY SETTINGS GOT THRU CHECKSUM.\r\n");// TODO CLeanup

   /* Server should close the socket now, and so should the SoM on its end. */
   return remoteResult;
}
