
#ifndef _StatusUpdateManager_h_
#define _StatusUpdateManager_h_

#include "mbed.h"
#include "StatusUpdateBacklog.h"
#include "SystemTimeKeeper.h"
#include "SecondTimer.h"
    
enum BatteryState
{
    BatteryStateUnspecified = 0,
    BatteryStateQuiescent = 1,
    BatteryStateCharging = 2,
    BatteryStateDischarging = 3
};

class StatusUpdateManager
{
public:  
    StatusUpdateManager() : 
        m_picConnected(true),
        m_picStateSet(false),
        m_picState(0xFFFF), // invalid default values
        m_picK(0xFFFF), 
        m_batteryState(BatteryStateUnspecified),
        m_lastVsp(-1e19F),
        m_lastVoem(-1e19F),
        m_lastV3(-1e19F),
        m_lastV4(-1e19F),
        m_lastA1(-1e19F),
        m_lastA2(-1e19F),
        m_lastTemperature(-1e19F),
        m_Vsp(-1e19F),
        m_Voem(-1e19F),
        m_V3(-1e19F),
        m_V4(-1e19F),
        m_A1(-1e19F),
        m_A2(-1e19F),
        m_temperature(-1e19F)
    {
        m_instance = this;
    }
    
    ~StatusUpdateManager()
    {
    }
    
    static StatusUpdateManager* instance()
    {
        return m_instance;   
    }
    
    void setPICConnected(bool connected)
    {
        if (m_picConnected != connected)
        {
            m_picConnected = connected;
            m_flagUpdateRequired();
        }
    }
    
    void setPICState(int state, int k)
    {
        bool changed = (m_picState != state) || (m_picK != k) || !m_picStateSet;
        m_picState = state;
        m_picK = k;
        m_picStateSet = true;
        if (changed) m_flagUpdateRequired();
    }

    void setBatteryState(BatteryState state)
    {
        if (state == BatteryStateUnspecified) return; // Can't set this; initial value only
        bool changed = (m_batteryState != state);
        m_batteryState = state;
        if (changed) m_flagUpdateRequired();
    }
    
    BatteryState getBatteryState() const
    {
        return m_batteryState;
    }
    
    bool setSettings(
        uint32_t statusUpdateEvictionTime,
        uint32_t voltageFilterLengthInSamples,
        float vspUpdateThreshold,
        float vspJumpThreshold,
        float voemUpdateThreshold,
        float voemJumpThreshold,
        float v3UpdateThreshold,
        float v3JumpThreshold,
        float v4UpdateThreshold,
        float v4JumpThreshold,
        uint32_t currentFilterLengthInSamples,
        float a1UpdateThreshold,
        float a1JumpThreshold,
        float a2UpdateThreshold,
        float a2JumpThreshold,
        uint32_t temperatureFilterLengthInSamples,
        float temperatureUpdateThreshold,
        float temperatureJumpThreshold)
    {
        m_statusUpdateEvictionTime = statusUpdateEvictionTime;
        m_voltageFilterLengthInSamples = voltageFilterLengthInSamples;
        m_vspUpdateThreshold = vspUpdateThreshold;
        m_vspJumpThreshold = vspJumpThreshold;
        m_voemUpdateThreshold = voemUpdateThreshold;
        m_voemJumpThreshold = voemJumpThreshold;
        m_v3UpdateThreshold = v3UpdateThreshold;
        m_v3JumpThreshold = v3JumpThreshold;
        m_v4UpdateThreshold = v4UpdateThreshold;
        m_v4JumpThreshold = v4JumpThreshold;
        m_currentFilterLengthInSamples = currentFilterLengthInSamples;
        m_a1UpdateThreshold = a1UpdateThreshold;
        m_a1JumpThreshold = a1JumpThreshold;
        m_a2UpdateThreshold = a2UpdateThreshold;
        m_a2JumpThreshold = a2JumpThreshold;
        m_temperatureFilterLengthInSamples = temperatureFilterLengthInSamples;
        m_temperatureUpdateThreshold = temperatureUpdateThreshold;
        m_temperatureJumpThreshold = temperatureJumpThreshold;
        return true;
    }
    
    int32_t getCurrentErrorCode() const
    {
        int32_t result = 0;
        // Currently there's only one error code: 0x80000000, for NO-PIC!
        if (!m_picConnected) result |= 0x80000000;
        return result;
    }
    
    
    void service(StatusUpdateBacklog& updateBacklog)
    {
        // Only send event update after 3+ seconds so any rapid flurry of activity on subsequent samples only becomes a single update
        if (m_updateDelayTimer.read_ms() >= 3200)
        {
            // We've had a state change and 100ms have gone by, so push it!
            
            StealthPowerStatusUpdate update;
            
            update.m_atTimestamp = SystemTimeKeeper::instance().uptime();
            update.m_picState = m_picState;
            update.m_picK = m_picK;
            update.m_batteryState = (int8_t) m_batteryState;
            update.m_error = getCurrentErrorCode();
            update.m_Vsp = (int32_t)(m_Vsp / 0.01F);
            update.m_Voem = (int32_t)(m_Voem / 0.01F);
            update.m_V3 = (int32_t)(m_V3 / 0.01F);
            update.m_V4 = (int32_t)(m_V4 / 0.01F);
            update.m_A1 = (int32_t)(m_A1 / 0.1F);
            update.m_A2 = (int32_t)(m_A2 / 0.1F);
            update.m_temperature = (int32_t)(m_temperature / 0.1F);
            //update.m_message remains empty for now.

            updateBacklog.push(update);
            
            m_updateEvictionTimer.stop();
            m_updateEvictionTimer.reset();
            m_updateEvictionTimer.start();

            m_updateDelayTimer.stop();
            m_updateDelayTimer.reset();
            
            // Mark the last analog values actually sent.
            m_lastVsp = m_Vsp;
            m_lastVoem = m_Voem;
            m_lastV3 = m_V3;
            m_lastV4 = m_V4;
            m_lastA1 = m_A1;
            m_lastA2 = m_A2;
            m_lastTemperature = m_temperature;
        }
        
        if (m_statusUpdateEvictionTime > 0 &&
            m_updateEvictionTimer.read() >= m_statusUpdateEvictionTime &&
            m_updateDelayTimer.read_ms() <= 0)
        {
            // If eviction timer has expired (i.e. nothing else has prompted an update
            // in a while) AND the updateDelay fuse isn't already burning, light the fuse.
            printf("Update Eviction.\r\n");
            m_updateEvictionTimer.stop();
            m_updateEvictionTimer.reset();
            m_updateEvictionTimer.start();
            m_flagUpdateRequired();
        }
    }
    
    static void _filteredUpdate(float& value, float& newValue, float lastSentValue, float jumpThreshold, float updateThreshold, uint32_t filterLength, bool& update)
    {
        float delta = (newValue - value);
        float absDelta = fabs(delta);

        // rate = alpha = 2 / (1 + length)
        if (filterLength < 1) filterLength = 1; // filter math goes nuts if this isn't at least 1 sample long.  1 means no filter.
        const float rate = 2.0F / (1.0F + (float)filterLength);
        
        //printf("_filt %1.3f, %1.3f, %1.3f, alpha %1.6f, %1.3f, %1.3f, %d\r\n",
        //   value, newValue, lastSentValue, rate, updateThreshold, jumpThreshold, filterLength);
        
        if (isnan(value) || isnan(rate)) value = newValue;
        if (isnan(newValue)) return;
        
        // Generally do a slow filter on the data.  Battery voltages don't change very rapidly, and if they do, it's load noise, not charge.
        if (absDelta < jumpThreshold)
        {
            // If it's not grossly wrong, do a 5%/95% exponential moving average
            value = value + delta * rate;
        }
        else
        {
            // If it is grossly wrong, jump straight to new value.  This could happen if something is suddenly connected or disconnected,
            // like the shore line.
            value = newValue;
        }
        
        // Signal that the update needs to be propagated upstream iff the amount we actually changed is greater than updateThreshold
        update = update || fabs(value - lastSentValue) >= updateThreshold;
    }
    
    void setLatestValues(
        float Vsp,
        float Voem,
        float V3,
        float V4,
        float A1,
        float A2,
        float temperature)
    {
        bool sendUpdate = false;
        
        // This is a rudimentary noise filter.  Voltage/current values may jump around a bit from moment to moment,
        // an we don't want to trigger a bandwidth-eating update every time a value has some noise, so we do a
        // simple exponential moving average (always move 5% toward the latest value) that acts as a low-pass (dc-only)
        // filter.
        //
        // Most of the time we're looking at smoothly-varying discharge curves and filtering to DC is desirable.
        //
        // However, there are times when the measurements will have step functions (shore connect/disconnect, on/off,
        // battery disconnect, etc) where slow-moving filtered values are undesirable.
        //
        // To handle this, there's a threshold, typically 1.0 V or A, above which deltas (between the filtered value
        // and the most recent raw value) are immediately reflected in the filtered value.
        // i.e. If the last filtered value was 12.5V, and the latest sample is 12.6V, then the new filtered value becomes
        // 12.505V.
        // But if the last filtered value is 12.5V and the latest is 0.37V, then the new filtered value jumps immediately
        // to 0.37 because |0.37 - 12.5| > 1.0.

        // For voltages, if the new voltage is within 1.0 V of the old, then update with a sluggish low pass filter;
        // but if it's more than 1V, then snap to new value immediately.
        // Also, if the updated value is more than 0.05V different from the last value transmitted upstream, flag that
        // we need to send an update.
        //             member          new val     old val member      jump?                        send?  (out)                    should send update?
        
        _filteredUpdate(m_Vsp,         Vsp,         m_lastVsp,         m_vspJumpThreshold,          m_vspUpdateThreshold,           m_voltageFilterLengthInSamples,     sendUpdate);
        _filteredUpdate(m_Voem,        Voem,        m_lastVoem,        m_voemJumpThreshold,         m_voemUpdateThreshold,          m_voltageFilterLengthInSamples,     sendUpdate);
        _filteredUpdate(m_V3,          V3,          m_lastV3,          m_v3JumpThreshold,           m_v3UpdateThreshold,            m_voltageFilterLengthInSamples,     sendUpdate);
        _filteredUpdate(m_V4,          V4,          m_lastV4,          m_v4JumpThreshold,           m_v4UpdateThreshold,            m_voltageFilterLengthInSamples,     sendUpdate);
        _filteredUpdate(m_A1,          A1,          m_lastA1,          m_a1JumpThreshold,           m_a1UpdateThreshold,            m_currentFilterLengthInSamples,     sendUpdate);
        _filteredUpdate(m_A2,          A2,          m_lastA2,          m_a2JumpThreshold,           m_a2UpdateThreshold,            m_currentFilterLengthInSamples,     sendUpdate);
        _filteredUpdate(m_temperature, temperature, m_lastTemperature, m_temperatureJumpThreshold,  m_temperatureUpdateThreshold,   m_temperatureFilterLengthInSamples, sendUpdate);
        if (sendUpdate)
        {
            printf("StatusUpdateManager - Analog Values Changed\r\n");
            if (m_updateDelayTimer.read_ms() <= 0) m_flagUpdateRequired();
        }
    }
    
    
    float getFilteredA1() const
    {
        return m_A1;   
    }
    
private:

    void m_flagUpdateRequired()
    {
        // Start the timer if it isn't already
        if (m_updateDelayTimer.read_ms() <= 0)
        {
            m_updateDelayTimer.reset();
            m_updateDelayTimer.start();
        }
    }

    bool m_picConnected;
    bool m_picStateSet;
    int m_picState;
    int m_picK;
    
    BatteryState m_batteryState;
    
    float m_lastVsp;
    float m_lastVoem;
    float m_lastV3;
    float m_lastV4;
    float m_lastA1;
    float m_lastA2;
    float m_lastTemperature;
    
    float m_Vsp;
    float m_Voem;
    float m_V3;
    float m_V4;
    float m_A1;
    float m_A2;
    float m_temperature;
    
    
    
    uint32_t m_statusUpdateEvictionTime;
    uint32_t m_voltageFilterLengthInSamples;
    float m_vspUpdateThreshold;
    float m_vspJumpThreshold;
    float m_voemUpdateThreshold;
    float m_voemJumpThreshold;
    float m_v3UpdateThreshold;
    float m_v3JumpThreshold;
    float m_v4UpdateThreshold;
    float m_v4JumpThreshold;
    uint32_t m_currentFilterLengthInSamples;
    float m_a1UpdateThreshold;
    float m_a1JumpThreshold;
    float m_a2UpdateThreshold;
    float m_a2JumpThreshold;
    uint32_t m_temperatureFilterLengthInSamples;
    float m_temperatureUpdateThreshold;
    float m_temperatureJumpThreshold;

    Timer m_updateDelayTimer;
    
    SecondTimer m_updateEvictionTimer;

    static StatusUpdateManager* m_instance;
};



#endif

