
#ifndef _RadioManager_h_
#define _RadioManager_h_

#include "mtsas.h"

#include <time.h>

// static const char RadioAPN[] = "m2m.com.attz"; // Works with developer SIMs from ATT
static const char RadioAPN[] = "10569.mcs"; // We hope this works with new SIMs

class RadioManager
{
public:
    RadioManager() :
        m_requestedConnection(false),
        m_hasEverConnected(false),
        m_io(NULL),
        m_cellular(NULL),
        m_radioEnablePin(PC_13)
    {
    }
    
    ~RadioManager()
    {
        if (m_cellular)
        {
            m_cellular->disconnect();
            delete m_cellular;
            m_cellular = NULL;
        }
        if (m_io)
        {
            delete m_io;
            m_io = NULL;
        }
    }
    
    void resumeWatchdog()
    {
        //printf("resumeWatchdog() already %1.3f\r\n", (float)m_radioRebootWatchdog.read());
        if (m_radioRebootWatchdog.read() > 0) return; // Already started, don't reset it
        
        // Need to start it
        m_radioRebootWatchdog.reset();
        m_radioRebootWatchdog.start();
    }
    
    void feedWatchdog()
    {
        //printf("feedWatchdog()\r\n");
        // Reset it but set to 0
        m_radioRebootWatchdog.stop();
        m_radioRebootWatchdog.reset();
        m_radioRebootWatchdog.start();
    }
    
    void stopWatchdog()
    {
        //printf("stopWatchdog()\r\n");
        m_radioRebootWatchdog.reset();
        m_radioRebootWatchdog.stop();
    }
    
    bool serviceWatchdog()
    {
        // Check if radio wdt has elapsed and reboot the radio if so
        if (m_radioRebootWatchdog.read() > 60*30) // If any radio operation takes longer than this # of seconds, it's time to reboot the radio.
        {
            printf("Radio watchdog tripped!  Rebooting radio.\r\n");
            return initialize();
        }

        return true; // true means OK
    }
    
    int watchdogValue()
    {
        return (int)m_radioRebootWatchdog.read();
    }

    bool initialize()
    {
        // Always start/feed the radio watchdog when initializing; we mustn't wait until
        // contact has been made to start wondering if contact is going to be made.
        feedWatchdog();
        
        // Hard-reboot the modem for starters.
        m_radioEnablePin = 0;
        wait_ms(100); // must be <<< than 1 second for hard reset.
        m_radioEnablePin = 1;
        // Wait for radio to reboot
        wait_ms(8500);
        printf("Radio rebooted.\r\n");
        
        if (!m_io)
        {
            m_io = new mts::MTSSerialFlowControl(RADIO_TX, RADIO_RX, RADIO_RTS, RADIO_CTS);
        }
        if (!m_io) return false;
            
        // Radio baud rate is 115200
        m_io->baud(115200);
        
        // Send a ctrl+z and a \n sequence to escape out of any half-entered commands
        char escapeSequence[4] = {0};
        snprintf(escapeSequence,4,"%c%c%c",0x1a,0x1b,'\n');
        m_io->write(escapeSequence, 3);
        
        // Build cellular object
        if (!m_cellular)
        {
            m_cellular = mts::CellularFactory::create(m_io);
        }
        if (!m_cellular)
        {
            delete m_io;
            m_io = NULL; 
            return false;
        }        
        
        std::string result;
 
        // ConnId 1, cid 1, 1500 byte packets, 1000 seconds to socket auto-close, 15 seconds to connect timeout, 50 milliseconds auto send-flush timer.
        result = m_cellular->sendCommand("AT#SCFG=1,1,1500,1000,15,50", 5000);
        if (!result.find("\nOK\n"))
        {
            printf("Failed to configure radio (at#SCFG)\r\n");
            return false;
        }
        
        // Configure better socket connection timeout than default
        result = m_cellular->sendCommand("at#SKTCT=150", 5000); // 100 x 0.1s = 10sec
        if (!result.find("\nOK\n"))
        {
            printf("Failed to configure radio (at#SKTCT)\r\n");
            return false;
        }
        
        // Configure the socket-inactivity timeout
        result = m_cellular->sendCommand("at#SKTTO=1000", 8000); // 800 sec (10 min) inactivity timeout for socket--basically keep the socket open.
        if (!result.find("\nOK\n"))
        {
            printf("Failed to configure radio (at#SKTTO=120)\r\n");
            return false;
        }
        
        /// Configure APN
        printf("Configuring radio APN '%s'\r\n", RadioAPN);
        if (m_cellular->setApn(RadioAPN) != MTS_SUCCESS)
        {
            printf("Failed to set Radio APN to '%s'\r\n", RadioAPN);
            delete m_cellular;
            m_cellular = NULL;
            delete m_io;
            m_io = NULL;
            return false;
        }

        return true;
    }
    
    bool hasEverConnected() const
    {
        return m_hasEverConnected;   
    }
    
    bool connect()
    {
        if (!m_cellular) return false;
        m_requestedConnection = true;
        if (m_cellular->connect())
        {
            m_hasEverConnected = true;
            return true;
        }
        return false;
    }

    void disconnect()
    {
        if (!m_cellular) return;
        m_requestedConnection = false;
        m_cellular->disconnect();
    }
    
    bool isConnected()
    {
        if (!m_io) return false;
        if (!m_cellular) return false;
        return m_cellular->isConnected();    
    }
    
    string sendCommand(const std::string& command, unsigned int timeoutMillis, char esc = CR)
    {
        if (!m_cellular) return std::string("NO RADIO\nERROR\n");
        return m_cellular->sendCommand(command, timeoutMillis, esc);
    }
    
    string sendCommandWithBinaryPayload(const std::string& command, const char* payload, size_t payloadSize, unsigned int timeoutMillis, char esc = CR)
    {
        if (!m_cellular) return std::string("NO RADIO\nERROR\n");
        return m_cellular->sendCommandWithBinaryPayload(command, payload, payloadSize, timeoutMillis, esc);
    }

    int64_t getTime()
    {
        if (!m_cellular) return 0;
        
        string response = m_cellular->sendCommand("AT+CCLK?", 500);
        size_t found = response.find("CCLK: \"");
        if (found == string::npos) return 0;
        const char* timeSubstr = &response.c_str()[found+7];
        int year, month, dom, hour, min, sec, tzoffset;
        char plusMinusChar = '\0';
        if (8 != sscanf(timeSubstr, "%d/%d/%d,%d:%d:%d%c%d", 
                        &year,
                        &month,
                        &dom,
                        &hour,
                        &min,
                        &sec,
                        &plusMinusChar,
                        &tzoffset)) return -1;
        std::tm time;
        memset(&time, 0, sizeof(time));
        time.tm_sec = sec;
        time.tm_min = min;
        time.tm_hour = hour;
        time.tm_mday = dom;
        time.tm_mon = month - 1;
        time.tm_year = year + 2000 - 1900;
        time.tm_isdst = 0;
        time.tm_yday = 0;
        int t = mktime(&time);
        // Magic number 15 below: modem reports the timezone in 15-minute increments, so
        // we scale that up to seconds.
        t -= 15*60*tzoffset * (plusMinusChar == '-' ? -1 : 1);
        return (int64_t)t;
    }

private:
    bool m_requestedConnection;
    bool m_hasEverConnected;
    mts::MTSSerialFlowControl* m_io;
    mts::Cellular* m_cellular;
    DigitalOut m_radioEnablePin;
    
    Timer m_radioRebootWatchdog;
};


#endif //_RadioManager_h_
