
#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include "PICInterface.h"


// Internal PIC Interface Helper Functions
uint8_t _PICExtractChecksum(const char* data)
{
   uint32_t result = 0xffffffff;
   if (1 != sscanf(data, "%02X", &result))
   {
      return 0; // which could itself be a checksum value...
   }
   //printf("Extracted checksum 0x%X from '%02s'\r\n", result, data);
   return result;
}

uint8_t _PICCalculateChecksum(const char* data, size_t checksumOffset)
{
   uint8_t checksum = 0;
   for (size_t i = 0; i < checksumOffset; ++i) checksum += (uint8_t)data[i];
   checksum = (checksum ^ 0xff);
   //printf("calculated cs = %02x\r\n", checksum);
   return checksum;
}

void _PICAppendChecksum(char* data, size_t checksumOffset)
{
   char checksumString[3] = { 0 };
   snprintf(checksumString, sizeof(checksumString), "%02X", (uint8_t)_PICCalculateChecksum(data, checksumOffset));
   data[checksumOffset] = checksumString[0];
   data[checksumOffset+1] = checksumString[1];
}

bool _PICValidateChecksum(const char* data, size_t checksumOffset)
{
   return _PICExtractChecksum(&data[checksumOffset]) == _PICCalculateChecksum(data, checksumOffset);
}

PIC_result_t _PICTransmitPacket(char* data, size_t checksumOffset)
{
   // data must be allocated 3 bytes larger than checksumOffset, since the checksum must be
   // written followed by a '\n'
   _PICAppendChecksum(data, checksumOffset);
   data[checksumOffset+2] = '\n';
   return PICTransmitCallback(data, checksumOffset+3);
}

PIC_result_t PICTransmitRequestConfigPacket()
{
   char buffer[] = "G;\0\0\n";
   return _PICTransmitPacket(buffer, sizeof(buffer)-3);
}


// Client calls this whenever a line/packet is received
PIC_result_t PICConsumePacket(const char* packet)
{
   if (!packet) return PICResultPacketParameterError;
   size_t length = strlen(packet);
   //printf("Consuming PIC packet '%s'\r\n", packet);
   if (length < 3)
   {
       printf("packet too short.\r\n");
       return PICResultPacketUnknownError;
   }

   // Verify checksum first
   if (!_PICValidateChecksum(packet, length-3))
   {
      printf("Packet fails checksum.\r\n");
      return PICResultPacketChecksumError;
   }

   // Respond to packet
   switch(packet[0])
   {
      case 'I':
      case 'E':
      {
         char type = '\0';
         int state = -1;
         int Vsp = 0;
         int Voem = 0;
         int V3 = 0;
         int V4 = 0;
         int A1 = 0;
         int A2 = 0;
         int K = 0;
         if (9 != sscanf(packet, "%c;%d;%d;%d;%d;%d;%d;%d;%d;",
            &type,
            &state,
            &Vsp,
            &Voem,
            &V3,
            &V4,
            &A1,
            &A2,
            &K))
         {
            return PICResultPacketFormatError;
         }
         return PICParametricDataCallback(type, state, Vsp, Voem, V3, V4, A1, A2, K);
      }
      case 'R':
      {
         int version = -1;
         int Vcutoff = -1;
         int Vcharge = -1;
         int Tdisch = -1;
         int ProgSel = -1;
         if (5 != sscanf(packet, "R;%d;%d;%d;%d;%d;",
            &version,
            &Vcutoff,
            &Vcharge,
            &Tdisch,
            &ProgSel))
         {
            return PICResultPacketFormatError;
         }
         return PICConfigurationDataCallback(version, Vcutoff, Vcharge, Tdisch, ProgSel);
      }
      default:
      {
         return PICResultPacketUnknownError;
      }
   }
}


#if ENABLE_PIC_TEST_CASES

// Test Stuff:

int TestPIC()
{
   int result = 0;

   result = PICConsumePacket("E;5;4;30;021;0;19;205;192;84\n");
   printf("result = %d\r\n", result);

   result = PICConsumePacket("I;5;4;30;021;0;19;205;192;88\n");
   printf("result = %d\r\n", result);

   result = PICConsumePacket("nonsense!\n");
   printf("result = %d\r\n", result);

   result = PICConsumePacket("R;102;250;11;19;202;3e\n");
   printf("result = %d\r\n", result);

   result = PICConsumePacket("R;102;250;11;19;202;3f\n");
   printf("result = %d\r\n", result);

   return result;
}


PIC_result_t PICTransmitCallback(const char* bytes, size_t count)
{
   return 0;
}

PIC_result_t PICConnectionCallback(int revision, int major, int minor)
{
   printf("Got 'C', %d.%d.%d\r\n", revision, major, minor);
   return 0;
}

PIC_result_t PICParametricDataCallback(
   char type,
   int state,
   int Vsp,
   int Voem,
   int V3,
   int V4,
   int A1,
   int A2,
   int K)
{
   printf("Got '%c', state %d, %d %d %d %d %d %d %d\r\n", type, state, Vsp, Voem, V3, V4, A1, A2, K);
   return 0;
}

PIC_result_t PICNackCallback()
{
   return 0;
}

PIC_result_t PICAckCallback()
{
   return 0;
}

PIC_result_t PICConfigurationDataCallback(int version, int Vcutoff, int Vcharge, int Tdisch, int ProgSel)
{
   printf("Got config, version %d, cutoff %d charge %d Tdisch %d Prog %d\r\n", version, Vcutoff, Vcharge, Tdisch, ProgSel );
   return 0;
}

#endif // ENABLE_PIC_TEST_CASES
