#include "mbed.h"

#include "SystemTimeKeeper.h"
#include "RadioManager.h"
#include "StealthPowerSampler.h"
#include "Logo.h"
#include "SampleBacklog.h"
#include "StatusUpdateBacklog.h"
#include "PICManager.h"
#include "RadarSocketManager.h"
#include "SecondTimer.h"
#include "SystemSerialNumber.h"
#include "StatusUpdateManager.h"
#include "SettingsManager.h"
#include "WatchdogManager.h"

#define HOSTNAME_LENGTH 128

// This line controls the regulator's battery charger.
// BC_NCE = 0 enables the battery charger
// BC_NCE = 1 disables the battery charger
DigitalOut bc_nce(PB_2);

// Declare all the firmware subcomponents/objects globally here
SpiFlash25 flash(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS1);
SettingsManager settingsManager(flash);
Ticker picTicker;
RadioManager radioManager;
SystemTimeKeeper systemTime;
Serial dbg(USBTX, USBRX);
Ticker main1HzTicker;
StealthPowerSampler intervalSampler;
StealthPowerSampler eventSampler;
StatusUpdateManager updateManager;
SampleBacklog sampleBacklog;
StatusUpdateBacklog statusUpdateBacklog;
RadarSocketManager radarSocketManager(radioManager);
PICManager picManager(&intervalSampler, &eventSampler, &sampleBacklog);
SecondTimer timeSinceSettingsCheck;


void picUpdateServiceCallback()
{
    picManager.servicePIC();
    updateManager.service(statusUpdateBacklog);
}


void main1HzCallback()
{
    // This gets called exactly once a second and is where we take data samples.
    // NOTE: We are in a Ticker callback (ISR) Context, so BE CAREFUL!
    systemTime.advance();

    int64_t now = systemTime.uptime();
    
    // Make sure all samplers know the range of time they've measured even if
    // we don't have data for them
    intervalSampler.sampleTime(now);
    eventSampler.sampleTime(now);
    
    if (intervalSampler.intervalIsComplete())
    {
        printf("Interval sample rollover.\r\n");
        
        if (intervalSampler.isWorthSending())
        {
            StealthPowerSample sample;
            intervalSampler.populateSample(sample);
            sample.m_trigger = StealthPowerSample::TriggerInterval;
            sampleBacklog.push(sample);
        }
        intervalSampler.reset(now);
    }
}


const int64_t MinimumValidTime = 1459007487; // 3/26/2016.

void synchronizeTimeToCellular()
{
    // Query the radio for the exact wall time
    int64_t currentRadioTime = radioManager.getTime();

    // If we got a valid time (after 3/26/2016, and not -1 (error))
    if (currentRadioTime >= MinimumValidTime && currentRadioTime != -1) {
        __disable_irq();
        int delta = abs(systemTime.walltime() - currentRadioTime);
        // Because the phase of the update Ticker is unaligned with this update,
        // leave 1s of slack between the two and only update the system
        // time if they get more than 1s apart.
        if (delta > 1) {
            printf("Resynchronized to wall time = %lld\r\n", currentRadioTime);
            systemTime.setWalltime(currentRadioTime);
        }
        __enable_irq();
    }
}

bool flushQueuesToAPI()
{
    // Push all the explicit status-update events
    int sentItemsCount = 0;
    radar_result_t result = RADAR_OK;
    StealthPowerStatusUpdate update;
    __disable_irq();
    while (statusUpdateBacklog.peek(update) && update.m_atTimestamp >= MinimumValidTime) {
        __enable_irq();
        result = Radar_PublishStatus(
                     (time_t)update.m_atTimestamp,
                     update.m_picState,
                     update.m_picK,
                     update.m_batteryState,
                     update.m_error,
                     update.m_message.c_str(),
                     update.m_Vsp,
                     update.m_Voem,
                     update.m_V3,
                     update.m_V4,
                     update.m_A1,
                     update.m_A2,
                     update.m_temperature);

        __disable_irq();
        if (result == RADAR_OK) {
            // Only once it's transmitted do we remove it from queue.
            statusUpdateBacklog.shift_if(update);
            ++sentItemsCount;
        }
        if (result != RADAR_OK) {
            __enable_irq();
            return false;
        }
    }
    __enable_irq();
    if (sentItemsCount > 0) printf("Transmitted %d event(s).\r\n", sentItemsCount);

    // And the interval/trigger samples:
    sentItemsCount = 0;
    StealthPowerSample sample;
    __disable_irq();
    while (sampleBacklog.peek(sample) && sample.m_beginTimestamp >= MinimumValidTime) {
        __enable_irq();
        
        int triggerType = RADAR_TRIGGER_INTERVAL;
        if (sample.m_trigger == StealthPowerSample::TriggerDischarge) triggerType = RADAR_TRIGGER_BATTERY_DISCHARGE;
        if (sample.m_trigger == StealthPowerSample::TriggerCharge) triggerType = RADAR_TRIGGER_BATTERY_CHARGE;
        
        printf("SENDING SAMPLE, sample.m_Vsp is %f\r\n", sample.m_Vsp);
        result = Radar_PublishSample(
                     triggerType,
                     (time_t)sample.m_beginTimestamp,
                     (time_t)sample.m_endTimestamp,
                     sample.m_Vsp * 1000.0F,
                     sample.m_Vsp_min * 1000.0F,
                     sample.m_Vsp_max * 1000.0F,
                     sample.m_Voem * 1000.0F,
                     sample.m_Voem_min * 1000.0F,
                     sample.m_Voem_max * 1000.0F,
                     sample.m_V3 * 1000.0F,
                     sample.m_V3_min * 1000.0F,
                     sample.m_V3_max * 1000.0F,
                     sample.m_V4 * 1000.0F,
                     sample.m_V4_min * 1000.0F,
                     sample.m_V4_max * 1000.0F,
                     sample.m_A1 * 1000.0F,
                     sample.m_A1_min * 1000.0F,
                     sample.m_A1_max * 1000.0F,
                     sample.m_A2 * 1000.0F,
                     sample.m_A2_min * 1000.0F,
                     sample.m_A2_max * 1000.0F,
                     sample.m_temperatureAvg * 1000.0F,
                     sample.m_temperatureMin * 1000.0F,
                     sample.m_temperatureMax * 1000.0F,
                     sample.m_sampleCount,
                     sample.m_dischargeCount,
                     sample.m_chargeCount);
        __disable_irq();
        if (result == RADAR_OK) {
            // Only once it's transmitted do we remove it from queue.
            sampleBacklog.shift_if(sample);
            ++sentItemsCount;
        }
        if (result != RADAR_OK) {
            __enable_irq();
            return false;
        }
    }
    __enable_irq();
    if (sentItemsCount > 0) printf("Transmitted %d sample(s).\r\n", sentItemsCount);

    return true;
}


bool applySettingsToSubsystems()
{
    // Copy the settings from the settings manager into all the subystems that
    // should be affected by them.  This should be invoked after every 
    // GetTelemetrySettings() call, or whenever any settings are changed or
    // loaded (e.g. at boot).
    bool success = true;

    success = success && radarSocketManager.setServers(
        settingsManager.getHost1(),
        settingsManager.getPort1(),
        settingsManager.getHost2(),
        settingsManager.getPort2());
    
    success = success && intervalSampler.setSampleInterval(
        settingsManager.getSampleIntervalInSeconds());

    success = success && picManager.setThresholds(
        settingsManager.getDischargeThreshold(),
        settingsManager.getChargeThreshold(),
        settingsManager.getCurrentHysteresis(),
        settingsManager.getUseFilteredA1ForBatteryState(),
        settingsManager.getMinimumEventSampleDuration());

    success = success && updateManager.setSettings(
        settingsManager.getStatusUpdateEvictionTime(),
        settingsManager.getVoltageFilterLengthInSamples(),
        settingsManager.getVspUpdateThreshold(),
        settingsManager.getVspJumpThreshold(),
        settingsManager.getVoemUpdateThreshold(),
        settingsManager.getVoemJumpThreshold(),
        settingsManager.getV3UpdateThreshold(),
        settingsManager.getV3JumpThreshold(),
        settingsManager.getV4UpdateThreshold(),
        settingsManager.getV4JumpThreshold(),
        settingsManager.getCurrentFilterLengthInSamples(),
        settingsManager.getA1UpdateThreshold(),
        settingsManager.getA1JumpThreshold(),
        settingsManager.getA2UpdateThreshold(),
        settingsManager.getA2JumpThreshold(),
        settingsManager.getTemperatureFilterLengthInSamples(),
        settingsManager.getTemperatureUpdateThreshold(),
        settingsManager.getTemperatureJumpThreshold());

    // No need to do anything for settingsCheckInterval--settingsManager
    // is the final recipient of that information already.

    return success;
}


bool updateSettingsFromServer()
{
    int32_t dischargeThreshold = 0;
    int32_t chargeThreshold = 0;
    int32_t currentHysteresis = 0;
    int32_t regularSampleInterval = -1;
    int32_t settingsCheckInterval = -1;
    hostname_t primaryHost = "";
    uint32_t primaryPort = 0;
    hostname_t secondaryHost = "";
    uint32_t secondaryPort = 0;
    uint32_t statusUpdateEvictionTime = 0;
    uint32_t voltageFilterLengthInSamples = 0;
    int32_t vspUpdateThreshold = 0;
    int32_t vspJumpThreshold = 0;
    int32_t voemUpdateThreshold = 0;
    int32_t voemJumpThreshold = 0;
    int32_t v3UpdateThreshold = 0;
    int32_t v3JumpThreshold = 0;
    int32_t v4UpdateThreshold = 0;
    int32_t v4JumpThreshold = 0;
    uint32_t currentFilterLengthInSamples = 0;
    int32_t a1UpdateThreshold = 0;
    int32_t a1JumpThreshold = 0;
    int32_t a2UpdateThreshold = 0;
    int32_t a2JumpThreshold = 0;
    uint32_t temperatureFilterLengthInSamples = 0;
    int32_t temperatureUpdateThreshold = 0;
    int32_t temperatureJumpThreshold = 0;
    int32_t useFilteredA1ForBatteryState = 0;
    int32_t minimumEventSampleDuration = 0;
    

    radar_result_t result = Radar_GetTelemetrySettings(
        &dischargeThreshold,
        &chargeThreshold,
        &currentHysteresis,
        &regularSampleInterval,
        &settingsCheckInterval,
        primaryHost,
        &primaryPort,
        secondaryHost,
        &secondaryPort,
        &statusUpdateEvictionTime,
        &voltageFilterLengthInSamples,
        &vspUpdateThreshold,
        &vspJumpThreshold,
        &voemUpdateThreshold,
        &voemJumpThreshold,
        &v3UpdateThreshold,
        &v3JumpThreshold,
        &v4UpdateThreshold,
        &v4JumpThreshold,
        &currentFilterLengthInSamples,
        &a1UpdateThreshold,
        &a1JumpThreshold,
        &a2UpdateThreshold,
        &a2JumpThreshold,
        &temperatureFilterLengthInSamples,
        &temperatureUpdateThreshold,
        &temperatureJumpThreshold,
        &useFilteredA1ForBatteryState,
        &minimumEventSampleDuration);

    printf("Radar_GetTelemetrySettings result = %d\r\n", result); // TODO: Cleanup
    if (result == RADAR_OK)
    {
        printf("Retrieved latest settings.\r\n");
        settingsManager.setAll(
            primaryHost,
            primaryPort,
            secondaryHost,
            secondaryPort,
            dischargeThreshold,
            chargeThreshold,
            currentHysteresis,
            regularSampleInterval,
            settingsCheckInterval,
            statusUpdateEvictionTime,
            voltageFilterLengthInSamples,
            vspUpdateThreshold,
            vspJumpThreshold,
            voemUpdateThreshold,
            voemJumpThreshold,
            v3UpdateThreshold,
            v3JumpThreshold,
            v4UpdateThreshold,
            v4JumpThreshold,
            currentFilterLengthInSamples,
            a1UpdateThreshold,
            a1JumpThreshold,
            a2UpdateThreshold,
            a2JumpThreshold,
            temperatureFilterLengthInSamples,
            temperatureUpdateThreshold,
            temperatureJumpThreshold,
            useFilteredA1ForBatteryState,
            minimumEventSampleDuration);

        printf("Latest Settings:\r\n");
        settingsManager.printSettings();

        return applySettingsToSubsystems();
    }
    else
    {
        printf("FAILED to retrieve settings!\r\n");
    }
    return false;
}



int main()
{
    // Initial boot delay fixes issue with occasional loss of first lines of
    // console output.
    wait_ms(1000);
    
    // All the demos do this to disable the (modem) battery charger.
    bc_nce = 1;

    // Radio state management
    bool isRadioConnected = false;
    bool isCommunicationConnected = false;
    bool haveReasonToContactServer = false;
    bool everGotSettingsFromServer = false;

    // For debugging:
    //mts::MTSLog::setLogLevel(mts::MTSLog::TRACE_LEVEL);
    
    // For normal use:
    mts::MTSLog::setLogLevel(mts::MTSLog::WARNING_LEVEL);

    //*********************************
    dbg.baud(115200);
    printf("Initializing Serial Console - OK\r\n"); fflush(stdout);
    printStealthPowerLogo();
    
    //*********************************
    printf("Initializing System Watchdog Timer - OK\r\n"); fflush(stdout);
    WatchdogManager::Initialize();

    //*********************************
    printf("Initializing Serial Number: "); fflush(stdout);
    printf("%s", getSystemSerialNumber());
    printf(" - OK\r\n");

    //*********************************
    printf("Initializing Settings: "); fflush(stdout);
    settingsManager.initializeSettings();
    printf("- OK\r\n");
    settingsManager.printSettings();
    
    //*********************************
    printf("Applying Settings: "); fflush(stdout);
    if (applySettingsToSubsystems())
    {
        printf("- OK\r\n");
    }
    else
    {
        printf("- ERROR\r\n");   
    }

    //*********************************
    printf("Initializing Real Time Clock"); fflush(stdout);
    systemTime.reset();
    printf(" - OK\r\n");
    
    //*********************************
    printf("Initializing Sample Engine"); fflush(stdout);
    intervalSampler.reset(systemTime.uptime());
    eventSampler.reset(systemTime.uptime());
    main1HzTicker.attach(main1HzCallback, 1.0);
    printf(" - OK\r\n");

    //*********************************
    printf("Initializing PIC Communications"); fflush(stdout);
    
    if (picManager.initialize())
        printf(" - OK\r\n");
    else
        printf(" - FAILED (will keep trying)\r\n");
    // Service the PIC uart rapidly--most of the time there won't be anything to do.
    picTicker.attach(picUpdateServiceCallback, 0.1);
   
    //*********************************
    // Radio Init is last, because it may block if no signal; don't want to hold off
    // sampling engine on a blocking radio call
    printf("Initializing Radio"); fflush(stdout);
    //wait_us(50);
    radioManager.initialize();
    radioManager.connect();
    if (radioManager.hasEverConnected())
        printf(" - OK\r\n");
    else
        printf(" - FAILED (but will keep trying)\r\n");

    //*********************************
    printf("Entering Main Loop\r\n");
    while(1)
    {
        // ---------------------------------------------------------------------------------------------
        // (0) Feed the pseudo-hardware watchdog ticker loop
        WatchdogManager::Feed();
        
        // ---------------------------------------------------------------------------------------------
        // (1) Make sure the radio is connected if it possibly can be.  (Not necessarily the socket.)
        isRadioConnected = false;

        // If connected, ensure time-sync to cellular network wall time.
        if (radioManager.isConnected()) {
            isRadioConnected = true;
            synchronizeTimeToCellular();
        }
        else
        {
            printf("Connecting radio\r\n");
            isRadioConnected = radioManager.connect();
            if (isRadioConnected)
            {
                printf("Radio connected.\r\n");
            }
            else
            {
                printf("Radio failed to connect.\r\n");
            }
        }

        // ---------------------------------------------------------------------------------------------
        // (2) Then, make sure that if we have a valid wall-time, all samples that were timestamped
        //     with system uptime are translated to wall-time.
        if (systemTime.hasWalltime())
        {
            __disable_irq();
            sampleBacklog.retroactivelyTimestampSamples(systemTime.uptime(), systemTime.walltimeDelta());
            statusUpdateBacklog.retroactivelyTimestampSamples(systemTime.uptime(), systemTime.walltimeDelta());
            __enable_irq();
        }

        // ---------------------------------------------------------------------------------------------
        // (3) Determine if there's a pressing need to contact the server right now--if not, don't waste
        //     bandwidth connecting and hello'ing a socket we don't have anything to say on.
        haveReasonToContactServer = (sampleBacklog.backlog() > 0) ||
                                   (statusUpdateBacklog.backlog() > 0) ||
                                   (timeSinceSettingsCheck.read() >= settingsManager.getSettingsUpdateIntervalInSeconds()) ||
                                   (!everGotSettingsFromServer);

        // ---------------------------------------------------------------------------------------------
        // (4) Make sure we're connected via TCP to the server if we can be, and need to be, assuming the
        //     radio is successfully connected.
        if (haveReasonToContactServer)
        {
            // First, if there's any reason to try connecting, that's a reason to start the radio watchdog.
            // This basically means "I better have some luck with the radio/server within X amount of time
            // or we'll hard reboot the radio."
            radioManager.resumeWatchdog();
            
            isCommunicationConnected = false;
            if (isRadioConnected)
            {
                if (!radarSocketManager.isConnected())
                {
                    printf("Connecting Stealth Radar API TCP socket\r\n");
                    if (radarSocketManager.connect()) { //(re-)connect to configured server
                        printf("Connected socket; sending Hello()\r\n");
                        radar_result_t result = Radar_Hello(
                                                    0x00010000,  // build 01.0.0
                                                    (time_t)systemTime.uptime(), // useful for detecting reboots
                                                    (time_t)systemTime.walltime(),
                                                    getSystemSerialNumber());
                        if (result != RADAR_OK)
                        {
                            printf("Error from Radar_Hello, shutting down socket.\r\n");
                            radarSocketManager.disconnect();
                            isCommunicationConnected = false;
                        }
                        else
                        {
                            printf("Connected Stealth Radar API TCP socket.\r\n");
                            isCommunicationConnected = true;
                        }
                    }
                    else
                    {
                        printf("Failed to connect Stealth Radar API TCP socket!\r\n");
                        radarSocketManager.disconnect();
                        isCommunicationConnected = false; // just being explicit about this
                    }
                }
                else
                {
                    isCommunicationConnected = true;
                }
            }
            else
            {
                printf("Radar socket not connected because radio not connected.\r\n");
                // Radio not connected
                radarSocketManager.disconnect();
                isCommunicationConnected = false;
            }
        }

        // ---------------------------------------------------------------------------------------------
        // (5) If the communication socket is established, dump as much queued data ASAP to the
        //     server as possible
        if (isCommunicationConnected)
        {
            if (!flushQueuesToAPI())
            {
                printf("Problem transmitting sample/status updates to the server; destroying Radar API TCP Socket.\r\n");
                radarSocketManager.disconnect();
                isCommunicationConnected = false;
            }
            
            // Always get settings at the beginning of a connection
            if (timeSinceSettingsCheck.read() >= settingsManager.getSettingsUpdateIntervalInSeconds() || !everGotSettingsFromServer)
            {
                if (updateSettingsFromServer())
                {
                    // Reset the timer that watches how long it's been since we've asked for settings
                    timeSinceSettingsCheck.stop();
                    timeSinceSettingsCheck.reset();
                    timeSinceSettingsCheck.start();
                    everGotSettingsFromServer = true;
                }
                else
                {
                    printf("Problem getting settings from the server.\r\n");
                    radarSocketManager.disconnect();
                    isCommunicationConnected = false;
                }
            }
        }

        // ---------------------------------------------------------------------------------------------
        // (6) If we are *not* connected to the server due to radio or tcp problems, *and* there
        //     is any backlog of samples not sent yet, back them up to flash in case of power loss.
        if (!isCommunicationConnected)
        {
            /*if (sampleBacklog.backlog() > 0)
            {
                printf("Backing up queued sample data to flash.\r\n");
            }
            if (statusUpdateBacklog.backlog() > 0)
            {
                printf("Backing up queued status-update events to flash.\r\n");
            }*/           
            // Backup any wall-timestamped samples to flash since there's no connection right now and
            // power could go away at any moment.
            
            // Note -- we dropped this feature due to the complexity of dealing with 64KB sectors
            // on the SPI flash, but this is where you'd ideally keep everything backed up.
        }

        // ---------------------------------------------------------------------------------------------
        // (7) Make sure the latest setting are stored in the flash, in duplicate
        if (!settingsManager.serviceSettingsInFlash())
        {
            printf("Error updating settings in flash!\r\n");
        }

        // ---------------------------------------------------------------------------------------------
        // (8) Finally, spit out debug state and throttle the main loop rate (too fast seems to aggravate
        //     the modem, which is already touchy...)
        printf("MAIN: t = %d, radio = %d, socket = %d, samplebacklog = %d, updatebacklog = %d, radio wdt = %d\r\n",
            (int)SystemTimeKeeper::instance().walltime(),
            isRadioConnected,
            isCommunicationConnected,
            sampleBacklog.backlog(),
            statusUpdateBacklog.backlog(),
            radioManager.watchdogValue());
  
        // ---------------------------------------------------------------------------------------------
        // (7) Radio watchdog -- if we haven't had success using the radio in
        //     more than a couple hours, reboot the radio; it may be hosed.
        if (!radioManager.serviceWatchdog())
        {
            printf("Error handling radio watchdog!\r\n");
        }

        wait(2.0);
    }
}
