
#ifndef _StealthPowerSampler_h_
#define _StealthPowerSampler_h_

#include "StealthPowerSample.h"
#include <math.h>

class Integrator
{
public:
    Integrator() :
        m_count(0),
        m_integral(0),
        m_minimum(std::numeric_limits<double>::max()),
        m_maximum(-std::numeric_limits<double>::max())
    {
    }

    void reset()
    {
        m_count = 0;
        m_integral = 0;
        m_minimum = std::numeric_limits<double>::max();
        m_maximum = -std::numeric_limits<double>::max();
    }
        
    void integrate(double value)
    {
        ++m_count;
        m_integral += value;
        m_minimum = std::min(m_minimum, value);
        m_maximum = std::max(m_maximum, value);
    }
    
    double average() const
    {
        return m_integral / (double)m_count;
    }
    
    double minimum() const
    {
        return m_minimum;
    }
    
    double maximum() const
    {
        return m_maximum;
    }
    
private:    
    int32_t m_count;
    double m_integral;
    double m_minimum;
    double m_maximum;
};



class StealthPowerSampler
{
public:
    StealthPowerSampler() :
        m_beginTimestamp(0),
        m_endTimestamp(0),
        m_sampleCount(0),
        m_dischargeCount(0),
        m_chargeCount(0),
        m_dischargeCurrentThreshold(1.0F),
        m_chargeCurrentThreshold(1.0F),
        m_sampleInterval(600)
    {
    }
    
    void reset(int64_t time)
    {
        m_beginTimestamp = time;
        m_endTimestamp = 0;
        m_Vsp.reset();
        m_Voem.reset();
        m_V3.reset();
        m_V4.reset();
        m_A1.reset();
        m_A1.reset();
        m_temperature.reset();
        m_sampleCount = 0;
        m_dischargeCount = 0;
        m_chargeCount = 0;
    }
        
    void sampleTime(int64_t time)
    {
        m_endTimestamp = time;
    }
        
    void sample(int64_t time, float Vsp, float Voem, float V3, float V4, float A1, float A2, float temperature)
    {
        m_endTimestamp = time;
        m_Vsp.integrate(Vsp);
        m_Voem.integrate(Voem);
        m_V3.integrate(V3);
        m_V4.integrate(V4);
        m_A1.integrate(A1);
        m_A2.integrate(A2);
        m_temperature.integrate(temperature);        
        ++m_sampleCount;
        if (A1 >= m_dischargeCurrentThreshold) ++m_dischargeCount;
        if (A1 <= m_chargeCurrentThreshold) ++m_chargeCount;
    }

    bool intervalIsComplete() const
    {
        return (m_sampleInterval != 0) && (m_endTimestamp - m_beginTimestamp >= m_sampleInterval);
    }
    
    bool isWorthSending() const
    {
        if (m_sampleCount == 0) return false; // Never enqueue a sample with no data; that's useless.
        return true;
    }
    
    void populateSample(StealthPowerSample& s)
    {
        s.m_trigger = StealthPowerSample::TriggerNone;
        s.m_beginTimestamp = m_beginTimestamp;
        s.m_endTimestamp = m_endTimestamp;
        
        s.m_Vsp = m_Vsp.average();
        s.m_Vsp_min = m_Vsp.minimum();
        s.m_Vsp_max = m_Vsp.maximum();
        
        s.m_Voem = m_Voem.average();
        s.m_Voem_min = m_Voem.minimum();
        s.m_Voem_max = m_Voem.maximum();
        
        s.m_V3 = m_V3.average();
        s.m_V3_min = m_V3.minimum();
        s.m_V3_max = m_V3.maximum();
        
        s.m_V4 = m_V4.average();  
        s.m_V4_min = m_V4.minimum();  
        s.m_V4_max = m_V4.maximum();  
        
        s.m_A1 = m_A1.average();
        s.m_A1_min = m_A1.minimum();
        s.m_A1_max = m_A1.maximum();
        
        s.m_A2 = m_A2.average();
        s.m_A2_min = m_A2.minimum();
        s.m_A2_max = m_A2.maximum();
        
        s.m_sampleCount = m_sampleCount;
        s.m_dischargeCount = m_dischargeCount;
        s.m_chargeCount = m_chargeCount;
    }
    
    bool setSampleInterval(float interval)
    {
        if (isfinite(interval))
        {
            m_sampleInterval = (int)floor(0.5F + interval);
            return true;
        }
        else
        {
            return false;
        }
    }
    
    bool setThresholds(float discharge, float charge)
    {
        m_dischargeCurrentThreshold = discharge;
        m_chargeCurrentThreshold = charge;
        return true;
    }
    

private:
    int64_t m_beginTimestamp;
    int64_t m_endTimestamp;
    Integrator m_Vsp; 
    Integrator m_Voem;
    Integrator m_V3;
    Integrator m_V4;
    Integrator m_A1;
    Integrator m_A2;
    Integrator m_temperature;
    uint32_t m_sampleCount;
    uint32_t m_dischargeCount;
    uint32_t m_chargeCount;
    float m_dischargeCurrentThreshold;
    float m_chargeCurrentThreshold;
    int m_sampleInterval;
};

#endif //_StealthPowerSampler_h_
