

#ifndef _StatusUpdateBacklog_h_
#define _StatusUpdateBacklog_h_

#include "StealthPowerStatusUpdate.h"

class StatusUpdateBacklog
{
public:
    static const size_t MaxNumUpdates = 128;

    StatusUpdateBacklog() :
        m_backlog(0),
        m_writeIndex(0)
    {
    }
    
    void reset()
    {
        m_backlog = 0;
        m_writeIndex = 0;    
    }
    
    void push(const StealthPowerStatusUpdate& update)
    {
        // Note that this *always* succeeds, even if it means the oldest
        // sample gets overwritten.
        m_updates[m_writeIndex] = update;
        ++m_backlog;
        if (m_backlog > MaxNumUpdates) m_backlog = MaxNumUpdates;
        m_writeIndex = (m_writeIndex + 1) % MaxNumUpdates;
    }
    
    size_t backlog() const
    {
        return m_backlog;   
    }
    
    bool shift(StealthPowerStatusUpdate& update)
    {
        // Note: Since this gets called, presumably, outside of an ISR context, the caller
        // MUST take a lock or  disableg/reenable interrupts!!!
        if (m_backlog <= 0) return false;
        update = m_updates[(m_writeIndex + MaxNumUpdates - m_backlog) % MaxNumUpdates];
        --m_backlog;
        return true;
    }
    
    bool shift()
    {
        // Note: Since this gets called, presumably, outside of an ISR context, the caller
        // MUST take a lock or  disableg/reenable interrupts!!!
        if (m_backlog <= 0) return false;
        --m_backlog;
        return true;
    }
    
    bool shift_if(const StealthPowerStatusUpdate& update)
    {
        // Only shift if the front is still the same update that was peek'd a moment ago
        StealthPowerStatusUpdate front;
        if (!peek(front)) return false;
        if (front == update) return shift();
        return false;
    }
    
    bool peek(StealthPowerStatusUpdate& update)
    {
        // Note: Since this gets called, presumably, outside of an ISR context, the caller
        // MUST take a lock or  disableg/reenable interrupts!!!
        if (m_backlog <= 0) return false;
        update = m_updates[(m_writeIndex + MaxNumUpdates - m_backlog) % MaxNumUpdates];
        return true;
    }
    
    void retroactivelyTimestampSamples(int64_t currentUptime, int64_t walltimeDelta)
    {
        // Caller is responsible for thread safety again.
        
        // The MinimumValidTime here is the delta, in seconds, from Jan 1 1970 to 3/26/2016.
        // No data can ever have been taken beore that date, so any data timestamp before that
        // must be an uptime-timestamp rather than a wall-timestamp.
        const int64_t MinimumValidTime = 1459007487; // 3/26/2016.
        if (walltimeDelta < MinimumValidTime) return;

        for (size_t i = 0; i < m_backlog; ++i)
        {
            size_t index = (i + m_writeIndex + MaxNumUpdates - m_backlog) % MaxNumUpdates;
            StealthPowerStatusUpdate& update = m_updates[index];
            if (update.m_atTimestamp < MinimumValidTime)
            {
                //printf("Updating update %d in backlog\r\n", index);
                update.m_atTimestamp += walltimeDelta;
            }
        }
    }


private:
    size_t m_backlog;
    size_t m_writeIndex; 
    StealthPowerStatusUpdate m_updates[MaxNumUpdates];
};






#endif //_StatusUpdateBacklog_h_
