
#ifndef _SettingsManager_h_
#define _SettingsManager_h_

#include "mbed.h"
#include "SpiFlash25.h"


#define PAGE_SIZE               256
#define SECTOR_SIZE             64*1024
#define MEM_SIZE                2*1024*1024


class SettingsManager
{
public:
    static const size_t HOSTNAME_LENGTH = 128;
    
    static const uint32_t DefaultSignature = 0x130c8215;

    // This image appears at the top of the first two erasable sectors of the SPI flash.
#pragma pack(push,1)
    struct SettingsImage
    {
        uint32_t m_valid; // Must be exactly 1 for valid; any other value (ffffffff, 0) are invalid.
        uint32_t m_signature; // arbitrary value, change if you want to force an update
        //char m_apn[HOSTNAME_LENGTH]; // For now, RadioManager hardcodes the ATT M2M APN; it's only a liability to make it changeable remotely, at this stage.
        char m_host1[HOSTNAME_LENGTH];
        uint32_t m_port1;
        char m_host2[HOSTNAME_LENGTH];
        uint32_t m_port2;
        int32_t m_sampleIntervalInSeconds;
        int32_t m_dischargeThreshold; // in mA
        int32_t m_chargeThreshold; // in mA
        int32_t m_currentHysteresis; // in mA
        int32_t m_serverSettingsUpdateSeconds; // seconds after which to ask server for updated settings -- usually the socket/radio will go down before this anyway
        uint32_t m_statusUpdateEvictionTime;
        uint32_t m_voltageFilterLengthInSamples;
        int32_t m_vspUpdateThreshold;
        int32_t m_vspJumpThreshold;
        int32_t m_voemUpdateThreshold;
        int32_t m_voemJumpThreshold;
        int32_t m_v3UpdateThreshold;
        int32_t m_v3JumpThreshold;
        int32_t m_v4UpdateThreshold;
        int32_t m_v4JumpThreshold;
        uint32_t m_currentFilterLengthInSamples;
        int32_t m_a1UpdateThreshold;
        int32_t m_a1JumpThreshold;
        int32_t m_a2UpdateThreshold;
        int32_t m_a2JumpThreshold;
        uint32_t m_temperatureFilterLengthInSamples;
        int32_t m_temperatureUpdateThreshold;
        int32_t m_temperatureJumpThreshold;
        int32_t m_useFilteredA1ForBatteryState;
        int32_t m_minimumEventSampleDuration;
        
        // NOTE: Must pad this struct (before m_checksum) to an integral number of u32's if it isn't already...
        // char m_pad[3];
        uint32_t m_mtime; // The time_t that the settings were written--so the manager can always take the most recent
        uint32_t m_checksum; // Sum of all 32-bit chunks preceding this, XOR'd by the XOR of all preceding u32's.  Always the last thing in the struct.
        
        bool matches(const SettingsImage& other) const
        {
            return 0 == memcmp(this, &other, sizeof(SettingsImage));
            /*
            // Keep this around; useful for debugging settings update problems
            int i;
            const char* otherBuffer = (const char*)&other;
            const char* buffer = (const char*)this;
            bool matched = true;
            for (i = 0; i < sizeof(SettingsImage)-2*sizeof(uint32_t); ++i)
            {
                if (otherBuffer[i] != buffer[i])
                {
                    printf("Buffer mismatch offset %d, %02X != %02X\r\n", i, buffer[i], otherBuffer[i]);
                    matched = false;
                }
            }
            if (matched) printf("Everything matched.\r\n");
            return matched;
            */
        }
        
        
    };
#pragma pack(pop)

    SettingsManager(SpiFlash25& flash) :
       m_flash(flash)
    {
        m_settings.m_signature = DefaultSignature;
    }
    
    /*
    const char* getAPN() const
    {
        return m_settings.m_apn;
    }
    */
    
    const char* getHost1() const
    {
        return m_settings.m_host1;
    }
    
    uint16_t getPort1() const
    {
        return (uint16_t)m_settings.m_port1;
    }
    
    const char* getHost2() const
    {
        return m_settings.m_host2;
    }

    uint16_t getPort2() const
    {
        return (uint16_t)m_settings.m_port2;
    }
    
    float getSampleIntervalInSeconds() const
    {
        return (float)m_settings.m_sampleIntervalInSeconds;
    }
    
    float getSettingsUpdateIntervalInSeconds() const
    {
        return (float)m_settings.m_serverSettingsUpdateSeconds;
    }
    
    float getDischargeThreshold() const
    {
        return (float)(m_settings.m_dischargeThreshold * 0.001);
    }
    
    float getChargeThreshold() const
    {
        return (float)(m_settings.m_chargeThreshold * 0.001);
    }
    
    float getCurrentHysteresis() const
    {
        return (float)(m_settings.m_currentHysteresis * 0.001);
    }
    
    uint32_t getStatusUpdateEvictionTime() const
    {
        return (uint32_t)(m_settings.m_statusUpdateEvictionTime);
    }

    uint32_t getVoltageFilterLengthInSamples() const
    {
        return (uint32_t)(m_settings.m_voltageFilterLengthInSamples);
    }

    float getVspUpdateThreshold() const
    {
        return (float)(m_settings.m_vspUpdateThreshold * 0.001);
    }

    float getVspJumpThreshold() const
    {
        return (float)(m_settings.m_vspJumpThreshold * 0.001);
    }

    float getVoemUpdateThreshold() const
    {
        return (float)(m_settings.m_voemUpdateThreshold * 0.001);
    }

    float getVoemJumpThreshold() const
    {
        return (float)(m_settings.m_voemJumpThreshold * 0.001);
    }

    float getV3UpdateThreshold() const
    {
        return (float)(m_settings.m_v3UpdateThreshold * 0.001);
    }

    float getV3JumpThreshold() const
    {
        return (float)(m_settings.m_v3JumpThreshold * 0.001);
    }

    float getV4UpdateThreshold() const
    {
        return (float)(m_settings.m_v4UpdateThreshold * 0.001);
    }

    float getV4JumpThreshold() const
    {
        return (float)(m_settings.m_v4JumpThreshold * 0.001);
    }

    uint32_t getCurrentFilterLengthInSamples() const
    {
        return (uint32_t)(m_settings.m_currentFilterLengthInSamples);
    }

    float getA1UpdateThreshold() const
    {
        return (float)(m_settings.m_a1UpdateThreshold * 0.001);
    }

    float getA1JumpThreshold() const
    {
        return (float)(m_settings.m_a1JumpThreshold * 0.001);
    }

    float getA2UpdateThreshold() const
    {
        return (float)(m_settings.m_a2UpdateThreshold * 0.001);
    }

    float getA2JumpThreshold() const
    {
        return (float)(m_settings.m_a2JumpThreshold * 0.001);
    }

    uint32_t getTemperatureFilterLengthInSamples() const
    {
        return (uint32_t)(m_settings.m_temperatureFilterLengthInSamples);
    }

    float getTemperatureUpdateThreshold() const
    {
        return (float)(m_settings.m_temperatureUpdateThreshold * 0.001);
    }

    float getTemperatureJumpThreshold() const
    {
        return (float)(m_settings.m_temperatureJumpThreshold * 0.001);
    }
    
    bool getUseFilteredA1ForBatteryState() const
    {
        return m_settings.m_useFilteredA1ForBatteryState != 0;
    }
    
    int32_t getMinimumEventSampleDuration() const
    {
        return m_settings.m_minimumEventSampleDuration;        
    }

    void loadDefaultSettings()
    {
        memset(&m_settings, 0, sizeof(m_settings));
        // Loads memory settings from hardcoded constants--don't use this unless the flash is toast
        m_settings.m_valid = 1;
        m_settings.m_signature = DefaultSignature;
        m_settings.m_mtime = 0; // default value, earliest possible value.
        //snprintf(m_settings.m_apn, HOSTNAME_LENGTH, "m2m.com.attz");
        memset(m_settings.m_host1, HOSTNAME_LENGTH, 0);
        snprintf(m_settings.m_host1, HOSTNAME_LENGTH, "iotdata1.idlereduction.com");
        m_settings.m_port1 = 4700;
        memset(m_settings.m_host2, HOSTNAME_LENGTH, 0);
        snprintf(m_settings.m_host2, HOSTNAME_LENGTH, "52.54.102.85"); // EC2 elastic IP
        m_settings.m_port2 = 4700;
        m_settings.m_sampleIntervalInSeconds = 60 * 60; // hourly reporting by default
        m_settings.m_dischargeThreshold = 2500; // mA
        m_settings.m_chargeThreshold = -2500; // mA
        m_settings.m_currentHysteresis = 250; // mA
        m_settings.m_serverSettingsUpdateSeconds = 60 * 60; // hourly updates for server settings by default
        m_settings.m_statusUpdateEvictionTime = 60*60; // at least report status hourly if no other reason
        m_settings.m_voltageFilterLengthInSamples = 100; // smooth exponentially over this many samples
        m_settings.m_vspUpdateThreshold = 50; // mV
        m_settings.m_vspJumpThreshold = 1000; // mV
        m_settings.m_voemUpdateThreshold = 50; // mV
        m_settings.m_voemJumpThreshold = 1000; // mV
        m_settings.m_v3UpdateThreshold = 50; // mV
        m_settings.m_v3JumpThreshold = 1000; // mV
        m_settings.m_v4UpdateThreshold = 50; // mV
        m_settings.m_v4JumpThreshold = 1000; // mV
        m_settings.m_currentFilterLengthInSamples = 0;
        m_settings.m_a1UpdateThreshold = 1000; // mA
        m_settings.m_a1JumpThreshold = 3000;
        m_settings.m_a2UpdateThreshold = 1000; // mA
        m_settings.m_a2JumpThreshold = 3000;
        m_settings.m_temperatureFilterLengthInSamples = 100;
        m_settings.m_temperatureUpdateThreshold = 2500; // thousands of degree
        m_settings.m_temperatureJumpThreshold = 2500;
        m_settings.m_useFilteredA1ForBatteryState = 0; //false
        m_settings.m_minimumEventSampleDuration = 5;
        _populateChecksum(m_settings);
    }
    
    void printSettings()
    {
        printf("                 dischargeThreshold: %d mA\r\n", m_settings.m_dischargeThreshold);
        printf("                    chargeThreshold: %d mA\r\n", m_settings.m_chargeThreshold);
        printf("                  currentHysteresis: %d mA\r\n", m_settings.m_currentHysteresis);
        printf("              regularSampleInterval: %d s\r\n", m_settings.m_sampleIntervalInSeconds);
        printf("              settingsCheckInterval: %d s\r\n", m_settings.m_serverSettingsUpdateSeconds);
        printf("                        primaryHost: '%s'\r\n", m_settings.m_host1);
        printf("                        primaryPort: %d\r\n", m_settings.m_port1);
        printf("                      secondaryHost: '%s'\r\n", m_settings.m_host2);
        printf("                      secondaryPort: %d\r\n", m_settings.m_port2);
        printf("           statusUpdateEvictionTime: %d s\r\n", m_settings.m_statusUpdateEvictionTime);
        printf("       voltageFilterLengthInSamples: %d\r\n", m_settings.m_voltageFilterLengthInSamples);
        printf("                 vspUpdateThreshold: %d mV\r\n", m_settings.m_vspUpdateThreshold);
        printf("                   vspJumpThreshold: %d mV\r\n", m_settings.m_vspJumpThreshold);
        printf("                voemUpdateThreshold: %d mV\r\n", m_settings.m_voemUpdateThreshold);
        printf("                  voemJumpThreshold: %d mV\r\n", m_settings.m_voemJumpThreshold);
        printf("                  v3UpdateThreshold: %d mV\r\n", m_settings.m_v3UpdateThreshold);
        printf("                    v3JumpThreshold: %d mV\r\n", m_settings.m_v3JumpThreshold);
        printf("                  v4UpdateThreshold: %d mV\r\n", m_settings.m_v4UpdateThreshold);
        printf("                    v4JumpThreshold: %d mV\r\n", m_settings.m_v4JumpThreshold);
        printf("       currentFilterLengthInSamples: %d\r\n", m_settings.m_currentFilterLengthInSamples);
        printf("                  a1UpdateThreshold: %d mA\r\n", m_settings.m_a1UpdateThreshold);
        printf("                    a1JumpThreshold: %d mA\r\n", m_settings.m_a1JumpThreshold);
        printf("                  a2UpdateThreshold: %d mA\r\n", m_settings.m_a2UpdateThreshold);
        printf("                    a2JumpThreshold: %d mA\r\n", m_settings.m_a2JumpThreshold);
        printf("   temperatureFilterLengthInSamples: %d s\r\n", m_settings.m_temperatureFilterLengthInSamples);
        printf("         temperatureUpdateThreshold: %d mDegC\r\n", m_settings.m_temperatureUpdateThreshold);
        printf("           temperatureJumpThreshold: %d mDegC\r\n", m_settings.m_temperatureJumpThreshold);
        printf("       useFilteredA1ForBatteryState: %d\r\n", m_settings.m_useFilteredA1ForBatteryState);
        printf("         minimumEventSampleDuration: %d\r\n", m_settings.m_minimumEventSampleDuration);
    }

    bool setAll(
        const char* primaryHost,
        uint16_t primaryPort,
        const char* secondaryHost,
        uint16_t secondaryPort,
        int32_t dischargeThreshold,     /* threshold in mV above which the stealth battery current is considered 'discharging' */
        int32_t chargeThreshold,        /* threshold below which the stealth battery current is considered 'charging' */
        int32_t currentHysteresis,      /* amount to ease the discharging/charging thresholds once already in that respective state */
        int32_t regularSampleInterval,  /* interval, in seconds, at which to send regularly-sampled data; 0 means "don't" */
        int32_t settingsCheckInterval,
        uint32_t statusUpdateEvictionTime,
        uint32_t voltageFilterLengthInSamples,
        int32_t vspUpdateThreshold,
        int32_t vspJumpThreshold,
        int32_t voemUpdateThreshold,
        int32_t voemJumpThreshold,
        int32_t v3UpdateThreshold,
        int32_t v3JumpThreshold,
        int32_t v4UpdateThreshold,
        int32_t v4JumpThreshold,
        uint32_t currentFilterLengthInSamples,
        int32_t a1UpdateThreshold,
        int32_t a1JumpThreshold,
        int32_t a2UpdateThreshold,
        int32_t a2JumpThreshold,
        uint32_t temperatureFilterLengthInSamples,
        int32_t temperatureUpdateThreshold,
        int32_t temperatureJumpThreshold,
        int32_t useFilteredA1ForBatteryState,
        int32_t minimumEventSampleDuration)
    {
        if (!primaryHost || strlen(primaryHost) >= HOSTNAME_LENGTH) return false;
        if (!secondaryHost || strlen(secondaryHost) >= HOSTNAME_LENGTH) return false;
        if (regularSampleInterval < 0) regularSampleInterval = 0;
        if (settingsCheckInterval < 20) settingsCheckInterval = 20; // min 20 sec between settings checks
        if (settingsCheckInterval > 24*60*60) settingsCheckInterval = 24*60*60; // max 1 day between settings check
        if (minimumEventSampleDuration < 0) minimumEventSampleDuration = 0;
         
        memset(m_settings.m_host1, 0, HOSTNAME_LENGTH);
        strncpy(m_settings.m_host1, primaryHost, HOSTNAME_LENGTH);
        m_settings.m_port1 = primaryPort;
        memset(m_settings.m_host2, 0, HOSTNAME_LENGTH);
        strncpy(m_settings.m_host2, secondaryHost, HOSTNAME_LENGTH);
        m_settings.m_port2 = secondaryPort;
        m_settings.m_dischargeThreshold = dischargeThreshold;
        m_settings.m_currentHysteresis = currentHysteresis;
        m_settings.m_chargeThreshold = chargeThreshold;
        m_settings.m_sampleIntervalInSeconds = regularSampleInterval;
        m_settings.m_serverSettingsUpdateSeconds = settingsCheckInterval;
        m_settings.m_statusUpdateEvictionTime = statusUpdateEvictionTime;
        m_settings.m_voltageFilterLengthInSamples = voltageFilterLengthInSamples;
        m_settings.m_vspUpdateThreshold = vspUpdateThreshold;
        m_settings.m_vspJumpThreshold = vspJumpThreshold;
        m_settings.m_voemUpdateThreshold = voemUpdateThreshold;
        m_settings.m_voemJumpThreshold = voemJumpThreshold;
        m_settings.m_v3UpdateThreshold = v3UpdateThreshold;
        m_settings.m_v3JumpThreshold = v3JumpThreshold;
        m_settings.m_v4UpdateThreshold = v4UpdateThreshold;
        m_settings.m_v4JumpThreshold = v4JumpThreshold;
        m_settings.m_currentFilterLengthInSamples = currentFilterLengthInSamples;
        m_settings.m_a1UpdateThreshold = a1UpdateThreshold;
        m_settings.m_a1JumpThreshold = a1JumpThreshold;
        m_settings.m_a2UpdateThreshold = a2UpdateThreshold;
        m_settings.m_a2JumpThreshold = a2JumpThreshold;
        m_settings.m_temperatureFilterLengthInSamples = temperatureFilterLengthInSamples;
        m_settings.m_temperatureUpdateThreshold = temperatureUpdateThreshold;
        m_settings.m_temperatureJumpThreshold = temperatureJumpThreshold;
        m_settings.m_useFilteredA1ForBatteryState = useFilteredA1ForBatteryState;
        m_settings.m_minimumEventSampleDuration = minimumEventSampleDuration;
        //printSettings();
        return true;
    }
   /* other settings you can think of here ... ? */



    bool loadSettingsFromFlash()
    {
        // Try to get settings from sector 0 and from sector 1.
        // If they're both valid, take the one with the newer mtime.
        // If neither is valid, return false.
        
        bool gotImage0 = false;
        SettingsImage image0;
        memset(&image0, 0, sizeof(SettingsImage));
        gotImage0 = _loadIfValid(0*SECTOR_SIZE, image0);
        
        bool gotImage1 = false;
        SettingsImage image1;
        memset(&image1, 0, sizeof(SettingsImage));
        gotImage1 = _loadIfValid(1*SECTOR_SIZE, image1);
        
        if (gotImage0 && !gotImage1)
        {
            m_settings = image0;
            printf("Loaded settings from sector 0 because sector 1 was bad\r\n");
        }
        else if (gotImage1 && !gotImage0)
        {
            m_settings = image1;
            printf("Loaded settings from sector 1 because sector 0 was bad\r\n");
        }
        else if (gotImage0 && gotImage1)
        {
            if (image0.m_mtime >= image1.m_mtime)
            {
                printf("Loaded settings from sector 0\r\n");
                m_settings = image0;
            }
            else
            {
                printf("Loaded settings from sector 1\r\n");
                m_settings = image1;
            }
        }
        
        return gotImage0 || gotImage1;
    }
        
    bool writeSettingsToFlash()
    {
        int mtime = SystemTimeKeeper::instance().walltime();
        
        // Compare to sector 0--if different, erase and write sector 0.
        SettingsImage image;
        memset(&image, 0, sizeof(SettingsImage));
        if (_load(0*SECTOR_SIZE, image))
        {
            if (!image.matches(m_settings) ||
                !_checkChecksum(image))
            {
                printf("Saving updated settings in sector 0.\r\n");
                // Update the settings' mtime iff we're committed to rewriting flash
                m_settings.m_mtime = mtime;
                _populateChecksum(m_settings);
                
                m_flash.clear_sector(0*SECTOR_SIZE);
                if (!m_flash.write(0*SECTOR_SIZE, sizeof(m_settings), (char*)&m_settings))
                {
                   printf("Error writing settings to sector 0.\r\n");   
                }
            }
            else
            {
                //printf("Settings already written in sector 0.\r\n");
            }
        }
        else
        {
            printf("Couldn't validate settings in sector 0.\r\n");
        }
        
        // Compare to sector 1--if different, erase and write sector 1.
        memset(&image, 0, sizeof(SettingsImage));
        if (_load(1*SECTOR_SIZE, image))
        {
            if (!image.matches(m_settings) ||
                !_checkChecksum(image))
            {
                printf("Saving updated settings in sector 1.\r\n");

                // Update the settings' mtime iff we're committed to rewriting flash
                m_settings.m_mtime = mtime;
                _populateChecksum(m_settings);

                m_flash.clear_sector(1*SECTOR_SIZE);
                if (!m_flash.write(1*SECTOR_SIZE, sizeof(m_settings), (char*)&m_settings))
                {
                   printf("Error writing settings to sector 1.\r\n");   
                }
            }
            else
            {
                //printf("Settings already written in sector 1.\r\n");
            }
        }
        else
        {
            printf("Couldn't validate settings in sector 1.\r\n");
        }
        
        return true;
    }
    
    bool serviceSettingsInFlash()
    {
        // This really means do what writeSettingsToFlash() already does--make sure
        // the settings we *have* are actually in the flash, in duplicate.
        return writeSettingsToFlash();
    }
    
    void initializeSettings()
    {
        /*
        //Keep this around--nice to dump flash contents to hex...
        uint32_t buffer[64] = {0};
        int i;
        flash.read(0*SECTOR_SIZE, 64*4, (char*)&buffer);
        for (i = 0; i < 64; ++i)
        {
            printf(" %08x", buffer[i]);
            if (i % 8 == 7) printf("\r\n");
        }
        printf("\r\n");
        flash.read(1*SECTOR_SIZE, 64*4, (char*)&buffer);
        for (i = 0; i < 64; ++i)
        {
            printf(" %08x", buffer[i]);
            if (i % 8 == 7) printf("\r\n");
        }
        //printf("\r\n");
        */
        
        if (!loadSettingsFromFlash())
        {
            printf("No settings found in Flash; forced to load hard-coded default settings.\r\n");
            loadDefaultSettings();
        }
        if (!writeSettingsToFlash())
        {
            printf("Error writing settings to flash.\r\n");
        }
    }
    
protected:
    bool _load(uint32_t offset, SettingsImage& image)
    {
        return m_flash.read(offset, sizeof(SettingsImage), (char*)&image);
    }
    
    bool _loadIfValid(uint32_t offset, SettingsImage& image)
    {
        SettingsImage tempImage;
        memset(&tempImage, 0, sizeof(SettingsImage));
        if (!_load(offset, tempImage)) return false;
        if (!_checkChecksum(tempImage)) return false;
        if (tempImage.m_valid != 1) return false;
        if (tempImage.m_signature != DefaultSignature) return false;
        image = tempImage;
        return true;
    }

    uint32_t _calculateChecksum(const SettingsImage& settings)
    {
        uint32_t* raw = (uint32_t*)&settings;
        uint32_t xorValue = 0;
        uint32_t sum = 0;
        while (raw < (uint32_t*)&settings.m_checksum)
        {
            sum += (*raw);
            xorValue ^= (*raw);
            ++raw;
        }
        uint32_t calculated = sum; // ^ xorValue;
        //printf("calculated = %08x\r\n", calculated);
        return calculated;
    }

    void _populateChecksum(SettingsImage& settings)
    {
        // Fill in the included settings.checksum is what it should be
        settings.m_checksum = _calculateChecksum(settings);
    }
    
    bool _checkChecksum(const SettingsImage& settings)
    {
        // See if the included settings.checksum is what it should be
        return settings.m_checksum == _calculateChecksum(settings);
    }

    SpiFlash25& m_flash;
    
    SettingsImage m_settings;
};

#endif //_SettingsManager_h_

