

#ifndef _SampleBacklog_h_
#define _SampleBacklog_h_

#include "StealthPowerSample.h"

class SampleBacklog
{
public:
    static const size_t MaxNumSamples = 128;

    SampleBacklog() :
        m_backlog(0),
        m_writeIndex(0)
    {
    }
    
    void reset()
    {
        m_backlog = 0;
        m_writeIndex = 0;    
    }
    
    void push(const StealthPowerSample& sample)
    {
        // Note that this *always* succeeds, even if it means the oldest
        // sample gets overwritten.
        m_samples[m_writeIndex] = sample;
        ++m_backlog;
        if (m_backlog > MaxNumSamples) m_backlog = MaxNumSamples;
        m_writeIndex = (m_writeIndex + 1) % MaxNumSamples;
    }
    
    size_t backlog() const
    {
        return m_backlog;   
    }
    
    bool shift(StealthPowerSample& sample)
    {
        // Note: Since this gets called, presumably, outside of an ISR context, the caller
        // MUST take a lock or  disableg/reenable interrupts!!!
        if (m_backlog <= 0) return false;
        sample = m_samples[(m_writeIndex + MaxNumSamples - m_backlog) % MaxNumSamples];
        --m_backlog;
        return true;
    }
    
    bool shift_if(StealthPowerSample& sample)
    {
        // Only shift if the front is still the same update that was peek'd a moment ago
        StealthPowerSample front;
        if (!peek(front)) return false;
        if (front == sample) return shift();
        return false;
    }
    
    
    bool shift()
    {
        // Note: Since this gets called, presumably, outside of an ISR context, the caller
        // MUST take a lock or  disableg/reenable interrupts!!!
        if (m_backlog <= 0) return false;
        --m_backlog;
        return true;
    }
    
    bool peek(StealthPowerSample& sample)
    {
        // Note: Since this gets called, presumably, outside of an ISR context, the caller
        // MUST take a lock or  disableg/reenable interrupts!!!
        if (m_backlog <= 0) return false;
        sample = m_samples[(m_writeIndex + MaxNumSamples - m_backlog) % MaxNumSamples];
        return true;
    }
    
    void retroactivelyTimestampSamples(int64_t currentUptime, int64_t walltimeDelta)
    {
        // Caller is responsible for thread safety again.
        
        // The MinimumValidTime here is the delta, in seconds, from Jan 1 1970 to 3/26/2016.
        // No data can ever have been taken beore that date, so any data timestamp before that
        // must be an uptime-timestamp rather than a wall-timestamp.
        const int64_t MinimumValidTime = 1459007487; // 3/26/2016.
        if (walltimeDelta < MinimumValidTime) return;

        for (size_t i = 0; i < m_backlog; ++i)
        {
            size_t index = (i+m_writeIndex+MaxNumSamples-m_backlog)%MaxNumSamples;
            StealthPowerSample& sample = m_samples[index];
            if (sample.m_beginTimestamp < MinimumValidTime)
            {
                //printf("Updating sample %d in backlog\r\n", index);
                sample.m_beginTimestamp += walltimeDelta;
            }
            if (sample.m_endTimestamp < MinimumValidTime)
            {
                sample.m_endTimestamp += walltimeDelta;
            }                
        }
    }


private:
    size_t m_backlog;
    size_t m_writeIndex; 
    StealthPowerSample m_samples[MaxNumSamples];
};






#endif //_SampleBacklog_h_
