#ifndef _RadarSocketManager_h_
#define _RadarSocketManager_h_

#include "mbed.h"
#include <stdio.h>
#include "StealthRadarAPI.h"
#include "RadioManager.h"
#include "mtsas.h"

// How many retries to make on one host+port or the other before switching
static const int32_t RegularServerRetries = 3;

// This is hardcoded at compile time and can't be messed up by the database.
// We only resort back to this one if a large number of regular connections
// have already failed.  After [FailoverRetries] attempts on the one or two
// configured hosts, always try the failover server+port, in case the config
// got corrupted.
static const int32_t FailoverRetries = RegularServerRetries * 2 * 5;
static const char* FailoverHostname = "iotfailover999.idlereduction.com";
static const uint16_t FailoverPort = 4700;


class RadarSocketManager
{
public:
    RadarSocketManager(RadioManager& radioManager) :
        m_radioManager(radioManager),
        m_connected(false),
        m_hostname1(),
        m_port1(0),
        m_hostname2(),
        m_port2(0),
        m_transmitBufferBacklog(0),
        m_currentHostIndex(0),
        m_currentHostRetries(RegularServerRetries),
        m_failoverRetries(FailoverRetries),
        m_recvChecksum(0)
    {
        m_instance = this;
    }

    bool setServers(
        const char* host1,
        uint16_t port1,
        const char* host2,
        uint16_t port2)
    {
        // It's fine if a hostname is non-null but empty--but that causes it to get skipped over.
        if (host1)
        {
            m_hostname1 = host1;
            m_port1 = port1;
        }
        else
        {
            m_hostname1.clear();
            m_port1 = 0;
        }
        if (host2)
        {
            m_hostname2 = host2;
            m_port2 = port2;
        }
        else
        {
            m_hostname2.clear();
            m_port2 = 0;
        }
        m_currentHostIndex = 0;
        m_currentHostRetries = RegularServerRetries;
        
        printf("Set servers: %s:%d, %s:%d\r\n",
            m_hostname1.c_str(),
            m_port1,
            m_hostname2.c_str(),
            m_port2);
        
        return true; // can't really fail
    }
    
    /*bool m_getHostAndPortByIndex(size_t index, std::string& hostname, uint16_t& port)
    {
        if (
    }*/
    
    bool connect()
    {
        if (isConnected()) return true;
        m_connected = false;
        std::string result;
        
        resumeRadioWatchdog(); // Make sure wdt is running, but don't reset it here, in case there are a bunch of failed connect() calls

        // First make sure modem knows we're not supposed to be connected right now
        m_radioManager.sendCommand("AT#SH=1", 5000); // Ignore result--worst case is it was already shut down.
        
        // Configure socket behavior/mode (socket 1, srMode = 1 (data mount), rx = 1 (hex), keepalive = 10 min, listenauto = 0 (?), send data = 1 (hex))
        result = m_radioManager.sendCommand("AT#SCFGEXT=1,1,1,10,0,1", 5000); // Ignore result--worst case is it was already shut down.
        if (result.find("\r\nOK\r\n") != std::string::npos)
        {
            //printf("Configured socket successfully.\r\n");
        }
        else
        {
            printf("Failed to configure socket, failed to connect!\r\n");
            return false;
        }
        
        const char* useHostname = m_hostname1.c_str();
        uint16_t usePort = m_port1;
        
        if (m_failoverRetries > 0 &&
            ((!m_hostname1.empty() && m_port1 != 0) ||
            (!m_hostname2.empty() && m_port2 != 0)))
        {
            // Haven't switched to hard-coded failover, so rotate through the hosts/ports
            // programmed in flash settings
        
            if (m_currentHostRetries <= 0)
            {
                printf("Maximum server retries exceeded; switching to other server/port (hard failover in %d)\r\n", m_failoverRetries);
                m_currentHostIndex = (m_currentHostIndex + 1) % 2; // For now, hardcoded to 2 hosts
                m_currentHostRetries = RegularServerRetries;
            }
            
            // Also, if one of them is empty, always use the other one.
            if (m_currentHostIndex > 0)
            {
                useHostname =  m_hostname2.c_str();
                usePort = m_port2;
            }
        }
        else
        {
            // We *have* either failed a lot of times, or there are no valid hostnames in the flash
            // settings, so use the compile-time hard-coded failover server instead.
            printf("Using failover hostname/port.\r\n");
            useHostname = FailoverHostname;
            usePort = FailoverPort;
            m_failoverRetries = FailoverRetries;
        }

        m_currentHostRetries -= 1;
        m_failoverRetries -= 1;

        if (!useHostname || strlen(useHostname) == 0 || usePort == 0)
        {
            printf("Aborting connection attempt; empty hostname/port.\r\n");
            m_currentHostRetries = 0;
            return false;
        }
        
        printf("Connecting to '%s:%d'\r\n", useHostname, (int)usePort);
        
        // Open socket in COMMAND mode so we don't lose control of the modem like MTSAS's crazy implementation does
        char cmdBuffer[256] = { 0 };
        snprintf(cmdBuffer, 256, "AT#SD=1,0,%d,\"%s\",0,0,1", usePort, useHostname);
        result = m_radioManager.sendCommand(cmdBuffer, 30000);
        
        if (result.find("\r\nOK\r\n") != std::string::npos ||
            result.find("\r\nCONNECTED\r\n") != std::string::npos)
        {
            m_connected = true;

            // Reset the retry counters--we found one that works.
            m_currentHostRetries = RegularServerRetries;
            m_failoverRetries = FailoverRetries;;

            // And don't let the watchdog expire
            stopRadioWatchdog();
        }

        return m_connected;
    }
    
    void disconnect()
    {
        //if (!isConnected()) return; // Never assume this is already done.
        m_connected = false;
        m_radioManager.sendCommand("AT#SH=1", 5000); // Ignore result--worst case is it was already shut down.
        // Disconnect socket, set blocking correctly, using m_radioManager.sendCommand()
    }

    bool isConnected() const
    {
        return m_connected;
    }
    
    static RadarSocketManager* instance()
    {
        return m_instance;
    }
    
    // owns the socket
    
    int sendBytes(const void* bytes,
       size_t size,
       uint32_t timeoutMS)
    {
        if (m_transmitBufferBacklog > 256 - size) return RADAR_NETWORK_ERROR; // too much, overflow xmit buffer
        for (int i = 0; i < size; ++i)
        {
            m_transmitBuffer[m_transmitBufferBacklog++] = ((char*)bytes)[i];
        }
        return size;
    }
    
    int flush(uint32_t timeoutMS)
    {
        // Whenever we attempt to transmit, start the radio wdt
        resumeRadioWatchdog();
        
        if (m_transmitBufferBacklog > 0)
        {
            int result = reallySendBytes(m_transmitBuffer, m_transmitBufferBacklog, timeoutMS);
            if (result == m_transmitBufferBacklog)
            {
                m_transmitBufferBacklog = 0;                
                return RADAR_OK;
            }
            else
            {
                return RADAR_NETWORK_ERROR;
            }
        }
        return 0;
    }
    
    uint32_t recvResetChecksum()
    {
        uint32_t temp = m_recvChecksum;
        m_recvChecksum = 0;
        return temp;
    }

    
    int reallySendBytes(
       const void* bytes,
       size_t size,
       uint32_t timeoutMS)
    {
        if (!m_connected) return RADAR_DISCONNECTED;
        
        char hexMiniBuffer[4] = { 0 };
        std::string hexPayload;
        std::string result;
        
        const char* characters = (const char*) bytes;
        //for (int j = 0; j < size; ++j) temp[j] = temp[j] | 0x80; // or in a bit so there's never a 0 transmitted.
        
        char cmdBuffer[32] = { 0 };
        
        
        hexPayload = "";
        for (int j = 0; j < size; ++j)
        {
            snprintf(hexMiniBuffer,3,"%02X",characters[j]);
            hexPayload += hexMiniBuffer;
        }
        hexPayload += (char)0x1A; // Ctrl+Z that ends the data line
        
        snprintf(cmdBuffer, 32, "AT#SSEND=1,%d", size);
        result = m_radioManager.sendCommandWithBinaryPayload(cmdBuffer, hexPayload.c_str(), hexPayload.length(), timeoutMS);       
        if (result.find("\r\nOK\r\n") == std::string::npos)
        {
            //Error!
            return RADAR_NETWORK_ERROR;
        }
        return size;
    }
    
    
    int getAvailableRecvBytes()
    {
        if (!m_connected) return RADAR_DISCONNECTED;
        std::string result;
        result = m_radioManager.sendCommand("AT#SI=1", 8000);
        // Expect back: #SI: 1,11,0,4,11\r\nOK\r\n
        
        if (result.find("\r\nOK\r") == std::string::npos ||
            result.find("\r\nERROR") != std::string::npos)
        {
            return RADAR_NETWORK_ERROR;
        }
        
        size_t foundpos = result.find("#SI: ");
        if (foundpos != std::string::npos)
        {
            int ignore1=0;
            int ignore2=0;
            int ignore3=0;
            int buffer_in = 0;
            int ignore4=0;
            if (5 != sscanf(result.c_str()+foundpos+5, "%d,%d,%d,%d,%d", &ignore1, &ignore2, &ignore3, &buffer_in, &ignore4))
            {
                return RADAR_NETWORK_ERROR;
            }
            else
            {
                // Got it.
                return buffer_in;
            }
        }
        else
        {
            return RADAR_NETWORK_ERROR;
        }
    }
    
    int recvBytes(
       void* bytes,
       size_t maxSize,
       uint32_t timeoutMS)
    {
        if (!m_connected) return RADAR_DISCONNECTED;
        std::string result;
        
        int availableBytes = 0;

        while (availableBytes >= 0 && availableBytes < maxSize)
        {
            availableBytes = getAvailableRecvBytes();
            if (availableBytes < maxSize) wait_ms(50); // Wait, and don't peg the processor.
        }
        
        if (availableBytes > maxSize) availableBytes = maxSize;
        
        char cmdBuffer[64] = {0};
        snprintf(cmdBuffer,64,"AT#SRECV=1,%d",availableBytes);
        
        
        result = m_radioManager.sendCommand(cmdBuffer, timeoutMS);       
        if (result.find("\r\nOK\r\n") == std::string::npos)
        {
            printf("Failed to recv!\r\n");
            return RADAR_NETWORK_ERROR;
        }
        
        size_t responseOffset = result.find("#SRECV:");
        if (responseOffset == std::string::npos)
        {
            printf("Couldn't find #SRECV\r\n");
            return RADAR_NETWORK_ERROR;
        }
        
        const char* dataSubstring = result.c_str() + responseOffset;
        while (*dataSubstring && *dataSubstring != '\n') ++dataSubstring; // go to the second \n
        ++dataSubstring;
        
        int index = 0;
        while (index < availableBytes && 1 == sscanf(dataSubstring, "%02hhX", &((char*)bytes)[index]))
        {
            m_recvChecksum += (uint32_t)((uint8_t*)bytes)[index];
            ++index;
            ++dataSubstring;
            if (!*dataSubstring) return RADAR_NETWORK_ERROR;
            ++dataSubstring;
            if (!*dataSubstring) return RADAR_NETWORK_ERROR;
        }
        
       // printf("At the end, index = %d, availableBytes was %d\r\n", index, availableBytes);
        
        if (index != availableBytes)
        {
            return RADAR_NETWORK_ERROR;
        }
        
        // Only once we have received bytes (proving 2-way comms are working) do we dsarm the radio wdt.
        // It would have been armed in the main() loop as soon as it knew it wanted to attempt any
        // radio communication.
        stopRadioWatchdog();

        return index;
    }
    
    void resumeRadioWatchdog()
    {
        m_radioManager.resumeWatchdog();
    }
    
    void feedRadioWatchdog()
    {
        m_radioManager.feedWatchdog();
    }
    
    void stopRadioWatchdog()
    {
        m_radioManager.stopWatchdog();
    }

protected:    
    RadioManager& m_radioManager;
    bool m_connected;
    std::string m_hostname1;
    uint16_t m_port1;
    std::string m_hostname2;
    uint16_t m_port2;
    static RadarSocketManager* m_instance;
    uint8_t m_transmitBuffer[256];
    size_t m_transmitBufferBacklog;
    
    int32_t m_currentHostIndex;
    int32_t m_currentHostRetries;
    int32_t m_failoverRetries; // retries, regardless of regular host used, before switching hardcoded failover

    uint32_t m_recvChecksum;
};



#endif //_RadarSocketManager_h_
