
#ifndef _PICManager_h_
#define _PICManager_h_

#include "mbed.h"
#include "mtsas.h"
#include "PICInterface.h"
#include <string.h>
#include "StealthPowerSampler.h"
#include "SystemTimeKeeper.h"
#include "StatusUpdateManager.h"
#include "SampleBacklog.h"

struct PICData
{
    PICData()
    {
        // Easier than initializer list...
        memset(this, 0, sizeof(*this));
    }
    bool receivedParametricData;
    int state;
    int Vsp;
    int Voem;
    int V3;
    int V4;
    int A1;
    int A2;
    int K;
    bool receivedConfigData;
    int version;
    int Vcutoff;
    int Vcharge;
    int Tdisch;
    int ProgSel;
};

class PICManager
{
public:
    static const size_t PICBufferLength = 128;
    static const int PICTimeoutMS = 5000;
    
    PICManager(StealthPowerSampler* intervalSampler, StealthPowerSampler* eventSampler, SampleBacklog* eventBacklog) :
        m_previousState(BatteryStateUnspecified),
        m_uart(NULL),
        m_activityTimer(),
        m_established(false),
        m_bufferSize(0),
        m_intervalSampler(intervalSampler),
        m_eventSampler(eventSampler),
        m_eventBacklog(eventBacklog),
        m_dischargeThreshold(0.5),
        m_chargeThreshold(-0.5),
        m_flaggedPICError(false),
        m_useFilteredA1(false),
        m_minimumEventSampleDuration(0),
        m_currentThresholdHysteresis(0.250)
    {
        m_instance = this;
        memset(m_buffer, 0, PICBufferLength);
    }

    bool initialize()
    {
        m_uart = new MTSSerial(SERIAL_TX, SERIAL_RX);
        m_uart->baud(19200);
        m_uart->format(8, SerialBase::None, 1);
        m_instance = this;
        m_established = false;
        m_awaitingConfigPacket = false;
        m_activityTimer.reset();
        m_activityTimer.start();
        while (m_activityTimer.read_ms() < PICTimeoutMS && !m_established)
        {
            __disable_irq(); // Normaly servicePIC gets called in an ISR/Ticker
            servicePIC();
            __enable_irq();
        }
        return m_established;
    }
    
    bool established() const
    {
        return m_established;
    }
    
    bool setThresholds(float dischargeThreshold, float chargeThreshold, float currentHysteresis, bool useFilteredA1, int32_t minimumEventSampleDuration)
    {
        m_dischargeThreshold = dischargeThreshold;
        m_chargeThreshold = chargeThreshold;
        m_useFilteredA1 = useFilteredA1;
        m_minimumEventSampleDuration = minimumEventSampleDuration;
        m_currentThresholdHysteresis = currentHysteresis;
        
        // The samplers need this as well for their charge/discharge counts
        if (m_eventSampler) m_eventSampler->setThresholds(dischargeThreshold, chargeThreshold);
        if (m_intervalSampler) m_intervalSampler->setThresholds(dischargeThreshold, chargeThreshold);
        
        return true;
    }
        
    
    void servicePIC()
    {
        if (!m_established && !m_awaitingConfigPacket)
        {
            PICTransmitRequestConfigPacket();
            m_awaitingConfigPacket = true;
        }
        
        while (m_uart->readable())
        {
            if (m_bufferSize > 126)
            {
                m_established = false;
                m_bufferSize = 0;
            }
            char c = 0;
            m_uart->read(c);
            m_buffer[m_bufferSize] = c;
            m_buffer[m_bufferSize+1] = '\0';
            if (m_buffer[m_bufferSize] == '\n' || m_buffer[m_bufferSize] == '\r' || m_buffer[m_bufferSize] == '\0')
            {
                m_buffer[m_bufferSize] = '\n'; // in case it was a \r
                if (m_bufferSize <= 1 || strlen(m_buffer) <= 1)
                {
                    // Stray newline/cr, ignore.  Definitely don't try to format it into a std::string(char*,char*)!
                    m_bufferSize = 0;
                    return;   
                }

                std::string printable(m_buffer, m_buffer+strlen(m_buffer)-1);
                printf("PIC: '%s' (%d)\r\n", printable.c_str(), m_bufferSize); fflush(stdout);
                
                PIC_result_t result = PICConsumePacket(m_buffer);
                if (result != PICResultOK)
                {
                    printf("Error %d from PIC packet.\r\n", (int)result);
                }
                m_bufferSize = 0;
            }
            else
            {
                ++m_bufferSize;
            }
        }
        
        if (m_activityTimer.read_ms() > PICTimeoutMS && !m_flaggedPICError)
        {
            m_established = false;
            m_flaggedPICError = true;
            printf("PIC: Data Timeout Error.\r\n");
            StatusUpdateManager::instance()->setPICConnected(m_established);
        }
        
    }
    
    static PICManager* instance()
    {
        return m_instance;
    }
    
    // Callbacks
    PIC_result_t TransmitCallback(
        const char* bytes,
        size_t count)
    {
        // WARNING: This does *not* handle overflows; it is expected that
        // only a reasonable amount of data will ever be transmitted at once.
        for (size_t i = 0; i < count; ++i)
        {
            m_uart->write(bytes[i], 1000);
        }
        return PICResultOK;
    }
    
    PIC_result_t ConfigurationDataCallback(
        int version,
        int Vcutoff,
        int Vcharge,
        int Tdisch,
        int ProgSel)
    {
        m_awaitingConfigPacket = false;
        m_established = true;
        m_flaggedPICError = false;
        m_activityTimer.reset();
        m_activityTimer.start();
        
        // We have no use for this PIC packet at this time.
        
        return PICResultOK;
    }
    

    PIC_result_t ParametricDataCallback(
        char type,
        int state,
        int Vsp,
        int Voem,
        int V3,
        int V4,
        int A1,
        int A2,
        int K)
    {
        m_established = true;
        StatusUpdateManager::instance()->setPICConnected(m_established);
        m_activityTimer.reset();
        m_activityTimer.start();

        m_picData.receivedParametricData = true;
        m_picData.state = state;
        m_picData.Vsp = Vsp;
        m_picData.Voem = Voem;
        m_picData.V3 = V3;
        m_picData.V4 = V4;
        m_picData.A1 = A1;
        m_picData.A2 = A2;
        m_picData.K = K;
        
        // If it was a regular-interval parametric data packet, update all the samplers that
        // will need to integrate this new sample.
        if (type == 'I')
        {
            if (m_intervalSampler)
            {
                m_intervalSampler->sample(
                    SystemTimeKeeper::instance().uptime(),
                    Vsp * 0.01,
                    Voem * 0.01,
                    V3 * 0.01,
                    V4 * 0.01,
                    A1  * 0.1,
                    A2 * 0.1,
                    0.0F /* no temperature data right now */ );
            }
        }
        
        // The status update manager always gets notified of latest system state, and it take care of deciding
        // whether that's important enough to forward to server (i.e. if something changed)
        StatusUpdateManager::instance()->setPICState(state, K);
        
        StatusUpdateManager::instance()->setLatestValues(
            Vsp * 0.01,
            Voem * 0.01,
            V3 * 0.01,
            V4 * 0.01,
            A1  * 0.1,
            A2 * 0.1,
            0.0F /* no temperature data right now */ );
        
        double batteryCurrent = A1 * 0.1; // Raw (unfiltered) battery current
        
        if (m_useFilteredA1)
        {
            // If filtered A1 was requested, use that value instead.
            batteryCurrent = StatusUpdateManager::instance()->getFilteredA1();
        }
        
        // If we're already discharging, lower the bar to stay discharging just a little bit
        float hysteresisDischargeThreshold = m_dischargeThreshold;
        if (m_previousState == BatteryStateDischarging) hysteresisDischargeThreshold -= m_currentThresholdHysteresis;
        
        // If we're already charging, lower the bar (raise the negative threshold) to stay charging just a bit
        float hysteresisChargeThreshold = m_chargeThreshold;
        if (m_previousState == BatteryStateCharging) hysteresisChargeThreshold += m_currentThresholdHysteresis;
        
        if (batteryCurrent >= hysteresisDischargeThreshold)
        {
            StatusUpdateManager::instance()->setBatteryState(BatteryStateDischarging);
            if (m_previousState != BatteryStateDischarging)
            {
                printf("Entered BATTERY-DISCHARGING state.\r\n");
                _concludeEventSample();
            }
        }
        else if (batteryCurrent <= hysteresisChargeThreshold)
        {
            StatusUpdateManager::instance()->setBatteryState(BatteryStateCharging);
            if (m_previousState != BatteryStateCharging)
            {
                printf("Entered BATTERY-CHARGING state.\r\n");
                _concludeEventSample();
            }
        }
        else
        {
            StatusUpdateManager::instance()->setBatteryState(BatteryStateQuiescent);
            if (m_previousState != BatteryStateQuiescent)
            {
                printf("Entered BATTERY-QUIESCENT state.\r\n");
                _concludeEventSample();
            }
        }
        
        // Sample into the event sampler *after* we've decided what event is currently happening
        if (type == 'I')
        {
            if (m_eventSampler)
            {
                m_eventSampler->sample(
                    SystemTimeKeeper::instance().uptime(),
                    Vsp * 0.01,
                    Voem * 0.01,
                    V3 * 0.01,
                    V4 * 0.01,
                    A1  * 0.1,
                    A2 * 0.1,
                    0.0F /* no temperature data right now */);
            }
        }
        return PICResultOK;
    }
    
    const PICData& latestPICData() const
    {
        return m_picData;
    }
    
    void _concludeEventSample()
    {
        if (m_eventSampler->isWorthSending())
        {
            StealthPowerSample sample;
            m_eventSampler->populateSample(sample);
            if (m_previousState == BatteryStateCharging)
            {
                sample.m_trigger = StealthPowerSample::TriggerCharge;
                if (sample.m_endTimestamp - sample.m_beginTimestamp >= m_minimumEventSampleDuration)
                {
                    printf("Enqueuing Battery-Charge Event Sample of length %d.\r\n",
                        (int)sample.m_endTimestamp - (int)sample.m_beginTimestamp);
                    m_eventBacklog->push(sample);
                }
                else
                {
                    printf("Dropped Battery-Charge Event Sample due to length %d/%d.\r\n",
                        (int)sample.m_endTimestamp - (int)sample.m_beginTimestamp,
                        (int)m_minimumEventSampleDuration);
                }
            }
            else if (m_previousState == BatteryStateDischarging)
            {
                sample.m_trigger = StealthPowerSample::TriggerDischarge;
                if (sample.m_endTimestamp - sample.m_beginTimestamp >= m_minimumEventSampleDuration)
                {
                    printf("Enqueuing Battery-Discharge Event Sample of length %d.\r\n",
                        (int)sample.m_endTimestamp - (int)sample.m_beginTimestamp);
                    m_eventBacklog->push(sample);
                }
                else
                {
                    printf("Dropped Battery-Discharge Event Sample due to length %d/%d.\r\n",
                        (int)sample.m_endTimestamp - (int)sample.m_beginTimestamp,
                        (int)m_minimumEventSampleDuration);
                }
            }
        }
        
        m_eventSampler->reset(SystemTimeKeeper::instance().uptime());
        m_previousState = StatusUpdateManager::instance()->getBatteryState();
    }
    
private:

    BatteryState m_previousState;
    static PICManager* m_instance;
    MTSSerial *m_uart;
    Timer m_activityTimer;
    volatile bool m_established;
    bool m_awaitingConfigPacket;
    char m_buffer[PICBufferLength];
    size_t m_bufferSize;
    PICData m_picData;
    StealthPowerSampler *m_intervalSampler;
    StealthPowerSampler *m_eventSampler;
    SampleBacklog* m_eventBacklog;
    float m_dischargeThreshold;
    float m_chargeThreshold;
    bool m_flaggedPICError;
    bool m_useFilteredA1;
    int32_t m_minimumEventSampleDuration;
    float m_currentThresholdHysteresis;
};




#endif //_PICManager_h_
