/*
 GroveStreams is a helper class to assist with pushing data into the   GroveStreams.com IoT platform.
*/
#ifndef GROVESTREAMS_H_
#define GROVESTREAMS_H_
#include "mbed.h"
#include "EthernetInterface.h"
#include "NetworkAPI/tcp/socket.hpp"
#include "LcdDiscoF746NgTracer.h"
class GroveStreams
{
protected:
    EthernetInterface _eth;
    LcdDiscoF746NgTracer* _pLcd;
    //const char* _domain;
    uint16_t _port;
    const char* _apiKey;
    const char* _myIPAddress;  //Don't Change. Set below from DHCP. Needed by GroveStreams to verify that a device is not uploading more than once every 10s.
public:
    /** Constructor with apiKey and Lcd tracer
    * \param apiKey the secret api_key for your GroveStreams organization  \param pLcd optional. Pass NULL or and LCD for tracing
    */
    GroveStreams(const char* apiKey, LcdDiscoF746NgTracer* pLcd);
    /** Constructor with apiKey and Lcd tracer. * \param apiKey the secret api_key for your GroveStreams organization  */
    GroveStreams(const char* apiKey);  
    /** Destructor *  */
    ~GroveStreams();
protected:
    int init(const char* apiKey, LcdDiscoF746NgTracer* pLcd);
    void printf(const char* format, ...);
public:
    /** Send sample data to GroveStreams. Attempts to restore dropped ethernet  connections
    * \param componentId The ID of the component to put the samples into \param samples a string that is URL Encoded and contains all of the 
    *  samples. Place an ampersand at the beggining of each sample substring.
    *  The substring is &streamId=value. 
    *  Example: &temp=6&amperes=78.231&on=true&position=on+ground \return 0 on success    */
    unsigned long send(const char* componentId, const char* samples);
    unsigned long sendJPA(const char* componentId, const char* samples);
    unsigned long sendInitDatasJPA(const char* componentId, const char* samples);   
    /** Send sample data to GroveStreams. Attempts to restore dropped ethernet  connections   *
    * \param componentId The ID of the component to put the samples into
    * \param samples a string that is URL Encoded and contains all of the  samples. Place an ampersand at the beggining of each sample substring.
    *  The substring is &streamId=value. 
    *  Example: &temp=6&amperes=78.231&on=true&position=on+ground * \param resultBuffer a string that will contain the http result body
    * \param resultBufferSize the size of the result buffer * \return 0 on success    */
    unsigned long send(const char* componentId, const char* samples, char* resultBuffer, size_t resultBufferSize);   
    /** Send sample data to GroveStreams. Attempts to restore dropped ethernet    connections   *
    * \param componentId The ID of the component to put the samples into    * \param samples a string that is URL Encoded and contains all of the 
    *  samples. Place an ampersand at the beggining of each sample substring.
    *  The substring is &streamId=value. 
    *  Example: &temp=6&amperes=78.231&on=true&position=on+ground    * \param componentName optional (NULL). the name assigned to a newly created component.
    * \param compTmplId optional (NULL). the component template ID to be used for creating a  new component 
    * \param resultBuffer a string that will contain the http result body* \param resultBufferSize the size of the result buffer
    * \return 0 on success    */
    unsigned long send(const char* componentId, const char* samples, const char* componentName, const char* compTmplId, char* resultBuffer, size_t resultBufferSize);
    unsigned long sendJPA(const char* componentId, const char* samples, const char* componentName, const char* compTmplId, char* resultBuffer, size_t resultBufferSize);
    unsigned long sendInitDatasJPA(const char* componentId, const char* samples, const char* componentName, const char* compTmplId, char* resultBuffer, size_t resultBufferSize);  
    /** Send sample data to GroveStreams. Does not attempt to restore dropped  ethernet connections.   *
    * \param componentId The ID of the component to put the samples into
    * \param samples a string that is URL Encoded and contains all of the 
    *  samples. Place an ampersand at the beggining of each sample substring.
    *  The substring is &streamId=value. 
    *  Example: &temp=6&amperes=78.231&on=true&position=on+ground
    * \param componentName optional (NULL). the name assigned to a newly created component.
    * \param compTmplId optional (NULL). the component template ID to be used for creating a  *  new component 
    * \param resultBuffer a string that will contain the http result body  * \param resultBufferSize the size of the result buffer
    * \return 0 on success    */
    unsigned long sendNoRetry(const char* componentId, const char* samples, const char* componentName, const char* compTmplId, char* resultBuffer, size_t resultBufferSize);
    unsigned long sendNoRetryJPA(const char* componentId, const char* samples, const char* componentName, const char* compTmplId, char* resultBuffer, size_t resultBufferSize);
    unsigned long sendNoRetryInitDatasJPA(const char* componentId, const char* samples, const char* componentName, const char* compTmplId, char* resultBuffer, size_t resultBufferSize);
    /** Starts ethernet. * Stops ethernet if it is running before starting it. * \return 0 on success   */
    int startEthernet();   
   /** Get the MAC address of your Ethernet interface -return a pointer to a string containing the MAC address */
    const char* getMACAddress();
};
#endif /* GROVESTREAMS_H_ */
