/**
@file N5110.h

@brief Header file containing member functions and variables

*/

#ifndef N5110_H
#define N5110_H

// Command Bytes - taken from Chris Yan's library
// More information can be found in the display datasheet
// H = 0 - Basic instructions
#define CMD_DC_CLEAR_DISPLAY   0x08
#define CMD_DC_NORMAL_MODE     0x0C
#define CMD_DC_FILL_DISPLAY    0x09
#define CMD_DC_INVERT_VIDEO    0x0D
#define CMD_FS_HORIZONTAL_MODE 0x00
#define CMD_FS_VERTICAL_MODE   0x02
#define CMD_FS_BASIC_MODE      0x00
#define CMD_FS_EXTENDED_MODE   0x01
#define CMD_FS_ACTIVE_MODE     0x00
#define CMD_FS_POWER_DOWN_MODE 0x04
// H = 1 - Extended instructions
#define CMD_TC_TEMP_0          0x04
#define CMD_TC_TEMP_1          0x05
#define CMD_TC_TEMP_2          0x06
#define CMD_TC_TEMP_3          0x07
#define CMD_BI_MUX_24          0x15
#define CMD_BI_MUX_48          0x13
#define CMD_BI_MUX_100         0x10
#define CMD_VOP_6V06           0xB2
#define CMD_VOP_7V38           0xC8

// number of pixels on display
#define WIDTH 84
#define HEIGHT 48
#define BANKS 6

#include "mbed.h"


class N5110
{

public:
    /** Create a N5110 object connected to the specified pins
    *
    * @param pwr Pin connected to Vcc on the LCD display (pin 1)
    * @param sce Pin connected to chip enable (pin 3)
    * @param rst Pin connected to reset (pin 4)
    * @param dc  Pin connected to data/command select (pin 5)
    * @param mosi Pin connected to data input (MOSI) (pin 6)
    * @param sclk Pin connected to serial clock (SCLK) (pin 7)
    * @param led Pin connected to LED backlight (must be PWM) (pin 8)
    *
    */
    N5110(PinName pwrPin, PinName scePin, PinName rstPin, PinName dcPin, PinName mosiPin, PinName sclkPin, PinName ledPin);

    /** Initialise display
    *
    *   Powers up the display and turns on backlight (50% brightness default).
    *   Sets the display up in horizontal addressing mode and with normal video mode.
    */
    void init();

    /** Turn off
    *
    *   Powers down the display and turns of the backlight.
    *   Needs to be reinitialised before being re-used.
    */
    void turnOff();

    /** Clears
    *
    *   Clears the screen.
    */
    void clear();

    /** Turn on normal video mode (default)
    *  Black on white
    */
    void normalMode();

    /** Turn on inverse video mode (default)
    *  White on black
    */
    void inverseMode();

    /** Set Brightness
    *
    *   Sets brightness of LED backlight.
    *   @param brightness - float in range 0.0 to 1.0
    */
    void setBrightness(float brightness);

    

    

    /** Set a Pixel
    *
    *   This function sets a pixel in the display. A call to refresh() must be made
    *   to update the display to reflect the change in pixels.
    *   @param  x - the x co-ordinate of the pixel (0 to 83)
    *   @param  y - the y co-ordinate of the pixel (0 to 47)
    */
    void setPixel(int x, int y);

    /** Clear a Pixel
    *
    *   This function clears pixel in the display. A call to refresh() must be made
    *   to update the display to reflect the change in pixels.
    *   @param  x - the x co-ordinate of the pixel (0 to 83)
    *   @param  y - the y co-ordinate of the pixel (0 to 47)
    */
    void clearPixel(int x, int y);

    /** Get a Pixel
    *
    *   This function gets the status of a pixel in the display.
    *   @param  x - the x co-ordinate of the pixel (0 to 83)
    *   @param  y - the y co-ordinate of the pixel (0 to 47)
    *   @returns
    *       0           - pixel is clear
    *       non-zero    - pixel is set
    */
    int getPixel(int x, int y);

    /** Refresh display
    *
    *   This functions refreshes the display to reflect the current data in the buffer.
    */
    void refresh();

    /** Randomise buffer
    *
    *   This function fills the buffer with random data.  Can be used to test the display.
    *   A call to refresh() must be made to update the display to reflect the change in pixels.
    *   The seed is not set and so the generated pattern will probably be the same each time.
    *   TODO: Randomise the seed - maybe using the noise on the AnalogIn pins.
    */
    void randomiseBuffer();

    


    


private:

    void setXYAddress(int x, int y);
    void initSPI();
    void turnOn();
    void reset();
    void clearRAM();
    void clearBuffer();
    void sendCommand(unsigned char command);
    void sendData(unsigned char data);

public:
    unsigned char buffer[84][6];  // screen buffer - the 6 is for the banks - each one is 8 bits;

private:  // private variables
    SPI*    spi;
    PwmOut* led;
    DigitalOut* pwr;
    DigitalOut* sce;
    DigitalOut* rst;
    DigitalOut* dc;

};



#endif
