/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@brief Revision 1.0.
@author Kyle Sturgess
@date   May 2015
*/

#ifndef MAIN_H
#define MAIN_H
#define USR_POWERDOWN   (0x104)
#define DIRECTION_TOLERANCE 0.05

#include "mbed.h"
#include "N5110.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

N5110 lcd(p7, p8, p9, p10, p11, p13, p21);/*!< Bus for the LCD, backlight on pwm pin */
unsigned char imgbuffer[3*84];/*!< buffer to hold the image */
FILE *img;/*!< creating a file calling img */
LocalFileSystem local("local");/*!< file system to retrieve files from flash memory */
///

/*!< connections for the joystick */
/// joystick button
DigitalIn button(p17);
/// potentiometer controlling X component
AnalogIn xPot(p15);
/// potentiometer controlling Y component
AnalogIn yPot(p16);
///

/*!< connections for objects using pwm, excluding the LCD backlight */
/// red LED 1 
PwmOut red1(p26);
/// red LED 2
PwmOut red2(p25); 
/// red LED 3
PwmOut red3(p24); 
/// yellow LED 
PwmOut yel(p23);
/// peizo buzzer 
PwmOut buzzer(p22);
///
/*!< inerrupting objects */
/// timer to count down the time for the lives 
Ticker count;
///timer to flash the pixel on a fewtimes a second 
Ticker flash;
/// button on the joystick 
InterruptIn butt(p17);
/// external button on the PCB 
InterruptIn push(p18);
/// timeout used for debouncing on buttons 
Timeout timeout;
/// ticker to regularly check the brightness level for backlight and LEDs
Ticker backlight; 
/// ticker to count up until the time limit is reached and then power down 
Ticker Power;
/// ticker to regularly read and update the possition of the joystick 
Ticker pollJoystick;
///
/*!< create enumerated type (0,1,2,3 etc. for the direction of the joystick)
can be extended to include diagonals etc. */
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    CENTRE,
    UNKNOWN
};
///
/*!< struct for Joystick */
typedef struct JoyStick Joystick;
struct JoyStick {
    float x;    /*!< current x value */
    float x0;   /*!< 'centred' x value */
    float y;    /*!< current y value */
    float y0;   /*!< 'centred' y value */
    int button; /*!< button state (assume pull-down used, so 1 = pressed, 0 = unpressed) */
    DirectionName direction;  /*!< current direction */
};
///
/*!< create a struct variable */
Joystick joystick;
///
/*!< variables to be initialised */
/// number of pixels that make up the horizontal component of the LCD 
int nx=84;
/// number of pixels that make up the vertical component of the LCD 
int ny=48;
/// horizontal coordinate of the pixel to be set 
int i;
/// vertical coordinate of the pixel to be set 
int j;
/// flag used to update the position of the players pixel 
int printFlag = 0;
/// flag to indicate that the goal area has been reached 
int winFlag=0;
/// number of attempts per level, as indicated with the red LEDs 
int life=3;
/// flag used to flash
int flashFlag=1;
/// flag used to keep track of the time taken to complete a level 
int countFlag=0;
/// flag for the start menu 
int startFlag=0;
/// various toggleable sub menu flags 
int l,r,u,d,c,b;
/// default brightness level out of ten for the red LEDs and backlight 
int bright=5;
/// flag used to load the different levels depending on the value 
int mazeFlag=1;
/// flag used to toggle the screen invertion 
int screen;
/// the score for the first level, highest value saved in flash memory 
int score1=0;
/// score for the second level, highest saved in flash memory 
int score2=0;
/// score for the third level, highest saved in flash memory 
int score3=0;
/// counts up to power down the device if nothing happens for a set time 
int power=0;
///
/*!< function prototypes */
void calibrateJoystick();
/*!< sets the default position of the joystick
when the device is initialised by checking the
rest position of the joystick potentiometers
@param joystick.x0
@param joystick.y0
@return xPot - inital position from 0.0 to 1.0 (0.5 if exactly centred)
@return yPot - inital position from 0.0 to 1.0 (0.5 if exactly centred)
*/

void updateJoystick();
/*!< checks the possiton fo the joystick by
comparing the two potentiometers against the initial values
and setting the relevant direction
@param joystick.x - current joystick position relative to initial (-0.5 ~ 0.5, 0.0 at centre)
@param joystick.y - current joystick position relative to initial (-0.5 ~ 0.5, 0.0 at centre)
@param DIRECTION_TOLERANCE
@return CENTRE - if neither POT is greater than the tolerance
@return DOWN - if the Y POT is possitive of the tolerance
@return UP if the Y POT is negative of the tolerance
@return RIGHT if the X POT is positive of the tolerance
@return LEFT if the X POT is negative of the tolerance 
*/

void move();
/*!< checks which direction the joystick is pointing and
update the position of the players pixel accordingly while
clearing off where the player has just been
@param DOWN - joystick direction down
@param UP - joystick direction up
@param LEFT - joystick direction left
@param RIGHT - joystick direction right
@return i++/-- ~ depending on the screen orientation and joystick direction
@return j++/-- ~ depending on the screen orientation and joystick direction
*/

void endMove();
/*!< the same as move but doesn't clear off where the pixel
has already been, the path remains, like an etch-a-sketch
@param DOWN - joystick direction down
@param UP - joystick direction up
@param LEFT - joystick direction left
@param RIGHT - joystick direction right
@return i++/-- ~ depending on the screen orientation and joystick direction
@return j++/-- ~ depending on the screen orientation and joystick direction
*/

void maze1();
/*!< checks the value of the mazeFlag and opens the
coresponding BMP file saved to flash memory
@param mazeFlag - flag to select which image file to load
@return img - name given to the file being loaded
@return TEST.BMP - name of the file to open for the first level
@return MAZE1.BMP - name of the file to open for the second level
@return MAZE2.BMP - name of the file to open for the third level
*/

void border();
/*!<
if the pixel is outide the range of the screen it is
moved back in range, essentially a hard wall boundary
preventing the pixel moving off the edge of the LCD
@param i - horizontal coordinate
@param j - vertical coordinate
@param nx - horizontal screen component
@param ny - vertical screen component
@return j - stops at top edge and bottom edge
@return i - stops at left edge and right edge 
*/

void endBorder();
/*!< same as the border, only for the mini game,
allowing the player to warp from one side of the LCD
to the other when they go off the edge
@param i - horizontal coordinate
@param j - vertical coordinate
@param nx - horizontal screen component
@param ny - vertical screen component
@return j - warps from top edge to bottom edge and vice versa
@return i - warps from left edge to right edge and vice versa 
*/

void winning();
/*!< checks to see if the goal area has been reached,
resets the player pixel to the start position
and updates the winFlag
@param j - vertical coordinate
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more
@return i - resets to the start position if the goal area (j==0) is reached
@return j - resets to the start position if the goal area (j==0) is reached
@return winFlag - sets the flag to signify the goal has been reached
*/

void clear();
/*!< clears off the pixel and then the rest of the LCD
and refreshes the screen 
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more
@return lcd.clear() - clears the LCD: see N5110 for more
@return lcd.refresh() - updates the CLD: see N5110 for more
*/

void lives();
/*!< checks the number of remaining lives and displays
them on the red LEDs, returning to the menu when no
lives remain
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more
@return lcd.clear() - clears the whole LCD: see N5110 for more
@return lcd.refresh() - updates the LCD: see N5110 for more 
*/

void counter();
/*!< called once a second to count up the time and remove a
life when the time runs out for that level, also warns player
with audio visual feedback on buzzer and yellow LED.
@param i - horizontal coordinate signifying that the player is in the safe zone
@param startFlag - if the player goes back to the main menu
@param winFlag - if the goal area has been reached the counter stops
@param mazeFlag - different time limits depending on which level is being played
@param countFlag - the number of increments that have passed
@return red1,red2,red3 - turns the red LEDs off if the player returns to the main menu
@return yel - turns off if the player returns to the main menu, flashes on and off if
the player is in the safe zone, or remains alight if the player is runing out of time
@return buzzer - plays a continuous note if the player is running out of time
@return life-- ~ takes one life away every time the player runs out of time
@return lcd.clearPixel(i,j) - clears the current pixel when the time is up: see N5110 for more
@return i - resets to the start position when time runs out
@return j - resets to the start position when time runs out 
*/

void Flash();
/*!< attached to flash ticker, toggles the flag between 1 and 0 and sets the pixel to flash
@return flashflag=!flashFlag - toggles the flag each time the function is called
@return lcd.setPixel(i,j) - sets the pixel: see N5110 for more 
*/

void Initialise();
/*!< all of the variables, flags, tickers, and anything else that needs to be set when the device first powers up
@return lcd.init() - initialises the LCD: see N5110 for more
@return Power.attach(&powerDown,1.0) - calls the powerDown() function once a second
@return backlight.attach(&brightness,1.0/10.0) - calls the brightness() function ten times a second
@return pollJoystick.attach(&updateJoystick,1.0/10.0) - calls the updateJoystick() function ten times a second
@return i=1 - sets the horizontal starting coordinate
@return j=24 - sets the vertical starting coordinate
@return life=3 - sets the maximum number of lives/attempts
@return winFlag=0 - sets the flag signifying the gaol area has yet to be reached
@return startFlag=1 - flag used in the main menu
@return buzzer.period(0.002)- sets the frequency at which PWM is sent to the buzzer
@return c=1 - flag used to set a while loop
*/

void mainGame();
/*!< contains all of the relevant loops/checks/functions that make up the main game
@return count.attach(&counter,1.0) - calls the counter() function once a second
@return flash.attach(&Flash,0.25) - calls the Flash() function four times a second
@return push.rise(&pause) - links the external button to the pause() function
@return butt.rise(&pause) - links the joystick button to the pause() function
@return life=3 - sets the maximum number of lives, ensures the number of lives gets reset when the game is restarted
@return startFlag=0 - clears the flag used in the main menu
@return winFlag=0 - resests the flag so the game can be played
@return clear() - calls the clear() function
@return maze1() - calls the maze1() function
@return i=1 - resets the horizontal starting coordinate
@return j=24 - resets the vertical starting coordinate
@param life - game can only be played if there are lives/attempts remaining
@param !winFlag - game can only be played if it hasn't been completed this round
@return power=0 - the device will not power down while the game is being played (while in the loop set by life&&!winFlag)
@return border() - calls the border() function (while in the loop set by life&&!winFlag)
@param printFlag - used to print determine when the pixel position is to be udated
@return printFlag=0 - resets the printFlag (in the if statement set by printFlag)
@return move() - calls the move() function (in the if statement set by printFlag)
@return collide1() - calls the collide1() function (while in the loop set by life&&!winFlag)
@return winning() - calls the winning() function (while in the loop set by life&&!winFlag)
@return lives() - calls the lives() function (while in the loop set by life&&!winFlag)
@param flashFlag - flag used to flash the player pixel
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (in the if statement set by flashFlag)
@param !flashFlag||joystick.direction!=CENTRE - whether the flashFlag isn't current active or the joystick is being moved
@return lcd.setPixel(i,j) - sets the current pixel: see N5110 for more (in the if statement set by !flashFlag||joystick.direction!=CENTRE)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in th eloop set by life&&!winFlag)
@param mazeFlag - the flag number allocated to the level
@return score#=mazeFlag*life*(100-countFlag) - sets the score to be written to flash memory depending on the level that was completed, the number of remaining lives/attempts, and the time taken to complete the level (# indicates that a number variable is attached to the statement)
@param screen - the flag used to indicate whether the screen is in normal or inverse mode
@return score#=score#+50 - adds 50 to the score of the level if the screen is inverted as this makes adds to the dificulty (# indicates that a number variable is attached to the statement)
@return int Score# - tells the device that the value to be read is an integer value (# indicates that a number variable is attached to the statement): see scoreRead1() for more
@return FILE *fp# = fopen("/local/score#.txt","r") - opens the file score#.txt saved to the flash memory as the vaiable fp# (# indicates that a number variable is attached to the statement): see scoreRead() for more
@return fscanf(fp#,"%i",&Score#) - searches the file for an integer value (# indicates that a number variable is attached to the statement): see scoreRead() for more
@return fclose(fp# - closes the file (# indicates that a number variable is attached to the statement): see scoreRead() for more
@param score#>Score# - compares the score that was just set to the score on the file to see if it's greater (# indicates that a number variable is attached to the statement)
@return scoreWrite#() - calls the scoreWrite#() function if the latest score is larger than the one on file (# indicates that a number variable is attached to the statement)
@return clear() - calls the clear() function (at the end of the mainGame() code) 
*/

void endGame();
/*!< the functions/variables/loops that make up the mini game
@return flash.attach(&Flash,0.25) - calls the Flash() function four times a second
@return red#=0 - sets the coresponding red LED to zero (# indicates that a number variable is attached to the statement)
@return yel=0 - sets the coresponding yellow LED to zero
@return buzzer=0 - turns off the buzzer
@return push.rise(&menus) - links the external button to the menus() function
@return butt.rise(&clear) - links the joystick button to the clear() function
@return lcd.clear() - clears the LCD: see N5110 for more
@return i=42 - sets the new horizontal starting coordinate
@return j=24 - sets the vertical starting coordinate
@param b - flag used to se a loop
@return power=0 - the device will not power down while the mini game is being played (while in the loop set by b)
@return endBorder() - calls the endBorder() function (while in the loop set by b)
@return lcd.setPixel - sets the current pixel: see N5110 for more (while in the loop set by b)
@param flashFlag - flag used to flash the player pixel
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (in the if statement set by flashFlag)
@param !flashFlag||joystick.direction!=CENTRE - whether the flashFlag isn't current active or the joystick is being moved
@return lcd.setPixel(i,j) - sets the current pixel: see N5110 for more (in the if statement set by !flashFlag||joystick.direction!=CENTRE)
@param printFlag - used to print determine when the pixel position is to be udated
@return printFlag=0 - resets the printFlag (in the if statement set by printFlag)
@return endMove() - calls the endMove() function (in the if statement set by printFlag)
@return lcd.refresh() - updates the LCD: see N5110 for more (in the if statement set by printFlag)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (outside the main loop on the endGame() function)
@return clear() - calls the clear() function (outside the main loop of the endGame() function)
@return lcd.refresh() - updates the LCD: see N5110 for more (outside the main loop of the endGame() function)
@return i=1 - resets the horizontal starting coordinate (outside the main loop of the endGame() function)
@return j=24 - resest the vertical starting coordinate (outside the main loop of the endGame() function) 
*/

void start();
/*!< all of the variable/loops/functions that make up the main menu
@return power=0 - restarts the counter, the device WILL power down if left in the menu for too long
@return clear() - calls the clear() function
@return lcd.refresh() - updates the LCD: see N5110 for more
@return butt.rise(&menus) - links the joystick button to the menus() function
@return push.rise(&menus) - links the external button to the menus() function
@return startFlag=1 - flag used to set a loop
@param c - flag used to set the while loop
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by c)
@return lcd.printString("Click",25,2) - prints the character string Click to the LCD: see N5110 for more (while in the loop set by c)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by c)
@return clear() - calls the clear() function
@param startFlag - flag used to set the while loop
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by startFlag)
@param joystick.direction!=CENTRE - while the joystick isn't centred (while in the loop set by startFlag)
@return lcd.clearPixel(i,j) - clears the current pxel: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("^ Options ^",10,0) - prints the character string ^ Options ^ to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("< Score Inv. >",0,2) - prints the character string < Score Inv. > to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("v Levels v",12,5) - prints the character string v Levels v to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@param joystick.direction==CENTRE - if the joystick is centred (oustide the loop set by joystick.direction!=CENTRE and inside the loop set by startFlag)
@return lcd.clearPixel(i,j) - clears the current pxel: see N5110 for more (if in the statement set by joystick.direction==CENTRE)
@return lcd.printString("^ Options ^",10,0) - prints the character string ^ Options ^ to the LCD: see N5110 for more (if in the statement set by joystick.direction==CENTRE)
@return lcd.printString("< Score Inv. >",0,2) - prints the character string < Score Inv. > to the LCD: see N5110 for more (if in the statement set by joystick.direction==CENTRE)
@return lcd.printString("v Levels v",12,5) - prints the character string v Levels v to the LCD: see N5110 for more (if in the statement set by joystick.direction==CENTRE)
@return lcd.refresh() - updates the LCD: see N5110 for more (if in the statement set by joystick.direction==CENTRE)
@param joystick.direction==UP - if the joystick  is pointed up
@return power=0 - resets the power flag (if in the statement set by joystick.direction==UP)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==UP)
@return toggle() - calls the toggle() function (if in the statement set by joystick.direction==UP)
@return Options() - calls the Options() function (if in the statement set by joystick.direction==UP)
@param joystick.direction==DOWN - if the joystick is pointing down
@return power=0 - resets the power flag (if in the statement set by joystick.direction==DOWN)
@return lcd.clearPixel(i,j) - clears the current pixel: see N510 for more (if in the statement set by joystick.direction==DOWN)
@return toggle() - calls the toggle() function (if in the statement set by joystick.direction==DOWN)
@return Levels() - calls the Levels() function (if in the statement set by joystick.direction==DOWN)
@param joystick.direction==LEFT - if the joystick is pointing left
@return power=0 - resets the power flag (if in the statement set by joystick.direction==LEFT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==LEFT)
@return toggle() - calls the toggle() function (if in the statement set by joystick.direction==LEFT)
@return Score() - calls the Score() function (if in the statement set by joystick.direction==LEFT)
@param joystick.direction==RIGHT - if the joystick is pointing right
@return power=0 - resets the power flag (if in the statement set by joystick.direction==RIGHT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==RIGHT)
@return screen=!screen - toggles the flag used to se the screen mode (if in the statement set by joystick.direction==RIGHT)
@param screen - if the flag is set (if in the statement set by joystick.direction==RIGHT)
@return lcd.inverseMode() - sets the LCD to inverse mode: see N5110 for more (if in the statement set by screen)
@param !screen - if the flag is not set (if in the statement set by joystick.direction==RIGHT)
@return lcd.normalMode() - sets the LCD to normal mode: see N5110 for more (if in the statement set by !screen)
@return lcd.refresh() - updates the LCD: see N5110 for more (if in the statement set by joystick.direction==RIGHT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (outside the main loop for the start() function)
@return clear() - calls the clear() function (outside the main loop for the start() function) 
*/

void Options();
/*!< all of the variables/loops/functions that make up the options menu
@return power=0 - resets the power flag
@return lcd.ckearPixel(i,j) - clears the current pixel: see N5110 for more
@return clear() - calls the clear() function
@param u - flad used to set the while loop
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by u)
@param joystick.direction!=CENTRE - while the joystick is not centred (while in the loop set by u)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("~~~~~~~~~~~~~~",0,0) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("^ Brightness ^",0,1) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("< Reset",10,2) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("End >",30,3) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("v Back v",15,4) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("~~~~~~~~~~~~~~",0,5) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.directioon!=CENTRE)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by u)
@return lcd.printString("~~~~~~~~~~~~~~",0,0) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by u)
@return lcd.printString("^ Brightness ^",0,1) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by u)
@return lcd.printString("< Reset",10,2) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by u)
@return lcd.printString("End >",30,3) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by u)
@return lcd.printString("v Back v",15,4) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by u)
@return lcd.printString("~~~~~~~~~~~~~~",0,5) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by u)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by u)
@param joystick.direction==DOWN - if the joystick is pointing down (while in the loop set by u)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==DOWN)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==DOWN)
@return start() - calls the start() function to exit back to the main menu (if in the statement set by joystick.direction==DOWN)
@param joystick.direction==UP - if the joystick is pointing up (while in the loop set by u)
@return power=0 - reset the power flag (if in the statement set by joystick.diretion==UP)
@return lcd.ClearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==UP)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==UP)
@param b - flag used to set the loop (if in the statement set by joystick.direction==UP)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by b)
@return red#=0.1*bright - sets the red LEDs to 10% the value of bright (the # indicates that a number value is atached to the statement) (while in the loop set by b)
@return yel=0.1*bright - sets the yellow LED to 10% the value of bright (while in the loop set by b)
@return push.rise(&Dark) - links the external button to the Dark() function (while in the loop set by b)
@return butt.rise(&Light) - links the joystick button to the Light() function (while in the loop set by b)
@return lcd.printString("click joy - up",0,1) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by b)
@return lcd.printString("or pink - down",0,3) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by b)
@return lcd.printString("v Back v",15,5) - prints the character string inside the "" to the LCD: see N5110 for more (while in the loop set by b)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by b)
@param joystick.direction==DOWN - if the joystick is pointing down (while in the loop set by b)
@return Options() - calls the Options() function to escape back that menu (if in the statement set by joystick.direction==DOWN)
@param joystick.direction==LEFT - if the joystick is pointing left (while in the loop set by u)
@return power=0 - reset the power flag (if in the statement set by joystick.direction==LEFT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==LEFT)
@return toggle() - calls the toggle() function (if in the statement set by joystick.direction==LEFT)
@return score#=0 - sets the score for the relevant level to 0 (the # indicates that a number value is attached to the statement) (if in the statement set by joystick.direction==LEFT)
@return scoreWrite#() - calls the scoreWrite#() function relevant to the level (the # indicates that a number value is attached to the statement): see scoreWrite1() for more (if in the statement set by joystick.direction==LEFT)
@param joystick.direction==RIGHT - if the joystick is pointing right (while in the loop set by u)
@return power=0 - reset the power flag (if in the statement set by joystick.direction==RIGHT)
@return lcd.clearPixe(i,j) - clears the current pixel: see N5110 for more ( if inthe statement set by joystick.direction==RIGHT)
@return winFlag=1 - sets the winFlag, equivalent of reaching the goal state (if in the statement set by joystick.direction==RIGHT)
@return endGame() - calls the endGame() function to play the mini game (if in the statement set by joystick.direction==RIGHT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (outside the main loop for the Options() function)
@return clear() - calls the clear() function (outside the main look of the Options() function)
*/

void Levels();
/*!< all of the variables/loops/functions that make up the level select menu
@return power=0 - resets the power flag
@return push.rise(&menus) - links the external button to the menus() function
@return butt.rise(&menus) - links hte joystick button to the menus() funcion
@return clear() - calls the clear() function
@param d - flag used to set the loop
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by d)
@param joystick.direction!=CENTRE - while the joystick isn't centred (while in the loop set by d)
@return lcd.printString("~~~~~~~~~~~~~~",0,0) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("Left = 1",10,2) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("Down = 2",10,3) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("Right = 3",10,4) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("^ Back ^",15,1) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("~~~~~~~~~~~~~~",0,5) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("~~~~~~~~~~~~~~",0,0) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set d)
@return lcd.printString("Left = 1",10,2) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by d)
@return lcd.printString("Down = 2",10,3) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by d)
@return lcd.printString("Right = 3",10,4) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by d)
@return lcd.printString("^ Back ^",15,1) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by d)
@return lcd.printString("~~~~~~~~~~~~~~",0,5) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by d)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by d)
@param joystick.direction==UP - if the joystick is pointing up (while in the loop set by d)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==UP)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==UP)
@return toggle() - calls the togle() function (if in the statement set by joystick.direction==UP)
@return start() - calls the start() function to escape back to the menu (if in the statement set by joystick.direction==UP)
@param joystick.direction==LEFT - if the joystick is pointing left (while in the loop set by d)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==LEFT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==LEFT)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==LEFT)
@param joystick.direction==LEFT - while the joystick is pointing left (if in the statement set by joystick.direction==LEFT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by joystick.direction==LEFT)
@return lcd.printString("Level 1",15,2) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction==LEFT)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction==LEFT)
@return mazeFlag=1 - sets the mazeFlag to 1 to load the first level (while in the loop set by joystick.direction==LEFT)
@return clear() - calls the clear(0 function (if in the statement set by joystick.direction==LEFT)
@return mainGame() - calls the mainGame() function (if in the statement set by joystick.direction==LEFT)
@param joystick.direction==DOWN - if the joystick is pointing down (while in the loop set by d)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==DOWN)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==DOWN)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==DOWN)
@param joystick.direction==DOWN - while the joystick is pointing down (if in the statement set by joystick.direction==DOWN)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by joystick.direction==DOWN)
@return lcd.printString("Level 2",15,2) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction==DOWN)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction==DOWN)
@return mazeFlag=2 - sets the mazeFlag to 2 to load the second level (while in the loop set by joystick.direction==DOWN)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==DOWN)
@return mainGame() - calls the mainGame() function (if in the statement set by joystick.direction==Down)
@param joystick.direction==RIGHT - if the joystick is pointing right (whiel in the loop set by d)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==RIGHT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==RIGHT)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==RIGHT)
@param joystick.direction==RIGHT - while the joystick is pointing right (if in the statement set by joystick.direction==RIGHT)
@return power=0 - resets the power flag (while in the loop set by joystick.direction==RIGHT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by joystick.direction==RIGHT)
@return lcd.printString("Level 3",15,2) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction==RIGHT)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction==RIGHT)
@return mazeFlag=3 - sets the mazeFlag to 3 to load the third level (while in the loop set by joystick.direction==RIGHT)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==RIGHT)
@return mainGame() - calls the mainGame() function (if in the statement set by joystick.direction==RIGHT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (outside the main loop for the Levels() function)
@return clear() - calls the clear() function (outside the main loop for the Levels() function)
*/

void Score();
/*!< all of the variables/loops/functiond that make up the high scores menu
@return power=0 - resets the power flag
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more
@return push.rise(&menus) - links the external button to the menus() function
@return butt.rise(&menus) - links the joystick button to the menus() function
@return clear() - calls the clear() function
@param l - flag used to set the loop
@return lcd.clearPixel(i,j) - clears the current pixel; see N5110 for more (while in the loop set by l)
@param joystick.direction!=CENTRE - while the joystick isn't centred (while in the loop set by l)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("~~~~~~~~~~~~~~",0,0) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("High Score",10,1) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("> Back >",15,4) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.printString("~~~~~~~~~~~~~~",0,5) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction!=CENTRE)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by l)
@return lcd.printString("~~~~~~~~~~~~~~",0,0) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by l)
@return lcd.printString("High Score",10,1) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by l)
@return lcd.printString("> Back >",15,4) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by l)
@return lcd.printString("~~~~~~~~~~~~~~",0,5) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by l)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by l)
@param joystick.direction==RIGHT - if the joystick is pointing right (while in the loop set by l)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==RIGHT)
@return start() - calls the start() function  to escape back to the menu (if in the statement set by joystick.direction==RIGHT)
@param joystick.direction==UP - if the joystick is pointing up (while in the loop set by l)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==UP)
@return lcd.clearPixel(i,j0 - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==UP)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==UP)
@param joystick.direction!=DOWN - while the joystick isn't pointing down (if in the statement set by joystick.direction==UP)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by joystick.direction!=DOWN)
@return lcd.printString("Level 1 best",7,1) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=DOWN)
@return scoreRead1() - calls the function scoreRead1() (while in the loop set by joystick.direction!=DOWN)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction!=DOWN)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==UP)
@param joystick.direction==LEFT - if the joystick is pointing left (while in the loop set by l)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==LEFT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==LEFT)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==LEFT)
@param joystick.direction!=RIGHT - while the joystick isn't pointing right (if in the statement set by joystick.direction==LEFT)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by joysick.direction!=RIGHT)
@return lcd.printString("Level 2 best",7,1) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=RIGHT)
@return scoreRead2() - calls the function scoreRead2() (while in the loop set by joystick.direction!=RIGHT)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction!=RIGHT)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==LEFT)
@param joystick.direction==DOWN - if the joystick is pointing down (while in the loop set by l)
@return power=0 - resets the power flag (if in the statement set by joystick.direction==DOWN)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by joystick.direction==DOWN)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==DOWN)
@param joystick.direction!=UP - while the joystick isn't pointing up (if in the statement set by joystick.direction==DOWN)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (while in the loop set by joystick.direction!=UP)
@return lcd.printString("Level 3 best",7,1) - prints the character string between the "" to the LCD: see N5110 for more (while in the loop set by joystick.direction!=UP)
@return scoreRead3() - calls the scoreRead3() function (while in the loop seet by joystick.direction!=UP)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the loop set by joystick.direction!=UP)
@return clear() - calls the clear() function (if in the statement set by joystick.direction==DOWN)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (outside the main loop for the Score() function)
@return clear() - calls the clear() fucntion (outside the main loop for the Score() function)
*/

void menus();
/*!< essentially debouncing code calling the toggle() function, similar to the Light() and Dark() functions
@return power=0 - resets the power flag
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more
@return clear() - calls the clear() function
@return timeout.attach_us(&toggle,10) - delays the calling of the toggle() function by 10 microseconds
*/

void toggle();
/*!< the function called to toggle/set/clear various flags
@return power=0 - resets the power flag
@return l=!l - toggles the l flag
@return r=!r - toggles the r flag
@return u=!u - toggles the u flag
@return d=!d - toggles the d flag
@return c=0 - clears the c flag
@return b=!b - toggles the b flag
@return startFlag=1 - sets the startFlag
*/

void light();
/*!< the function called to increase the LED and backlight brightness
@return bright++ - adds to the bright variable
@param bright>10 - if the variable is greater than 10, so the LEDs and backlight don't appear too intense
@return bright=10 - the bright variable is equal to 10 (if in the statement set by bright>10)
*/

void dark();
/*!< the function called to decrease the LED and backlight brightness
@return bright-- ~ subtracts from the bright variable
@param bright<1 - if the variable is less than 1, so the LEDs and backlight don't appear too dim
@return bright=1 - the bright variable is equal to 1 (if in the statement set by bright<1)
*/

void Light();
/*!< debouncing code used to ensure the corresponding code is only called once per button press
@return timeout.attach_us(&light,100) - delays the calling of the light() function by 100 microseconds
*/

void Dark();
/*!< debouncing code used to ensure the corresponding code is only called once per button press
@return timeout.attach_us(&dark,100) - delays the calling of the dark() function by 100 microseconds
*/

void brightness();
/*!< function called to set the backlight brightness of the LCD
@return lcd.setBrightness(0.1*bright) - sets the backlight brightness to ten percent of the variable bright, 10% brightness being the lowest and 100% the highest: see N5110 for more
*/

void All();
/*!< all of the variables/loops/functions that make up the entire game, used to decrease the size of the main
@return calibrateJoystick() - calls the calibrateJoystick() function to get the cnetres values of the joystick
@return bright=5 - sets the variable bright to 5, the default/initial brightness of the LEDs and backlight is 50%
@param 1 - sets the game to run in an infinite loop
@param screen - if the flag is set (while in the infinite loop)
@return lcd.inverseMode() - sets the LCD to inverse mode: see N5110 for more (if in the statement set by screen)
@param !screen - if the flag is not set (while in the infinite loop)
@return lcd.normalMode() - sets the LCD to normal mode: see N5110 for more (if in the statement set by !screen)
@return lcd.refresh() - updates the LCD: see N5110 for more (while in the infinite loop)
@return Initialise() - calls the Initialise() function (while in the infinite loop)
@return start() - calls the start() function (while in the infinite loop)
*/

void collide1();
/*!< 
@return int block=0 - variable used to detect a collision
@param joystick.direction==UP - if the josytick is pointing up
@param lcd.getPixel(i,j)&&lcd.getPixel(i,j-1) - if the current pixel and the pixel above it are active: see N5110 for more (if in the statement set by joystick.direction==UP)
@return block++ ~ adds to the block variable (if in the statement set by lcd.getPixel(i,j)&&lcd.getPixel(i,j-1))
@param lcd.getPixel(i+1,j)||lcd.getPixel(i-1,j) - if the pixels to the left or right of the current pixel are active: see N5110 for more (if in the statement set by joystick.direction==UP)
@return block++ ~ adds to the block variable (if in the statement set by lcd.getPixel(i+1,j)||lcd.getPixel(i-1,j))
@param joystick.direction==DOWN - if the joystick is pointing down 
@param lcd.getPixel(i,j)&&lcd.getPixel(i,j+1) - if the current pixel and the pixel beneath it are active: see N5110 for more (if in the statement set by joystick.direction==DOWN)
@return block++ ~ adds to the block variable (if in the statement set by lcd.getPixel(i,j)&&lcd.getPixel(i,j+1))
@param lcd.getPixel(i+1,j)||lcd.getPixel(i-1,j) - if the pixels to the left or right of the current pixel are active: see N5110 for more (if in the statement set by joystick.direction==DOWN)
@return block++ ~ adds to the block variable (if in the statement set by lcd.getPixel(i+1,j)||lcd.getPixel(i-1,j))
@param joystick.direction==LEFT - if the joystick is pointing left
@param lcd.getPixel(i,j)&&lcd.getPixel(i-1,j) - if the current pixel and the one to the left are active: see N5110 for more (if in the statement set by joystick.direction==LEFT)
@return block++ ~ adds to the block variable (if in the statement set by lcd.getPixel(i,j)&&lcd.getPixel(i-1,j))
@param lcd.getPixel(i,j+1)||lcd.getPixel(i,j-1) - if the pixels above or below the current pixel are active: see N5110 for more (if in the statement set by joystick.direction==LEFT)
@return block++ ~ adds to the block variable (if in the statement set by lcd.getPixel(i,j+1)||lcd.getPixel(i,j-1))
@param joystick.direction==RIGHT - if the joystick is pointing right
@param lcd.getPixel(i,j)&&lcd.getPixel(i+1,j) - if the current pixel and the pixel to the right are active: see N5110 for more (if in the statement set by joystick.direction==RIGHT)
@return block++ ~ adds to the block variable (if in the statement set by lcd.getPixel(i,j)&&lcd.getPixel(i+1,j))
@param lcd.getPixel(i,j+1)||lcd.getPixel(i,j-1) - if the pixels above or below the current pixel are active: see N5110 for more (if in the statement set by joystick.direction==RIGHT)
@return block++ ~ adds to the block variable (if in the statement set by lcd.getPixel(i,j+1)||lcd.getPixel(i,j-1))
@param block==2 - if the block variable is equal to 2
@param !(i==0||i==1||i==2||i==3) - if the current pixel is outside the safe zone (if in the statement set by block==2)
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by !(i==0||i==1||i==2||i==3))
@return clear() - calls the clear() function (if in the statement set by !(i==0||i==1||i==2||i==3))
@return maze1() - calls the maze1() function (if in the statement set by !(i==0||i==1||i==2||i==3))
@return life-- ~ subtracts one from the life counter, the player dies if they touch a wall (if in the statement set by !(i==0||i==1||i==2||i==3))
@return i=1 - resets the horizontal starting coordinate (if in the statement set by !(i==0||i==1||i==2||i==3))
@return j=24 - resets the vertical starting coordinate (if in the statement set by !(i==0||i==1||i==2||i==3))
@return countFlag=0 - resets the countFlag (if in the statement set by !(i==0||i==1||i==2||i==3))
@param else - if the previous parameters weren't met
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more (if in the statement set by else)
@return clear() - calls the clear() function (if in the statement set by else)
@return maze1() - calls the maze1() function (if in the statement set by else)
@return i=1 - resets the horizontal starting coordinate (if in the statement set by else)
@return j=24 - resets the vertical starting coordinate (if in the statement set by else)
@return countFlag=0 - resets the countFlag (if in the statement set by else)
*/

void pause();
/*!< function called to end the game and return to the menu
@return power=0 - resets the power flag
@return lcd.clearPixel(i,j) - clears the current pixel: see N5110 for more
@return i=1 - resets the horizontal starting component
@return j=24 - resets the vertical starting component
@return winFlag=0 - clears the winFlag
@return life=0 - clears all remaining lives/attempts
*/

void scoreWrite1();
/*!< function called to record the score for the first level
@return FILE *fp#= fopen("/local/score#.txt","w") - opens the file score#.txt in the variable fp#, if the file doesn't exist it creates it (the # indicates that a number value may be attached to the statement)
@return fprintf(fp#,"%i",score#) - prints the value of score# as an integer to the file
@return fclose(fp#) - closes the file
*/

void scoreRead1();
/*!< function called to read the current saved value for the score on the first level
@return int Score# - tells the device that the value to be read is an integer value (# indicates that a number variable is attached to the statement)
@return FILE *fp# = fopen("/local/score#.txt","r") - opens the file score#.txt saved to the flash memory as the vaiable fp# (# indicates that a number variable is attached to the statement)
@return fscanf(fp#,"%i",&Score#) - searches the file for an integer value (# indicates that a number variable is attached to the statement)
@return fclose(fp# - closes the file (# indicates that a number variable is attached to the statement)
@return char Points#[14] - buffer to store the ponits value (# inidicates that a number variable is attached to the statement)
@return sprintf(Points#,"%i",Score#) - sends the score to the buffer (# indicates that a number variable is attached to the statement)
@return lcd.printString(Points#,30,3) - prints the contents of the Points# buffer to the LCD: see N5110 for more
*/

void scoreWrite2();
/*!< function called to record the score for the second level: see scoreWrite1() for more */

void scoreRead2();
/*!< function called to read the current saved value for the score on the second level: see scoreRead1() for more */

void scoreWrite3();
/*!< function called to record the score for the third level: see scoreWrite1() for more*/

void scoreRead3();
/*!< function called to read the current saved value for the score on the third level: see scoreRead1() for more */

void powerDown();
/*!< function that puts the device into a low power consumption state if nothing happens after a set time
@return power++ ~ adds to the power flag
@param power==55 - if the power flag is equal to 55, when 55 seconds have gone by without the flag being reset
@return red1=0 - turns off the LED red1 (if in the statement set by power==55)
@return red2=0.1*bright - sets the LED red2 to 10% of the variable bright (if in the statement set by power==55)
@return red3=0 - turns off the LED red3 (if in the statement set by power==55)
@return yel==0.1*bright - sets the LED yel to 10% of the variabel bright (if in the statement set by power==55)
@return buzzer=0.4 - sets the PWM for the buzzer to 40% (if in the statement set by power==55)
@param power==56 - if the power flag is equal to 56, when 56 seconds have gone by without the flag being reset
@return red1=0.1*bright - sets the LED red1 to 10% of the variable bright (if in the statement set by power==56)
@return red2=0 - turns off the LED red2 (if in the statement set by power==56)
@return red3=0.1*bright - sets the LED red3 to 10% of the variable bright (if in the statement set by power==56)
@return yel=0 - turns off the LED yel (if in the statement set by power==56)
@return buzzer=0.9 - sets the PWM for the buzzer to 90% (if in the statement set by power==56)
@param power==57 - if the power flag is equal to 57, when 57 seconds have gone by without the flag being reset
@return the same as when power==55 (if in the statement set by power==57)
@param power==58 - if the power flag is equal to 58, when 58 seconds have gone by without the flag being reset
@return the same as when the power==56 (if in the statement set by power==58)
@param power==59 - if the power flag is equal to 59, when 57 seconds have gone by without the flag being reset
@return the same as when power==55 (if in the statement set by power==59)
@param power>=60 - if the power flag is equal to or greater than 60, when at least 60 seconds have gone by without the flag being reset
@return the same as when the power==56 (if in the statement set by power>=60)
@return DeepPowerDown() - calls the function DeepPowerDown() : see PowerControl for more (if in the statement set by power>=60)
*/

#endif