/** mDot_TTN_Node - Simple mDot temperature sensor using Dallas Semiconductors DS18B20 OneWire temperature sensor.
 * It used the AUTO_OTA join mode with parameters for The Things Network.
 *
 *
 * Uses MultiTech mDot developer board http://www.multitech.com/models/94558010LF
 * Requires a MultiTech MultiConnect Conduit http://www.multitech.com/models/94557203LF
 * http://www.multitech.net/developer/software/lora/conduit-mlinux-convert-to-basic-packet-forwarder/
 * http://forum.thethingsnetwork.org/t/setting-up-multitech-conduit-gateway-for-ttn/216/35
 *
 * Register a device and generate a random AppKey for the currently used application Id:
 * (You need to use your own device IDs, the ones shown here are examples only)
 *
 *./ttnctl devices register 0080000000000000
 * INFO Generating random AppKey...
 * INFO Registered device                        AppKey=000102030405060708090A0B0C0D0E0F DevEUI=0080000000000000
 *
 * or to specify the same AppKey for a new device or to reregister the same device again:
 *
 * ./ttnctl devices register 0080000000000000 000102030405060708090A0B0C0D0E0F
 *
 * ./ttnctl devices info 0080000000000000
 * Dynamic device:
 *
 *  AppEUI:  70B3D50000000000
 *           {0x70, 0xB3, 0xD5, 0x00, 0x00, 0x00, 0x00, 0x00}
 *
 *  DevEUI:  0080000000000000
 *           {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
 *
 *  AppKey:  000102030405060708090A0B0C0D0E0F
 *           {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F}
 *
 *
 * Copy the AppEUI and AppKey values provided in hex array notation above to the AppEUI and AppKey parameters below.
 *
 */

#include "mbed.h"
#include "DS1820.h"
//#include "BME280.h"
#include "mDot.h"
#include "MTSLog.h"
#include "MTSText.h"
#include <string>
#include <vector>
#include "MMA845x.h"
#include "GPSPARSER.h"
#include "DOGS102.h"
#include "font_6x8.h"

using namespace mts;

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))

#define NUMSAMPLES 1000



typedef struct {
    int lat_d;
    int lat_m;
    int lat_s;
    int lng_d;
    int lng_m;
    int lng_s;
    int x;
    int y;
    int z;
} Coordinate;

// AppEUI
uint8_t AppEUI[8]= {0x02, 0x00, 0x00, 0x00, 0x00, 0xEE, 0xFF, 0xC0};
// AppKey
uint8_t AppKey[16]= {0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16};

// Some defines for the LoRa configuration
/*
 * EU868 Datarates
 * ---------------
 * DR0 - SF12BW125
 * DR1 - SF11BW125
 * DR2 - SF10BW125
 * DR3 - SF9BW125
 * DR4 - SF8BW125
 * DR5 - SF7BW125
 * DR6 - SF7BW250
 *
 * US915 Datarates
 * ---------------
 * DR0 - SF10BW125
 * DR1 - SF9BW125
 * DR2 - SF8BW125
 * DR3 - SF7BW125
 * DR4 - SF8BW500
 */
#define LORA_SF mDot::DR4
#define LORA_ACK 0
#define LORA_TXPOWER 20

// Ignoring sub band for EU modules.
static uint8_t config_frequency_sub_band = 1;
volatile int stoppls = 0;

SPI lcd_spi(SPI1_MOSI, SPI1_MISO, SPI1_SCK);
DigitalOut lcd_spi_cs(SPI1_CS, 1);
DigitalOut lcd_cd(XBEE_ON_SLEEP, 1);

// Serial via USB for debugging only
Serial pc(USBTX,USBRX);

MMA845x_DATA accldata;

void gpsthread(void const *args)
{
    while(1) {
        GPSPARSER *instance = (GPSPARSER*)args;
        instance->readNemaSentence();
    }
}

void acclthread(void const *args) {
    
    /* Create global i2c handle */
    I2C i2chandle = I2C(PC_9, PA_8);

    MMA845x instance = MMA845x(i2chandle, MMA845x::SA0_VSS);

    instance.setCommonParameters(MMA845x::RANGE_8g,MMA845x::RES_MAX,MMA845x::LN_OFF,
                                   MMA845x::DR_50,MMA845x::OS_NORMAL,MMA845x::HPF_OFF );

    instance.activeMode();

    /* Accelerometer testing code */
    while(1) {
        
        if (!stoppls) {
            int result = instance.getStatus();
            if((result & MMA845x::XYZDR) != 0 ) {
                accldata = instance.getXYZ();
    
                logInfo("Accl: %d %d %d\n\r", instance.getXYZ()._x, instance.getXYZ()._y, instance.getXYZ()._z);
            }
        }
        
    }
}

int main() {
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> send_data;
    std::vector<uint8_t> recv_data;
    std::vector<uint8_t> nwkId;
    std::vector<uint8_t> nwkKey;

    dot = mDot::getInstance();
    dot->setLogLevel(MTSLog::INFO_LEVEL);
    logInfo("Starting...\n\r");

    GPSPARSER* gps;
    MTSSerial gps_serial(XBEE_DOUT, XBEE_DIN, 256, 2048);
    gps = new GPSPARSER(&gps_serial);
    Thread gpst(gpsthread, gps);

    Thread acclt(acclthread);

    DOGS102* lcd = new DOGS102(lcd_spi, lcd_spi_cs, lcd_cd);
    
    DigitalIn button(PA_11);
    button.mode(PullUp);

#ifdef ENABLE_LORA
    logInfo("Checking Config");

    uint8_t *it = AppEUI;
    for (uint8_t i = 0; i<8; i++)
        nwkId.push_back((uint8_t) *it++);

    it = AppKey;
    for (uint8_t i = 0; i<16; i++)
        nwkKey.push_back((uint8_t) *it++);

    logInfo("Resetting Config");
    // reset to default config so we know what state we're in
    dot->resetConfig();

    // Set Spreading Factor, higher is lower data rate, smaller packets but longer range
    // Lower is higher data rate, larger packets and shorter range.
    logInfo("Set SF");
    if((ret = dot->setTxDataRate( LORA_SF )) != mDot::MDOT_OK) {
        logError("Failed to set SF %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set TxPower");
    if((ret = dot->setTxPower( LORA_TXPOWER )) != mDot::MDOT_OK) {
        logError("Failed to set Tx Power %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set Public mode");
    if((ret = dot->setPublicNetwork(true)) != mDot::MDOT_OK) {
        logError("failed to set Public Mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set AUTO_OTA Join mode");
    if((ret = dot->setJoinMode(mDot::AUTO_OTA)) != mDot::MDOT_OK) {
        logError("Failed to set AUTO_OTA Join Mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set Ack");
    // 1 retries on Ack, 0 to disable
    if((ret = dot->setAck( LORA_ACK)) != mDot::MDOT_OK) {
        logError("Failed to set Ack %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    // Library ignores the frequency sub band for 868MHz in EU
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        logError("Failed to set frequency sub band %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Set Network Id");
    if ((ret = dot->setNetworkId(nwkId)) != mDot::MDOT_OK) {
        logError("Failed to set Network Id %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    logInfo("Set Network Key");
    if ((ret = dot->setNetworkKey(nwkKey)) != mDot::MDOT_OK) {
        logError("Failed to set Network Id %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("Saving Config");
    // Save config
    if (! dot->saveConfig()) {
        logError("failed to save configuration");
    }

    pc.printf("Device ID {");
    std::vector<uint8_t> deviceId;
    deviceId = dot->getDeviceId();
    for (std::vector<uint8_t>::iterator it = deviceId.begin() ; it != deviceId.end(); ++it) {
        pc.printf("0x%2.2X",*it );
        pc.printf("%s", it != (deviceId.end() -1 ) ? ", " : " " );
    }
    pc.printf("}\r\n");

    std::vector<uint8_t> netId;
    pc.printf("Network Id/App EUI {");
    netId = dot->getNetworkId();
    for (std::vector<uint8_t>::iterator it = netId.begin() ; it != netId.end(); ++it) {
        pc.printf("0x%2.2X", *it );
        pc.printf("%s", it != (netId.end() -1 ) ? ", " : " " );
    }
    pc.printf("}\r\n");

    std::vector<uint8_t> netKey;
    pc.printf("Network Key/App Key {");
    netKey = dot->getNetworkKey();
    for (std::vector<uint8_t>::iterator it = netKey.begin() ; it != netKey.end(); ++it) {
        pc.printf("0x%2.2X", *it );
        pc.printf("%s", it != (netKey.end() -1 ) ? ", " : " " );
    }
    pc.printf("}\r\n");

    // Display LoRa parameters
    // Display label and values in different colours, show pretty values not numeric values where applicable
    /*
        pc.printf("Public Network: %s\r\n", (char*)(dot->getPublicNetwork() ? "Yes" : "No") );
        pc.printf("Frequency: %s\r\n", (char*)mDot::FrequencyBandStr(dot->getFrequencyBand()).c_str() );
        pc.printf("Sub Band: %s\r\n", (char*)mDot::FrequencySubBandStr(dot->getFrequencySubBand()).c_str() );
        pc.printf("Join Mode: %s\r\n", (char*)mDot::JoinModeStr(dot->getJoinMode()).c_str() );
        pc.printf("Join Retries: %d\r\n", dot->getJoinRetries() );
        pc.printf("Join Byte Order: %s\r\n", (char*)(dot->getJoinByteOrder() == 0 ? "LSB" : "MSB") );
        pc.printf("Link Check Count: %d\r\n", dot->getLinkCheckCount() );
        pc.printf("Link Check Thold: %d\r\n", dot->getLinkCheckThreshold() );
        pc.printf("Tx Data Rate: %s\r\n", (char*)mDot::DataRateStr(dot->getTxDataRate()).c_str() );
        pc.printf("Tx Power: %d\r\n", dot->getTxPower() );
        pc.printf("TxWait: %s, ", (dot->getTxWait() ? "Y" : "N" ));
        pc.printf("CRC: %s, ", (dot->getCrc() ? "Y" : "N") );
        pc.printf("Ack: %s\r\n", (dot->getAck() ? "Y" : "N")  );
    */
    logInfo("Joining Network");

    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
        wait_ms(dot->getNextTxMs() + 1);
    }

    logInfo("Joined Network");

    // Display Network session key and data session key from Join command
    /*
        std::vector<uint8_t> tmp = dot->getNetworkSessionKey();
        pc.printf("Network Session Key: ");
        pc.printf("%s\r\n", mts::Text::bin2hexString(tmp, " ").c_str());

        tmp = dot->getDataSessionKey();
        pc.printf("Data Session Key: ");
        pc.printf("%s\r\n", mts::Text::bin2hexString(tmp, " ").c_str());
    */
#endif /* ENABLE_LORA */

    char dataBuf[50];
    uint16_t count = 0;
    Coordinate points[NUMSAMPLES];
    int mode = 0;
    while( 1 ) {
        
        lcd->writeText(0, 0, font_6x8, "Not saving", 10);
        
        if (!button.read()) {
            count = 0;
            mode = 1; 
        }

        if (mode) {
            
            lcd->writeText(0, 0, font_6x8, "Saving    ", 10);
            
            Coordinate a = { gps->getLongitude().degrees, gps->getLongitude().minutes, gps->getLongitude().seconds, gps->getLatitude().degrees, gps->getLatitude().minutes, gps->getLatitude().seconds , accldata._x, accldata._y, accldata._z};
            points[count] = a;
            count++;
        }

        lcd->writeText(0, 1, font_6x8, (gps->getLockStatus() ? "GPS Locked " : "No GPS Lock"), 11);

        if (gps->getLockStatus()) {

            sprintf(dataBuf, "Lat: %d,%d,%d", gps->getLatitude().degrees, gps->getLatitude().minutes, gps->getLatitude().seconds);
            lcd->writeText(0, 2, font_6x8, dataBuf, strlen(dataBuf));

            sprintf(dataBuf, "Long: %d,%d,%d", gps->getLongitude().degrees, gps->getLongitude().minutes, gps->getLongitude().seconds);
            lcd->writeText(0, 3, font_6x8, dataBuf, strlen(dataBuf));

            sprintf(dataBuf, "Sat: %d", gps->getNumSatellites());
            lcd->writeText(0, 4, font_6x8, dataBuf, strlen(dataBuf));

        }

        sprintf(dataBuf, "%04d,%04d,%04d  ", accldata._x, accldata._y, accldata._z);
        lcd->writeText(0, 5, font_6x8, dataBuf, strlen(dataBuf));

        sprintf(dataBuf, "Time: %02d", gps->getTimestamp().tm_sec);
        lcd->writeText(0, 6, font_6x8, dataBuf, strlen(dataBuf));


#ifdef ENABLE_LORA
        send_data.clear();
        // probably not the most efficent way to do this
        for( int i=0; i< strlen(dataBuf); i++ ) {
            send_data.push_back( dataBuf[i] );
        }

        if ((ret = dot->send(send_data)) != mDot::MDOT_OK) {
            logError("failed to send: [%d][%s]", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("send data: %s", Text::bin2hexString(send_data).c_str());
        }
#endif /* ENABLE_LORA */

        sprintf(dataBuf, "C: %03d B: %d", count, button.read());
        lcd->writeText(0, 7, font_6x8, dataBuf, strlen(dataBuf));

        osDelay(1000/50);
        
        if (count == NUMSAMPLES && mode) {
            // Stop sampling
            stoppls = 1;
            while (1) {   
                sprintf(dataBuf, "C: %03d B: %d", count, button.read());
                lcd->writeText(0, 7, font_6x8, dataBuf, strlen(dataBuf)); 
                
                if (!button.read()) {
                    logInfo("Starting...");
                    for (int i = 0; i < NUMSAMPLES; i++) {
                        logInfo("%d,%d,%d,%d,%d,%d,%d,%d,%d", points[i].lat_d, points[i].lat_m, points[i].lat_s, points[i].lng_d, points[i].lng_m, points[i].lng_s, points[i].x, points[i].y, points[i].z);
                        osDelay(1);
                    }
                    logInfo("DONE");
                }
                
                osDelay(100);
            }    
        }
    }
}
