/*
 * LCD code based on FRDM-KL46Z-LCD-rtc-Demo by Paul Staron.
 * Touch slider code based on serialSENS by Aaron Huang.
 */
#include "mbed.h"
#include "SLCD.h"

// Peripherals/Global Objects/Hardware Initialization
SLCD sLCD;                              //LCD Object Initialization
Timer t;                                //Initialize Timer object for measuring time stamp. 
Ticker light_sensor_ticker;             //Set up a timer for light sensor sampling
Ticker adc_ticker;                      //Set up a timer to be used for adc sampling
AnalogIn light_measurement(PTE22);      //Light Sensor Input Pin. 
InterruptIn event_0_to_1(PTD3);         //Initialize Interrupt for 0 to 1 event
PwmOut PwmPin (LED_RED);                   //Initialize PWM Output Pin
AnalogOut DAC(A4);                    //Initialize Analog Sine Wave Output Pin
Ticker Sample_Period;                   //Set up a timer to be used for sample rate interrupts

//Need these global counter 
volatile int i = 0;
float Analog_out_data[128];
char LCDbuffer[32];

// Read Real-time Clock MM:SS, for testing. 
void get_HHMMSS(int *minutes, int *seconds)
{
    time_t rtcTime = time(NULL);
    struct tm *t = localtime(&rtcTime);
    *minutes = t->tm_min;
    *seconds = t->tm_sec;
}

// Set Real-time Clock HH:MM:SS, HH is set to 0. 
void set_MMSS(int minutes, int seconds)
{
    struct tm t;
    if(minutes > 59) minutes = 0;
    t.tm_sec = seconds;
    t.tm_min = minutes;
    t.tm_hour = 0;
    t.tm_mday = 23;
    t.tm_mon = 2;
    t.tm_year = 114;
    set_time (mktime(&t));
}

//Show time
void showTime(void)
{
    time_t rtcTime = time(NULL);
    strftime(LCDbuffer, 4, "%M%S", localtime(&rtcTime));// display HH:MM 
    sLCD.printf(LCDbuffer);     // Send to LCD
}

//Interrupt on pin PTD3, when it goes from 0 to 1
void trigger(void)
{
    t.stop();
    //Send data back to pi, not implemented yet. 
    printf("Rising Event: The time taken was %f seconds\n", t.read());
    t.start();
}

//
void light_sensor()
{
    t.stop();
    //Send light-sensor data, and time stamp back to pi, not implemented yet. 
    printf("Light Sensor Time: The time taken was %f seconds\n", t.read());
    printf("Light Sensor Measurement: %f\n", light_measurement.read()); //Darkness is 1, Light is 0. 
    t.start();
}
//Measure ADC
void adc_measurement()
{
     t.stop();
    //Send ADC measurement data, and time stamp back to pi, not implemented yet.
    printf("ADC Time: The time taken was %f seconds\n", t.read());
    printf("ADC Measurement: %f\n", PTB0); //A0
    t.start();
}

//Rate is in Hz
void adc_rate(int rate, bool start)
{
    //Start the ticker with the given rate. 
    if(start){
        adc_ticker.attach(&adc_measurement, (1/rate));
    }else{
        //turn off the ticker
        adc_ticker.detach();    
    }
}

//True for driving pin to high, false for driving pin to low 
void set_pin(bool HIGH){
    //To turn an LED on, you give it 0, To turn an LED off, you give it ~0. 
    DigitalOut pin(PTA12);
    DigitalOut led_red(LED_RED);
    if (HIGH){
        pin = ~pin;
        led_red = ~led_red;
    }else{
        pin = 0;   
        led_red = 0;
    }
}

// Interrupt routine
// used to output next analog sample whenever a timer interrupt occurs
void Sample_timer_interrupt(void)
{
    // send next analog sample out to D to A
    DAC = Analog_out_data[i];
    // increment pointer and wrap around back to 0 at 128
    i = (i+1) & 0x07F;
}

//Rate is in Hz
void generate_sine_wave(int rate){

    // precompute 128 sample points on one sine wave cycle 
    // used for continuous sine wave output later
    for(int k=0; k<128; k++) {
        Analog_out_data[k]=((1.0 + sin((float(k)/128.0*6.28318530717959)))/2.0);
        // scale the sine wave from 0.0 to 1.0 - as needed for AnalogOut arg 
    }
    // turn on timer interrupts to start sine wave output
    // sample rate is 500Hz with 128 samples per cycle on sine wave
    Sample_Period.attach(&Sample_timer_interrupt, 1.0/(rate*128));
}

//Disable sine wave
void disable_sine_wave(){
    Sample_Period.detach();   
}    

main()
{
    //Set clock time
    int MM = 9;
    int SS = 19;
    set_MMSS(MM,SS);
    
    //Set Pulse
    PwmPin.period(1);  // Set frequency to 1 Hz (period = 5 us)
    
    PwmPin.pulsewidth(0.5);       // Set the duty cycle to 10%, 100ms, might need some delay for getting the rising edge to be on 1s
    
    //Asynchronous 0-1 event interrupt. 
    event_0_to_1.rise(&trigger);
    
    //Light Sensor
    light_sensor_ticker.attach(&light_sensor, 1); // setup ticker to call flip every 1 seconds
    
    //Start the clock for time stamp. 
    t.start();
    
    generate_sine_wave(1);
    while(true)
    {
        showTime(); //Might want to use a ticker on this function instead of while loop, to get rid of unneeded writes to LCD. 
        get_HHMMSS(&MM, &SS);
        //printf("MM: %d\n", MM);
        //printf("SS: %d\n", SS);
        wait(1);
    }
}


