#include "mbed.h"
#include "rga_func.h"
#include "Images/BinaryImage_RZ_A1H.h"
#include "DisplayBace.h"
#include "rtos.h"
#include "SI1143.h"

/**** LCD Parameter **********/
#define LCD_DE_MODE            (0)
#define LCD_SYNC_MODE          (1)

#define LCD_DOT_CLOCK          (13.40f)     // 13.4MHz

#define LCD_H_WIDTH            (480u)
#define LCD_H_BACK_PORCH       (43u)
#define LCD_H_FRONT_PORCH      (52u)
#define LCD_H_SYNC_WIDTH       (41u)

#define LCD_V_WIDTH            (272u)
#define LCD_V_BACK_PORCH       (12u)
#define LCD_V_FRONT_PORCH      (2u)
#define LCD_V_SYNC_WIDTH       (10u)

#define LCD_MODE               (LCD_SYNC_MODE)

/*****************************/

/* FRAME BUFFER Parameter */
#define FRAME_BUFFER_BYTE_PER_PIXEL         (2)
#define FRAME_BUFFER_STRIDE                 (((LCD_H_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL) + 31u) & ~31u)

enum {
    IMG_SETTING,
    IMG_START,
    IMG_RED,
    IMG_BLUE,
    IMG_YELLOW,
    IMG_GREEN
};

DigitalOut  lcd_pwon(P7_15);
DigitalOut  lcd_blon(P8_1);
DigitalOut  touch_reset(P4_0);
PwmOut      lcd_cntrst(P8_15);
DisplayBase Display;
SI1143      sensor(I2C_SDA, I2C_SCL);

static const graphics_image_t* image_file[6] = {
    Setting_jpg_File, Start_jpg_File, red_jpg_File, blue_jpg_File, yellow_jpg_File, green_jpg_File
};

static uint8_t user_frame_buffer[FRAME_BUFFER_STRIDE * LCD_V_WIDTH]__attribute((aligned(32)));  /* 32 bytes aligned */
static uint8_t user_frame_buffer2[FRAME_BUFFER_STRIDE * LCD_V_WIDTH]__attribute((aligned(32))); /* 32 bytes aligned */
static frame_buffer_t frame_buffer_info;
static volatile int32_t vsync_count = 0;
static int sense_old[3] = {0, 0, 0};
static int sense_new[3] = {0, 0, 0};
static const graphics_image_t* image_old = image_file[IMG_START];
static const graphics_image_t* image_new = image_file[IMG_START];

static void IntCallbackFunc_Vsync(DisplayBase::int_type_t int_type) {
    /* Interrupt callback function for Vsync interruption */
    if (vsync_count > 0) {
        vsync_count--;
    }
}

static void Wait_Vsync(const int32_t wait_count) {
    /* Wait for the specified number of times Vsync occurs */
    vsync_count = wait_count;
    while (vsync_count > 0) {
        /* Do nothing */
    }
}

static int Get_Direction(void) {
    int dire_sense[3];
    int ret;
    // Read each led sensor
    sense_new[0] = sensor.get_ps1(2);
    sense_new[1] = sensor.get_ps2(2);
    sense_new[2] = sensor.get_ps3(2);

    if ((sense_new[0] > 50) && (sense_new[1] > 50) && (sense_new[2] > 50)) {
        for (int cnt = 0; cnt < 3; cnt++) {
            dire_sense[cnt] = sense_new[cnt] - sense_old[cnt];
            if (dire_sense[cnt] <= 50) {        // less than 50 or minus
                dire_sense[cnt] = 0;
            }
        }
        if (((sense_old[0] - sense_old[1]) > 50) && ((sense_old[0] - sense_old[2]) > 50)) {
            /* sense1 to ... */
            if ((dire_sense[1] > dire_sense[0]) && (dire_sense[1] > dire_sense[2])) {
                ret = DIREC_UP;
            } else if ((dire_sense[2] > dire_sense[0]) && (dire_sense[2] > dire_sense[1])) {
                ret = DIREC_RIGHT;
            } else {
                ret = DIREC_NON;
            }
        } else if (((sense_old[1] - sense_old[0]) > 50) && ((sense_old[1] - sense_old[2]) > 50)) {
            /* sense2 to ... */
            if ((dire_sense[0] > dire_sense[1]) && (dire_sense[0] > dire_sense[2])) {
                ret = DIREC_DOWN;
            } else if ((dire_sense[2] > dire_sense[0]) && (dire_sense[2] > dire_sense[1])) {
                ret = DIREC_DOWN;
            } else {
                ret = DIREC_NON;
            }
        } else if (((sense_old[2] - sense_old[0]) > 50) && ((sense_old[2] - sense_old[1]) > 50)) {
            /* sense3 to ... */
            if ((dire_sense[0] > dire_sense[1]) && (dire_sense[0] > dire_sense[2])) {
                ret = DIREC_LEFT;
            } else if ((dire_sense[1] > dire_sense[0]) && (dire_sense[1] > dire_sense[2])) {
                ret = DIREC_UP;
            } else {
                ret = DIREC_NON;
            }
        } else {
            ret = DIREC_NON;
        }
    } else {
        ret = DIREC_NON;
    }
    sense_old[0] = sense_new[0];
    sense_old[1] = sense_new[1];
    sense_old[2] = sense_new[2];


    return ret;
}

static void Swap_FrameBuffer(frame_buffer_t * frmbuf_info) {
    if (frmbuf_info->draw_buffer_index == 1) {
        frmbuf_info->draw_buffer_index = 0;
    } else {
        frmbuf_info->draw_buffer_index = 1;
    }
}

static void Update_LCD_Display(frame_buffer_t * frmbuf_info) {
    Display.Graphics_Read_Change(DisplayBase::GRAPHICS_LAYER_0,
    (void *)frmbuf_info->buffer_address[frmbuf_info->draw_buffer_index]);
    Wait_Vsync(1);
}

static void Draw_Image(int direction) {
    int work_width_pos;

    if (direction == DIREC_RIGHT) {
        image_new = image_file[IMG_RED];
    } else if (direction == DIREC_LEFT) {
        image_new = image_file[IMG_BLUE];
    } else if (direction == DIREC_UP) {
        image_new = image_file[IMG_YELLOW];
    } else if (direction == DIREC_DOWN) {
        image_new = image_file[IMG_GREEN];
    }
    for (work_width_pos = 1; work_width_pos <= SCROLL_STEP_NUM; work_width_pos++) {
        /* Draw screen */
        Swap_FrameBuffer(&frame_buffer_info);
        RGA_Func_Scroll(&frame_buffer_info, image_old, image_new, direction, (float32_t)work_width_pos / (float32_t)SCROLL_STEP_NUM);
        Update_LCD_Display(&frame_buffer_info);
    }
    image_old = image_new;
}

int main(void) {
    /* Create DisplayBase object */
    DisplayBase::graphics_error_t error;
    int direction = 0;

    memset(user_frame_buffer, 0, sizeof(user_frame_buffer));
    memset(user_frame_buffer2, 0, sizeof(user_frame_buffer2));
    frame_buffer_info.buffer_address[0] = user_frame_buffer;
    frame_buffer_info.buffer_address[1] = user_frame_buffer2;
    frame_buffer_info.buffer_count      = 2;
    frame_buffer_info.show_buffer_index = 0;
    frame_buffer_info.draw_buffer_index = 0;
    frame_buffer_info.width             = LCD_H_WIDTH;
    frame_buffer_info.byte_per_pixel    = FRAME_BUFFER_BYTE_PER_PIXEL;
    frame_buffer_info.stride            = LCD_H_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL;
    frame_buffer_info.height            = LCD_V_WIDTH;
    frame_buffer_info.pixel_format      = PIXEL_FORMAT_RGB565;

    lcd_pwon = 0;
    lcd_blon = 0;
    touch_reset = 0;
    Thread::wait(100);
 
    lcd_pwon = 1;
    lcd_blon = 1;
    touch_reset = 1;
    Thread::wait(100);

    DisplayBase::lcd_config_t lcd_config;
    PinName lvds_pin[8] = {
        /* data pin */
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0
    };
    DisplayBase::rect_t rect;

    lcd_config.lcd_type             = DisplayBase::LCD_TYPE_LVDS;
    lcd_config.intputClock          = 66.67f;
    lcd_config.outputClock          = LCD_DOT_CLOCK;
    lcd_config.lcd_outformat        = DisplayBase::LCD_OUTFORMAT_RGB888;
    lcd_config.lcd_edge             = DisplayBase::EDGE_RISING;
#if(LCD_MODE) //SYNC Mode
    lcd_config.h_toatal_period      = (LCD_H_BACK_PORCH + LCD_H_WIDTH + LCD_H_FRONT_PORCH);
    lcd_config.v_toatal_period      = (LCD_V_BACK_PORCH + LCD_V_WIDTH + LCD_V_FRONT_PORCH);
 
    lcd_config.h_disp_widht         = (LCD_H_WIDTH);
    lcd_config.v_disp_widht         = (LCD_V_WIDTH);
    lcd_config.h_back_porch         = (LCD_H_BACK_PORCH);
    lcd_config.v_back_porch         = (LCD_V_BACK_PORCH);
 
    lcd_config.h_sync_port          = DisplayBase::LCD_TCON_PIN_2;
    lcd_config.h_sync_port_polarity = DisplayBase::SIG_POL_INVERTED;
    lcd_config.h_sync_width         = LCD_H_SYNC_WIDTH;
 
    lcd_config.v_sync_port          = DisplayBase::LCD_TCON_PIN_0;
    lcd_config.v_sync_port_polarity = DisplayBase::SIG_POL_INVERTED;
    lcd_config.v_sync_width         = LCD_V_SYNC_WIDTH;
 
    lcd_config.de_port              = DisplayBase::LCD_TCON_PIN_3;
    lcd_config.de_port_polarity     = DisplayBase::SIG_POL_NOT_INVERTED;
#else  //DE Mode
    lcd_config.h_toatal_period      = (LCD_H_WIDTH + 80u);
    lcd_config.v_toatal_period      = (LCD_V_WIDTH);

    lcd_config.h_disp_widht         = (LCD_H_WIDTH);
    lcd_config.v_disp_widht         = (LCD_V_WIDTH);
    lcd_config.h_back_porch         = (68u);
    lcd_config.v_back_porch         = (18u);

    lcd_config.h_sync_port          = DisplayBase::LCD_TCON_PIN_NON;
    lcd_config.h_sync_port_polarity = DisplayBase::SIG_POL_NOT_INVERTED;
    lcd_config.h_sync_width         = 0;

    lcd_config.v_sync_port          = DisplayBase::LCD_TCON_PIN_NON;
    lcd_config.v_sync_port_polarity = DisplayBase::SIG_POL_NOT_INVERTED;
    lcd_config.v_sync_width         = 0;

    lcd_config.de_port              = DisplayBase::LCD_TCON_PIN_3;
    lcd_config.de_port_polarity     = DisplayBase::SIG_POL_INVERTED;
#endif

    /* Graphics initialization process */
    error = Display.Graphics_init(&lcd_config);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Interrupt callback function setting (Vsync signal output from scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_LO_VSYNC, 0, IntCallbackFunc_Vsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    Display.Graphics_Lvds_Port_Init(lvds_pin, 8);
    rect.vs = 0;
    rect.vw = LCD_V_WIDTH;
    rect.hs = 0;
    rect.hw = LCD_H_WIDTH;

    Display.Graphics_Read_Setting(
        DisplayBase::GRAPHICS_LAYER_0,
        (void *)frame_buffer_info.buffer_address[0],
        FRAME_BUFFER_STRIDE,
        DisplayBase::GRAPHICS_FORMAT_RGB565,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        &rect
    );

    /* Display Start */
    Set_RGAObject(&frame_buffer_info);
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_0);
    lcd_cntrst.write(1.0);

    // Setup the baseline
    RGA_Func_DrawTopScreen(&frame_buffer_info, image_file[IMG_SETTING]);
    printf("SI1143 Gesture Sensor setting...\n");
    sensor.bias(1,5);
    Thread::wait(1000);
    RGA_Func_DrawTopScreen(&frame_buffer_info, image_file[IMG_START]);
    printf("SI1143 Gesture Sensor setting finished!\n");

    while (1) {
        /* Get coordinates */
        direction = Get_Direction();
        if (direction != 0) {
            Draw_Image(direction);
        }
    }
}
