/* nRF51 mbed opentrigger example
 * Copyright (c) 2016 Florian Fida
 *
 * based on 'Demo for an Input Service'
 * https://developer.mbed.org/teams/Bluetooth-Low-Energy/code/BLE_Button/
 * 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"

DigitalOut  led1(LED1); /* RED   : Indicating a pressed button */
DigitalOut  led2(LED2); /* GREEN : Blinking to indicate that the device is running */
DigitalOut  led3(LED3); /* BLUE  : Only on when advertising */

InterruptIn button0(P0_15);
InterruptIn button1(P0_16);
InterruptIn button2(P0_17);
InterruptIn button3(P0_18);
InterruptIn button4(P0_19);
InterruptIn button5(P0_20);

const uint16_t        MIN_ADVERT_INTERVAL = 50; /* msec */
static uint8_t        gpio_states[] = {0x01, 0x01, 0x01, 0x01, 0x01, 0x01}; /* initial state is up */
static uint8_t        event_id = 0;

enum {
    RELEASED = 0,
    PRESSED,
    IDLE
};
static uint8_t buttonState = IDLE;

void blinkLed(void){
    led2 = !led2;
}

void button0PressedCallback(void)
{
    /* Note that the buttonPressedCallback() executes in interrupt context, so it is safer to access
     * BLE device API from the main thread. */
    buttonState = PRESSED;
    gpio_states[0]=0x00;
    event_id++;
}

void button0ReleasedCallback(void)
{
    /* Note that the buttonReleasedCallback() executes in interrupt context, so it is safer to access
     * BLE device API from the main thread. */
    buttonState = RELEASED;
    gpio_states[0]=0x01;
    event_id++;
}


// Callbacks for the other 5 Buttons
void button1PressedCallback(void) { buttonState = PRESSED;  gpio_states[1]=0x00; event_id++; }
void button1ReleasedCallback(void){ buttonState = RELEASED; gpio_states[1]=0x01; event_id++; }
void button2PressedCallback(void) { buttonState = PRESSED;  gpio_states[2]=0x00; event_id++; }
void button2ReleasedCallback(void){ buttonState = RELEASED; gpio_states[2]=0x01; event_id++; }
void button3PressedCallback(void) { buttonState = PRESSED;  gpio_states[3]=0x00; event_id++; }
void button3ReleasedCallback(void){ buttonState = RELEASED; gpio_states[3]=0x01; event_id++; }
void button4PressedCallback(void) { buttonState = PRESSED;  gpio_states[4]=0x00; event_id++; }
void button4ReleasedCallback(void){ buttonState = RELEASED; gpio_states[4]=0x01; event_id++; }
void button5PressedCallback(void) { buttonState = PRESSED;  gpio_states[5]=0x00; event_id++; }
void button5ReleasedCallback(void){ buttonState = RELEASED; gpio_states[5]=0x01; event_id++; }

// Just in case
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance().gap().startAdvertising();
}

void periodicCallback(void)
{
    blinkLed();
}

/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t bt_error)
{
    /* Initialization error handling should go here */
    
    printf("bt_error=%i\n\r",bt_error);
    error("Panic!");
}

bool gpioSet(void){
    return
        gpio_states[0] == 0x00 ||
        gpio_states[1] == 0x00 ||
        gpio_states[2] == 0x00 ||
        gpio_states[3] == 0x00 ||
        gpio_states[4] == 0x00 ||
        gpio_states[5] == 0x00
    ;
}

void updateManufacturerData(void){
    
    // disable interrupts while building the package to avoid a race condition
    __disable_irq();  
    
    // http://tokencube.com/bluetooth-sensor.html
    uint8_t data[] = { 
        0xee, 0xff,                         // Tokencube Manufacturer ID
        0x04,                               // Token Module Version
        0x01,                               // Firmware Version
        0x11,                               // Page 1 of 1
        0x07, gpioSet() ? 0x01 : 0x00,      // PIR value (0x00 = off, 0x01 = on)
        0x0F, event_id,                     // EventId (0x00 - 0xFF, every time a button gets pressed, the EventId changes)
        gpio_states[0], gpio_states[1],     // 6 GPIO Values (0x00 = down, 0x01 = up)
        gpio_states[2], gpio_states[3], 
        gpio_states[4], gpio_states[5],
    };
    
    __enable_irq();
    BLE::Instance().gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, data, sizeof(data));
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }
    
    uint8_t macAddress[6] = {0x00};
    BLEProtocol::AddressType_t pubAddr = BLEProtocol::AddressType::PUBLIC;
    ble.gap().getAddress(&pubAddr, macAddress);
    
    /* change device MAC address */
    //const uint8_t address1[] = {0xe7,0xAA,0xAA,0xAA,0xAA,0xAA}; 
    //ble.gap().setAddress(pubAddr, address1);

    ble.gap().onDisconnection(disconnectionCallback);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    
    ble.gap().setAdvertisingInterval(MIN_ADVERT_INTERVAL);
    updateManufacturerData();
    error = ble.gap().startAdvertising();
    if (error != BLE_ERROR_NONE) onBleInitError(ble, error);
    
    // Print Interval and MAC Address
    uint16_t minInterval = ble.gap().getMinAdvertisingInterval();
    printf("MinAdvertisingInterval = %i\n\r", minInterval);
    printf("MAC Address = %02X:%02X:%02X:%02X:%02X:%02X\n\r", macAddress[5], macAddress[4], macAddress[3], macAddress[2], macAddress[1], macAddress[0]);

}

int main(void)
{  
    // Turn off all LED's
    led1 = !0; led2 = !0; led3 = !0;
    
    // Put all GPIO's in PullUp mode, so we do not need pullup resistors.
    button0.mode(PullUp);
    button1.mode(PullUp);
    button2.mode(PullUp);
    button3.mode(PullUp);
    button4.mode(PullUp);
    button5.mode(PullUp);
    
    // Setup handler for all buttons
    button0.fall(button0PressedCallback);
    button0.rise(button0ReleasedCallback);
    button1.fall(button1PressedCallback);
    button1.rise(button1ReleasedCallback);
    button2.fall(button2PressedCallback);
    button2.rise(button2ReleasedCallback);
    button3.fall(button3PressedCallback);
    button3.rise(button3ReleasedCallback);
    button4.fall(button4PressedCallback);
    button4.rise(button4ReleasedCallback);
    button5.fall(button5PressedCallback);
    button5.rise(button5ReleasedCallback);
    
    // This will make our green led blink
    Ticker ticker;
    ticker.attach(periodicCallback, 1);

    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);
    
    /* SpinWait for initialization to complete. This is necessary because the BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }
    
    uint8_t cnt = 0;
    while (true) {
        
        // we only care about situations where a button was pressed
        if (buttonState != IDLE) {
            if(buttonState == PRESSED)  led1=!1; 
            if(buttonState == RELEASED) led1=!0; 
            ticker.attach(periodicCallback, 1);
            updateManufacturerData();
            cnt = 0;
            buttonState = IDLE;
        }

        
        // start advertising
        if(cnt < 1){
            ble.gap().startAdvertising();
            led3 = !1;
        }else if(cnt == 1){
            updateManufacturerData();
        }
        
        // stop advertising after 3 cycles
        if(cnt > 3){
            ble.gap().stopAdvertising();
            led3 = !0;
            led2 = !0;
            ticker.detach();
            deepsleep();
        }else{
            if(!gpioSet()) cnt++;
        }
        
        ble.waitForEvent();
        //printf("cnt=%i\n\r",cnt);
    }
}
