#include "mbed.h"
#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>
#include <math.h>

#define RETURN_OK              0
#define RETURN_ERR            -1
//define baudrate
#define PC_BAUDRATE            115200

/* functions prototype */

int configNetwork(void);
int joinNetwork(void);
int send_data(char *str);

/* Global variables */
Serial pc       (USBTX, USBRX);  // tx, rx

mDot* dot;

static std::string config_network_name = "chinaiot";
static std::string config_network_pass = "password";
static uint8_t config_frequency_sub_band = 1; 

int main() 
{    
    int32_t ret, counter;
    char dataBuf[15];

    pc.baud(PC_BAUDRATE);
    pc.printf("\n\r\n\r\n\r");
    pc.printf("============================================\n\r");
    pc.printf("SiFOX remote card read system!\n\r");
    pc.printf("============================================\n\r");
    pc.printf("PC COM RS232 baudrate: %d \n\r", PC_BAUDRATE);

    //*******************************************
    // Configurate Network
    //*******************************************
    ret = configNetwork();
    if(ret != RETURN_OK)         
        dot->sleep(5, mDot::RTC_ALARM); //sleep a while and restart
    
    //*******************************************
    // Join Network
    //*******************************************
    ret = joinNetwork();
    if(ret != RETURN_OK)         
        dot->sleep(5, mDot::RTC_ALARM); //sleep a while and restart
   
    counter = 0;
    while(1) {
        sprintf(dataBuf, "%d", counter);
        
        /* Send data to gateway */
        send_data(dataBuf);        
        /* wait a while */
        wait(10);
        
        logInfo("RSSI Last:%d", dot->getRssiStats().last);
        logInfo("SNR Last:%d", dot->getSnrStats().last);

        counter++;          
    }
}

    
int configNetwork(void)
{
    int32_t ret; 
    
    dot = mDot::getInstance();
    
    // print library version information
    pc.printf("version: %s\n\r", dot->getId().c_str());
    
    //*******************************************
    // configuration
    //*******************************************
    // reset to default config so we know what state we're in
    dot->resetConfig();
    
    dot->setTxPower(20);

    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    // set up the mDot with our network information: frequency sub band, network name, and network password
    // these can all be saved in NVM so they don't need to be set every time - see mDot::saveConfig()

    // frequency sub band is only applicable in the 915 (US) frequency band
    // if using a MultiTech Conduit gateway, use the same sub band as your Conduit (1-8) - the mDot will use the 8 channels in that sub band
    // if using a gateway that supports all 64 channels, use sub band 0 - the mDot will use all 64 channels
    //pc.printf("Setting TX frequency band as 868MHz\n\r");
    //if ((ret = dot->setTxFrequency(mDot::FB_868)) != mDot::MDOT_OK){
    //    pc.printf("Error:failed to set TX frequency band %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
    //    return RETURN_ERR;
    //}
    //pc.printf("Setting RX frequency band as 868MHz\n\r");
    //if ((ret = dot->setRxFrequency(mDot::FB_868)) != mDot::MDOT_OK){
    //    pc.printf("Error:failed to set RX frequency band %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
    //    return RETURN_ERR;
    //}
    pc.printf("Setting frequency sub band\n\r");
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        pc.printf("Error:failed to set frequency sub band %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
        return RETURN_ERR;
    }

    pc.printf("Setting network name\n\r");
    if ((ret = dot->setNetworkName(config_network_name)) != mDot::MDOT_OK) {
        pc.printf("Error:failed to set network name %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
        return RETURN_ERR;
    }

    pc.printf("Setting network password\n\r");
    if ((ret = dot->setNetworkPassphrase(config_network_pass)) != mDot::MDOT_OK) {
        pc.printf("Error:failed to set network password %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
        return RETURN_ERR;
    }
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    pc.printf("Setting TX spreading factor\n\r");
    if ((ret = dot->setTxDataRate(mDot::SF_10)) != mDot::MDOT_OK) {
        pc.printf("Error:failed to set TX datarate %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
        return RETURN_ERR;
    }

    // request receive confirmation of packets from the gateway
    pc.printf("Enabling ACKs\n\r");
    if ((ret = dot->setAck(1)) != mDot::MDOT_OK) {
        pc.printf("Error:failed to enable ACKs %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
        return RETURN_OK;
    }

    // save this configuration to the mDot's NVM
    pc.printf("Saving config\n\r");
    if (! dot->saveConfig()) {
        pc.printf("Error:failed to save configuration\n\r");
    }
    
    return RETURN_OK;
}   // end of configuration


int joinNetwork(void)
{    
    int32_t ret,i; 
    std::vector<uint8_t> sendData;
    char _header[] = "Reset!";
    
    // attempt to join the network
    pc.printf("Joining network...\n\r");
    
    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        pc.printf("Error: failed to join network %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        return RETURN_ERR;
    }

    sendData.clear(); 
    // format data for sending to the gateway
    for( i=0; i< strlen(_header); i++ )
               sendData.push_back( _header[i] );
               
    // send the data to the gateway
    pc.printf("Send header to the gateway\n\r");

    if ((ret = dot->send(sendData)) != mDot::MDOT_OK) {
        pc.printf("Error: failed to send %d:%s\n\r", ret, mDot::getReturnCodeString(ret).c_str());
        return RETURN_ERR;
    } else {
        pc.printf("Successfully sent data to gateway\n\r");
    } 
    
    return RETURN_OK;
}

int send_data(char *str)
{
    int32_t i, ret;
    std::vector<uint8_t> sendData;
    
    //Send the data to Gateway
    sendData.clear();   
    // probably not the most efficent way to do this
    for(i=0; i< strlen(str); i++ )
        sendData.push_back(str[i] );
        
    // send the data to the gateway
    pc.printf("Send %s to Gateway \n\r", str); 
    
    if ((ret = dot->send(sendData)) != mDot::MDOT_OK) {
        pc.printf("Error:failed to send\n\r", ret, mDot::getReturnCodeString(ret).c_str());
        return RETURN_ERR;
    } else {            
        pc.printf("Sent data to gateway successfully!\n\r");
    }
    return RETURN_OK;
}


