/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "UbloxATCellularInterface.h"
#include "UbloxATCellularInterfaceN2xx.h"

// You must select the correct interface library for your board, by
// uncommenting the correct line below. Supported combinations are
// indicated with a "Y" in the table below.
//
//                            C030_U201   C030_N211      C027
// UbloxATCellularInterface       Y            -           Y
// UbloxATCellularInterfaceN2xx   -            Y           -
// Note: the N211 module supports only UDP, not TCP

// UbloxATCellularInterface and UbloxATCellularInterfaceN2xx
// uses an IP stack on the cellular module and hence uses less RAM (significant on C027).
// This also allows other AT command operations (e.g. sending an SMS) to happen
// during a data transfer (for which you should replace the
// UbloxATCellularInterface library with the UbloxATCellularInterfaceExt
// library).  However, it is slower than using the LWIP/PPP on the mbed
// MCU interface since more string parsing is required.
#define INTERFACE_CLASS  UbloxATCellularInterface
//#define INTERFACE_CLASS  UbloxATCellularInterfaceN2xx

// The credentials of the SIM in the board.  If PIN checking is enabled
// for your SIM card you must set this to the required PIN.
#define PIN "0000"

// Network credentials.  You should set this according to your
// network/SIM card.  For C030 non-N2xx boards, leave the parameters as NULL
// otherwise, if you do not know the APN for your network, you may
// either try the fairly common "internet" for the APN (and leave the
// username and password NULL), or you may leave all three as NULL and then
// a lookup will be attempted for a small number of known networks
// (see APN_db.h in mbed-os/features/netsocket/cellular/utils).
#define APN         NULL
#define USERNAME    NULL
#define PASSWORD    NULL
#define TCP_SERVER "os.mbed.com"

//comment out the following line to disable modem current measurement
#define CURRENT_MEASUREMENT

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

// The user button
volatile bool buttonPressed = false;
#ifdef CURRENT_MEASUREMENT
AnalogIn ain_icellular(MDMCURRENTSENSE);
Thread icell_thread;
#endif

static void good() {
    ledGreen = 0;
    ledBlue = 1;
    ledRed = 1;
}

static void bad() {
    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;
}

static void event() {
    ledBlue = 0;
    ledRed = 1;
    ledGreen = 1;
}

static void pulseEvent() {
    event();
    wait_ms(500);
    good();
}

static void ledOff() {
    ledBlue = 1;
    ledRed = 1;
    ledGreen = 1;
}

static void printNtpTime(char * buf, int len)
{
    time_t timestamp = 0;
    struct tm *localTime;
    char timeString[25];
    time_t TIME1970 = 2208988800U;

    if (len >= 43) {
        timestamp |= ((int) *(buf + 40)) << 24;
        timestamp |= ((int) *(buf + 41)) << 16;
        timestamp |= ((int) *(buf + 42)) << 8;
        timestamp |= ((int) *(buf + 43));
        timestamp -= TIME1970;
        localTime = localtime(&timestamp);
        if (localTime) {
            if (strftime(timeString, sizeof(timeString), "%a %b %d %H:%M:%S %Y", localTime) > 0) {
                printf("NTP timestamp is %s.\n", timeString);
            }
        }
    }
}

static void cbButton()
{
    buttonPressed = true;
}

#ifdef CURRENT_MEASUREMENT
float calculate_icellular_samples() {
    float ain=0.0f;
    float icellular_val;
    float c_number_of_analog_samples = 50;

    ain = 0;
    icellular_val = 0;
    for(int i = 0; i < c_number_of_analog_samples; i++) {
        ain = (ain + ain_icellular.read());
        wait_ms(20);
    }
    ain = ain/c_number_of_analog_samples;
    icellular_val = (ain*1.8*1000)/7.0f;


    printf("Voltage in mV: %f\n", icellular_val * 7.0f);
    printf("Current draw in mA: %f\n\n", icellular_val);

    return icellular_val;
}

void icell_thread_handler() {

    while(1) {
        calculate_icellular_samples();
    }
}
#endif

/* This example program for the u-blox C030 and C027 boards instantiates
 * the UbloxATCellularInterface and uses it to make a simple sockets
 * connection to a server, using 2.pool.ntp.org for UDP and developer.mbed.org for TCP.
 * For a more comprehensive example, where higher layer protocols
 * make use of the same sockets interface, see example-ublox-mbed-client.
 * Progress may be monitored with a serial terminal running at 9600 baud.
 * The LED on the C030 board will turn green when this program is
 * operating correctly, pulse blue when a sockets operation is completed
 * and turn red if there is a failure.
 */

int main()
{
#ifdef CURRENT_MEASUREMENT
    //current monitoring using Icellular
    icell_thread.start(icell_thread_handler);
#endif
    INTERFACE_CLASS *interface = new INTERFACE_CLASS();
    // If you need to debug the cellular interface, comment out the
    // instantiation above and uncomment the one below.
    // For the N2xx interface, change xxx to MBED_CONF_UBLOX_CELL_BAUD_RATE,
    // while for the non-N2xx interface change it to MBED_CONF_UBLOX_CELL_N2XX_BAUD_RATE.
//    INTERFACE_CLASS *interface = new INTERFACE_CLASS(MDMTXD, MDMRXD,
//                                                     xxx,
//                                                     true);
#ifndef TARGET_UBLOX_C030_N211
    TCPSocket sockTcp;
#endif
    UDPSocket sockUdp;
    SocketAddress udpServer;
    SocketAddress udpSenderAddress;
    SocketAddress tcpServer;
    char buf[1024];
    int x;
#ifdef TARGET_UBLOX_C027
    // No user button on C027
    InterruptIn userButton(NC);
#else
    InterruptIn userButton(SW0);
#endif

    // Attach a function to the user button
    userButton.rise(&cbButton);

    interface->set_credentials(APN, USERNAME, PASSWORD);
    interface->init(PIN);

    printf("Checking if already registered!\n");

    if ( (interface->is_registered_csd() || interface->is_registered_psd() || interface->is_registered_eps()) ) {
        printf("De-registering...\n\n");
        interface->nwk_deregistration();
    }

    if (interface->set_modem_rat(UbloxATCellularInterface::GPRS_EGPRS)) {
        printf("GPRS_EGPRS RAT configured\n");
    }
    else {
        printf("GPRS_EGPRS RAT not configured\n");
    }

    if (interface->reboot_modem()) {
        printf("Reboot successful");
    }

    good();
    printf("Starting up, please wait up to 180 seconds for network registration to complete...\n");
    for (x = 0; interface->connect(PIN) != 0; x++) {
        if (x > 0) {
            bad();
            printf("Retrying (have you checked that an antenna is plugged in and your APN is correct?)...\n");
        }
    }
    pulseEvent();

#ifdef TARGET_UBLOX_C030_R41XM
    int status;
    if (interface->get_idle_mode(&status)) {
        printf("\n\nIdle mode is %s\n", status ? "Enabled" : "Disabled");
    }
    if (!status) {
        printf("Enabling idle mode...\n");
        if (interface->set_idle_mode(true) == false)
        {
            printf("Unable to set idle mode, is PSM enabled?\n");
        }
        if (interface->get_idle_mode(&status)) {
            printf("Idle mode is %s\n", status ? "Enabled" : "Disabled");
        }
    }
#ifdef TARGET_UBLOX_C030_R412M
    if (interface->set_power_saving_mode(30, 30) == false)
    {
        printf("Unable to set PSM, is idle mode enabled?\n\n");
    }
#endif
#endif

    printf("Getting the IP address of \"" TCP_SERVER "\" and \"2.pool.ntp.org\"...\n");
    if ((interface->gethostbyname("2.pool.ntp.org", &udpServer) == 0) &&
        (interface->gethostbyname(TCP_SERVER, &tcpServer) == 0)) {
        pulseEvent();

        udpServer.set_port(123);
        tcpServer.set_port(80);
        printf("\"2.pool.ntp.org\" address: %s on port %d.\n", udpServer.get_ip_address(), udpServer.get_port());
        printf("\"os.mbed.com\" address: %s on port %d.\n", tcpServer.get_ip_address(), tcpServer.get_port());

        printf("Performing socket operations in a loop (until the user button is pressed on C030 or forever on C027)...\n");
        while (!buttonPressed) {
            // UDP Sockets
            printf("=== UDP ===\n");
            printf("Opening a UDP socket...\n");
            if (sockUdp.open(interface) == 0) {
                pulseEvent();
                printf("UDP socket open.\n");
                sockUdp.set_timeout(10000);
                printf("Sending time request to \"2.pool.ntp.org\" over UDP socket...\n");
                memset (buf, 0, sizeof(buf));
                *buf = '\x1b';
                if (sockUdp.sendto(udpServer, (void *) buf, 48) == 48) {
                    pulseEvent();
                    printf("Socket send completed, waiting for UDP response...\n");
                    x = sockUdp.recvfrom(&udpSenderAddress, buf, sizeof (buf));
                    if (x > 0) {
                        pulseEvent();
                        printf("Received %d byte response from server %s on UDP socket:\n"
                               "-------------------------------------------------------\n",
                               x, udpSenderAddress.get_ip_address());
                        printNtpTime(buf, x);
                        printf("-------------------------------------------------------\n");
                    }
                }
                printf("Closing socket...\n");
                sockUdp.close();
                pulseEvent();
                printf("Socket closed.\n");
            }

#ifndef TARGET_UBLOX_C030_N211
            // TCP Sockets
            printf("=== TCP ===\n");
            printf("Opening a TCP socket...\n");
            if (sockTcp.open(interface) == 0) {
                pulseEvent();
                printf("TCP socket open.\n");
                sockTcp.set_timeout(10000);
                printf("Connecting socket to %s on port %d...\n", tcpServer.get_ip_address(), tcpServer.get_port());
                if (sockTcp.connect(tcpServer) == 0) {
                    pulseEvent();
                    printf("Connected, sending HTTP GET request to %s over socket...\n", TCP_SERVER);
                    strcpy (buf, "GET /media/uploads/mbed_official/hello.txt HTTP/1.0\r\n\r\n");
                    // Note: since this is a short string we can send it in one go as it will
                    // fit within the default buffer sizes.  Normally you should call sock.send()
                    // in a loop until your entire buffer has been sent.
                    if (sockTcp.send((void *) buf, strlen(buf)) == (int) strlen(buf)) {
                        pulseEvent();
                        printf("Socket send completed, waiting for response...\n");
                        x = sockTcp.recv(buf, sizeof (buf));
                        if (x > 0) {
                            pulseEvent();
                            printf("Received %d byte response from server on TCP socket:\n"
                                   "----------------------------------------------------\n%.*s"
                                   "----------------------------------------------------\n",
                                    x, x, buf);
                        }
                    }
                }
                printf("Closing socket...\n");
                sockTcp.close();
                pulseEvent();
                printf("Socket closed.\n");
            }
#endif
            printf("Sleeping for 20 seconds \n");
            wait_ms(20000); //modem will enter sleep after 6 seconds. This can be observed from Icellular value.
            printf("Waking up\n");
#ifndef TARGET_UBLOX_C027
            printf("[Checking if user button has been pressed]\n");
#endif
        }

        pulseEvent();
        printf("User button was pressed, stopping...\n");
        interface->disconnect();
        ledOff();
        printf("Stopped.\n");
    } else {
        bad();
        printf("Unable to get IP address of \"developer.mbed.org\" or \"2.pool.ntp.org\".\n");
    }
}

// End Of File
