/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define __STDC_FORMAT_MACROS
#include <inttypes.h>
#include <string>
#include <sstream>
#include <vector>

#include "mbed.h"
#include "mbedtls/entropy_poll.h"
#include "UbloxATCellularInterface.h"
#include "OnboardCellularInterface.h"
#include "UbloxATCellularInterfaceN2xx.h"
#include "simpleclient.h"
#include "security.h"
#include "mbed_trace.h"
#include "mbed.h"

// You must select the correct interface library for your board, by
// uncommenting the correct line below. Supported combinations are
// indicated with a "Y" in the table below.
//
//                            C030_U201   C030_N211
// UbloxATCellularInterface       Y            -
// OnboardCellularInterface       Y            -
// UbloxATCellularInterfaceN2xx   -            Y
// Note: the N211 module supports only UDP, not TCP

// OnboardCellularInterface uses LWIP and the PPP cellular interface
// on the mbed MCU, while using UbloxATCellularInterface and
// UbloxATCellularInterfaceN2xx uses an IP stack on the cellular
// module and hence uses less RAM.  This also allows other AT command
// operations (e.g. sending an SMS) to happen during a data transfer
// (for which you should replace the UbloxATCellularInterface library with
// the UbloxATCellularInterfaceExt library).  However, it is slower than
// using the LWIP/PPP on the mbed MCU interface since more string parsing
// is required.
#define INTERFACE_CLASS  UbloxATCellularInterface
//#define INTERFACE_CLASS  OnboardCellularInterface
//#define INTERFACE_CLASS  UbloxATCellularInterfaceN2xx

// The credentials of the SIM in the board.  If PIN checking is enabled
// for your SIM card you must set this to the required PIN.
#define PIN "0000"

// Network credentials.  You should set this according to your
// network/SIM card.  For C030 non-N2xx boards, leave the parameters as NULL
// otherwise, if you do not know the APN for your network, you may
// either try the fairly common "internet" for the APN (and leave the
// username and password NULL), or you may leave all three as NULL and then
// a lookup will be attempted for a small number of known networks
// (see APN_db.h in mbed-os/features/netsocket/cellular/utils).
#define APN         NULL
#define USERNAME    NULL
#define PASSWORD    NULL

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

// The user button
volatile bool buttonPressed = false;

static void good() {
    ledGreen = 0;
    ledBlue = 1;
    ledRed = 1;
}

static void bad() {
    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;
}

static void event() {
    ledBlue = 0;
    ledRed = 1;
    ledGreen = 1;
}

static void pulseEvent() {
    event();
    wait_ms(500);
    good();
}

static void ledOff() {
    ledBlue = 1;
    ledRed = 1;
    ledGreen = 1;
}

// Resource values for the Device Object
struct MbedClientDevice device = {
        "Manufacturer", // Manufacturer
        "Type",         // Type
        "ModelNumber",  // ModelNumber
        "SerialNumber"  // SerialNumber
        };

class LedResource {
public:
    LedResource() {
        ledObject = M2MInterfaceFactory::create_object("3311");
        M2MObjectInstance *inst = ledObject->create_object_instance();

        // An observable resource
        M2MResource *onResource = inst->create_dynamic_resource("5850", "On/Off", M2MResourceInstance::BOOLEAN, true);
        onResource->set_operation(M2MBase::GET_PUT_ALLOWED);
        onResource->set_value(false);

        // An multi-valued resource
        M2MResource *dimmerResource = inst->create_dynamic_resource("5851", "Dimmer", M2MResourceInstance::BOOLEAN, false);
        dimmerResource->set_operation(M2MBase::GET_PUT_ALLOWED);
        dimmerResource->set_value(false);
    }

    ~LedResource() {
    }

    M2MObject *get_object() {
        return ledObject;
    }

private:
    M2MObject *ledObject;
};

static void cbButton()
{
    buttonPressed = true;
    pulseEvent();
}

/* This example program for the u-blox C030 board instantiates mbedClient
 * and runs it over a UbloxATCellularInterface or a OnboardCellularInterface to
 * the mbed connector.
 * Progress may be monitored with a serial terminal running at 9600 baud.
 * The LED on the C030 board will turn green when this program is
 * operating correctly, pulse blue when an mbedClient operation is completed
 * and turn red if there is a failure.
 *
 * Note: mbedClient malloc's around 34 kbytes of RAM, more than is available on
 * the C027 platform, hence this example will not run on the C027 platform.
 *
 * IMPORTANT to use this example you must first register with the mbed Connector:
 *
 * https://connector.mbed.com/
 *
 * ...using your mbed developer credentials and generate your own copy of the file
 * security.h, replacing the empty one in this directory with yours and
 * recompiling/downloading the code to your board.
 */

int main()
{
    INTERFACE_CLASS *interface = new INTERFACE_CLASS();
    // If you need to debug the cellular interface, comment out the
    // instantiation above and uncomment the one below.
    // For the N2xx interface, change xxx to MBED_CONF_UBLOX_CELL_N2XX_BAUD_RATE,
    // while for the non-N2xx interface change it to MBED_CONF_UBLOX_CELL_BAUD_RATE.
//    INTERFACE_CLASS *interface = new INTERFACE_CLASS(MDMTXD, MDMRXD,
//                                                     xxx,
//                                                     true);
    MbedClient *mbedClient = new MbedClient(device);
    M2MObjectList objectList;
    M2MSecurity *registerObject;
    M2MDevice *deviceObject;
    LedResource ledResource;
    unsigned int seed;
    size_t len;
    InterruptIn userButton(SW0);

    // Attach a function to the user button
    userButton.rise(&cbButton);
    
#if MBED_CONF_MBED_TRACE_ENABLE
    mbed_trace_init();
#endif
    srand(seed);

    // Randomize source port
    mbedtls_hardware_poll(NULL, (unsigned char *) &seed, sizeof seed, &len);

    good();
    printf("Starting up, please wait up to 180 seconds for network registration to complete...\n");
    pulseEvent();
        
    // Create endpoint interface to manage register and unregister
    mbedClient->create_interface("coap://api.connector.mbed.com:5684", interface);

    // Create objects of varying types, see simpleclient.h for more details on implementation.
    registerObject = mbedClient->create_register_object(); // Server object specifying connector info
    deviceObject = mbedClient->create_device_object();     // Device resources object

    // Add objects to list
    objectList.push_back(deviceObject);
    objectList.push_back(ledResource.get_object());

    // Set endpoint registration object
    mbedClient->set_register_object(registerObject);

    printf("Updating object registration in a loop (with a 30 second refresh period) until the user button is presed...\n");
    interface->set_credentials(APN, USERNAME, PASSWORD);
    while (!buttonPressed) {
        // Make sure cellular is connected
        if (interface->connect(PIN) == 0) {
            pulseEvent();
            printf("[Still] connected to packet network.\n");
            if (mbedClient->register_successful()) {
                printf("Updating registration (follow progress at https://connector.mbed.com/#home)...\n");
                mbedClient->test_update_register();
            } else {
                printf("Registering with connector (follow progress at https://connector.mbed.com/#home)...\n");
                mbedClient->test_register(registerObject, objectList);
            }
        } else {
            bad();
            printf("Failed to connect, will retry (have you checked that an antenna is plugged in and your APN is correct?)...\n");
        }
        Thread::wait(30000);
        printf("[Checking if user button has been pressed]\n");
    }
    
    pulseEvent();
    printf("User button was pressed, stopping...\n");
    mbedClient->test_unregister();
    interface->disconnect();
    ledOff();
    printf("Stopped.\n");
    M2MDevice::delete_instance();
}

// End Of File
