/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "UbloxCellularDriverGen.h"

// The credentials of the SIM in the board.  If PIN checking is enabled
// for your SIM card you must set this to the required PIN.
#define PIN "0000"

// Network credentials.  You should set this according to your
// network/SIM card.  For C030 boards, leave the parameters as NULL
// otherwise, if you do not know the APN for your network, you may
// either try the fairly common "internet" for the APN (and leave the
// username and password NULL), or you may leave all three as NULL and then
// a lookup will be attempted for a small number of known networks
// (see APN_db.h in mbed-os/features/netsocket/cellular/utils).
#define APN         NULL
#define USERNAME    NULL
#define PASSWORD    NULL

// If you wish, you may add a mobile phone number here, e.g.
// #define DESTINATION_SMS_NUMBER "+441234567901"
// Note: no spaces are allowed.
// If you do so this program will send a text message to that
// number and then wait for a reply
#define DESTINATION_SMS_NUMBER ""

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

// The user button
volatile bool buttonPressed = false;

static void good() {
    ledGreen = 0;
    ledBlue = 1;
    ledRed = 1;
}

static void bad() {
    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;
}

static void event() {
    ledBlue = 0;
    ledRed = 1;
    ledGreen = 1;
}

static void pulseEvent() {
    event();
    wait_ms(500);
    good();
}

static void ledOff() {
    ledBlue = 1;
    ledRed = 1;
    ledGreen = 1;
}

// See section 12.1 of the u-blox-ATCommands_Manual_(UBX-13002752).pdf
static void printCallForwardingResponse(char * buf)
{
    int numValues, a, b;
    char number[32];

    numValues = sscanf(buf, "+CCFC: %d,%d,\"%32[^\"][\"]", &a, &b, number);
    if (numValues > 0) {
        if (a > 0) {
            printf("Calling Forwarding is active");
        } else {
            printf("Calling Forwarding is not active");
        }
        if (numValues > 1) {
            if (b > 0) {
                if (b & 0x01) {
                    printf(" for voice");
                }
                if (b & 0x02) {
                    printf(" for data");
                }
                if (b & 0x04) {
                    printf(" for fax");
                }
                if (b & 0x08) {
                    printf(" for SMS");
                }
                if (b & 0x10) {
                    printf(" for data circuit sync");
                }
                if (b & 0x20) {
                    printf(" for data circuit async");
                }
                if (b & 0x40) {
                    printf(" for dedicated packet access");
                }
                if (b & 0x80) {
                    printf(" for dedicated PAD access");
                }
            }
        }
        if (numValues > 2) {
            printf(" for %s.\n", number);
        } else {
            printf(".\n");
        }
    }
}

static void cbButton()
{
    buttonPressed = true;
    pulseEvent();
}

/* This example program for the u-blox C030 and C027 boards instantiates
 * the UbloxCellularDriverGen to do SMS, USSD and module file system operations.
 * Progress may be monitored with a serial terminal running at 9600 baud.
 * The LED on the C030 board will turn green when this program is
 * operating correctly, pulse blue when a USSD action is performed,
 * an SMS is sent or received or a file operation is completed,
 * and will turn red on a failure condition.
 */

int main()
{
    UbloxCellularDriverGen *driver = new UbloxCellularDriverGen();
    // If you need to debug the cellular driver, comment out the
    // instantiation above and uncomment the one below.
//    UbloxCellularDriverGen *driver = new UbloxCellularDriverGen(MDMTXD, MDMRXD,
//                                                                MBED_CONF_UBLOX_CELL_BAUD_RATE,
//                                                                true);
    char buf[1024];
    int x;
    int index[8];
    char number[32];
#ifdef TARGET_UBLOX_C027
    // No user button on C027
    InterruptIn userButton(NC);
#else
    InterruptIn userButton(SW0);
#endif
    
    // Attach a function to the user button
    userButton.rise(&cbButton);
    
    good();
    printf ("Starting up, please wait up to 180 seconds for network registration to complete...\n");
    if (driver->init(PIN)) {
        pulseEvent();

        // USSD OPERATIONS
        printf("=== USSD ===\n");
        printf("Sending *100#...\n");
        if (driver->ussdCommand("*100#", buf, sizeof (buf))) {
            pulseEvent();
            printf("Answer was %s.\n", buf);
        }
        printf("Getting the status of call forwarding...\n");
        if (driver->ussdCommand("*#21#", buf, sizeof (buf))) {
            pulseEvent();
            printf("Answer was \"%s\", which means:\n", buf);
            printCallForwardingResponse(buf);
        }
        
        // FILE SYSTEM OPERATIONS
        printf("=== Module File System ===\n");
        strcpy (buf, "Hello world!"); 
        x = strlen(buf);
        printf("Writing \"%s\" (%d byte(s)) to file \"test.txt\" on the module's file system...\n", buf, x);
        x = driver->writeFile("test.txt", buf, strlen(buf));
        printf("%d byte(s) written.\n", x);
        if (x == (int) strlen(buf)) {
            pulseEvent();
            *buf = 0;
            printf("Reading the file back from the module's file system...\n");
            x = driver->readFile("test.txt", buf, sizeof(buf));
            printf("%d byte(s) read.\n", x);
            if (x > 0) {
                pulseEvent();
                printf("File \"test.txt\" contained \"%s\".\n", buf);
            }
            printf("Deleting the file...\n");
            if (driver->delFile("test.txt")) {
                pulseEvent();
                printf("File deleted.\n");
            }
        }
        
        // SMS OPERATIONS
        printf("=== SMS ===\n");
        if (strlen(DESTINATION_SMS_NUMBER) > 0) {
            printf("Sending a text message to %s...\n", DESTINATION_SMS_NUMBER);
            if (driver->smsSend(DESTINATION_SMS_NUMBER, "This is your u-blox mbed board calling.")) {
                pulseEvent();
                printf("Text message sent.\n");
            }
        } else {
            printf("No destination SMS number has been defined (edit DESTINATION_SMS_NUMBER in this file to add one).\n");
        }
        
        printf("Waiting for SMS messages to arrive in a loop (until the user button is pressed on C030 or forever on C027)...\n");
        while (!buttonPressed) {
            x = driver->smsList("REC UNREAD", index, sizeof(index));
            for (int y = 0; y < x; y++) {
                *number = 0;
                if (driver->smsRead(index[y], number, buf, sizeof (buf))) {
                    pulseEvent();
                    printf("Received text message from %s: \"%s\"\n", number, buf);
                    printf("Replying...\n");
                    if (driver->smsSend(number, "mbed board responding: hello there!")) {
                        pulseEvent();
                        printf("Reply sent.\n");
                    }
                    printf("Deleting text message from SIM...\n");
                    if (driver->smsDelete(index[y])) {
                        pulseEvent();
                        printf("Message deleted.\n");
                    }
                }
            }
            wait_ms(2500);
#ifndef TARGET_UBLOX_C027
            printf("[Checking if user button has been pressed]\n");
#endif
        }
        
        pulseEvent();
        printf("User button was pressed, stopping...\n");
        driver->deinit();
        ledOff();
        printf("Stopped.\n");        
    } else {
        bad();
        printf("Unable to initialise the driver.\n");
    }
}

// End Of File