/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "battery_gauge_bq27441.h"

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);

/* This example program for the u-blox C030 board instantiates
 * the BQ27441 battery gauge and performs a few example calls to
 * the battery gauge API.  Progress may be monitored with a
 * serial terminal running at 9600 baud.  The LED on the C030
 * board will turn green when this program is operating correctly
 * and will turn red if there is a failure.
 */

int main()
{
    I2C i2C(I2C_SDA_B, I2C_SCL_B);
    BatteryGaugeBq27441 gauge;
    int32_t reading;
    bool stop = false;

    printf ("Starting up...\n");
    if (gauge.init(&i2C)) {        
        printf ("BQ27441 battery gauge chip is initialised.\n");
#ifdef TARGET_UBLOX_C030        
        if (gauge.disableBatteryDetect()) {
            printf ("Battery detection input disabled (it is not connected on C030).\n");
        }
#endif
        while (!stop) {
            if (gauge.isBatteryDetected()) {
                if (!gauge.isGaugeEnabled()) {
                    gauge.enableGauge();
                }
                if (gauge.isGaugeEnabled()) {
                    // All is good, gauging is enabled, take a few readings
                    ledGreen = 0;
                    ledRed = 1;
                    
                    if (gauge.getRemainingPercentage(&reading)) {
                        printf("Remaining battery percentage: %d%%.\n", (int) reading);
                    } else {
                        ledGreen = 1;
                        ledRed = 0;
                        wait_ms(1000);
                    }
                    
                    if (gauge.getRemainingCapacity(&reading)) {
                        printf("Remaining battery capacity: %.3f Ah.\n", ((float) reading) / 1000);
                    } else {
                        ledGreen = 1;
                        ledRed = 0;
                        wait_ms(1000);
                    }
                    
                    if (gauge.getVoltage(&reading)) {
                        printf("Battery voltage: %.3f V.\n", ((float) reading) / 1000);
                    } else {
                        ledGreen = 1;
                        ledRed = 0;
                        wait_ms(1000);
                    }
                    
                    if (gauge.getCurrent(&reading)) {
                        printf("Current drawn from battery: %.3f A.\n", ((float) reading) / 1000);
                    } else {
                        ledGreen = 1;
                        ledRed = 0;
                        wait_ms(1000);
                    }
                    
                    if (gauge.getTemperature(&reading)) {
                        printf("BQ27441 chip temperature: %d C.\n", (int) reading);
                    } else {
                        ledGreen = 1;
                        ledRed = 0;
                        wait_ms(1000);
                    }
					
                    if (gauge.getPower(&reading)) {
                        printf("Power consumption : %.3f W.\n", ((float) reading) / 1000);
                    } else {
                        ledGreen = 1;
                        ledRed = 0;
                        wait_ms(1000);
                    }
                    
                } else {
                    printf("Battery gauge could not be enabled.\n");
                    stop = true;
                }
            } else {
                ledGreen = 1;
                ledRed = 0;
                printf("Battery not detected.\n");
            }
                
            wait_ms(5000);
        }
    }
    
    ledGreen = 1;
    ledRed = 0;
    printf("Should never get here: do you have a battery connected?\n");
    MBED_ASSERT(false);
}

// End Of File