/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "battery_charger_bq24295.h"

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);

/* This example program for the u-blox C030 board instantiates
 * the BQ24295 battery charger interface and performs a few
 * example calls to the battery charger API, displaying information
 * about the state of the charger chip.  Progress may be
 * monitored with a serial terminal running at 9600 baud.  The
 * LED on the C030 board will turn green when this program is
 * operating correctly and will turn red if there is a failure.
 *
 * To understand the meaning of the API calls it is necessary to
 * refer to the data sheet for the TI BQ24295 chip.
 * IMPORTANT it is NOT necessary to use the BQ24295 class in order
 * to charge a battery connected to the C030 board, that will happen
 * automatically. The BQ24295 class is only required if the battery
 * charger is to be configured or monitored.
 */

int main()
{
    I2C i2C(I2C_SDA_B, I2C_SCL_B);
    BatteryChargerBq24295 charger;
    BatteryChargerBq24295::ChargerState state = BatteryChargerBq24295::CHARGER_STATE_UNKNOWN;
    char fault = 0;

    printf ("Starting up...\n");
    if (charger.init(&i2C)) {
        
        printf ("BQ24295 battery charger chip is initialised.\n");
        if (charger.isExternalPowerPresent()) {
            printf ("External power is present.\n");
        } else {
            printf ("External power is NOT present.\n");
        }
        if (charger.areInputLimitsEnabled()) {
            printf ("Input limits are enabled.\n");
        } else {
            printf ("Input limits are disabled.\n");
        }
        if (charger.isChargingEnabled()) {
            printf ("Charging is already enabled.\n");
        } else {
            printf ("Charging is disabled.\n");
        }
        if (charger.isOtgEnabled()) {
            printf ("OTG is enabled.\n");
        } else {
            printf ("OTG is disabled.\n");
        }
        if (charger.setWatchdog(160)) {
            printf ("Watchdog set to 160 seconds.\n");
        }
        
        while (state < BatteryChargerBq24295::MAX_NUM_CHARGER_STATES) {
            // Feed the watchdog timer to keep the charger chip in
            // Host mode while we are monitoring it. If we do not
            // feed the watchdog it will expire, which does no
            // harm at all (the chip is now simply in default mode again
            // and will return to host mode if we write to it), but
            // it will appear as a fault state.
            charger.feedWatchdog();
            fault = charger.getChargerFaults();
            if (fault == BatteryChargerBq24295::CHARGER_FAULT_NONE) {
                ledGreen = 0;
                ledRed = 1;
                state = charger.getChargerState();
                switch (state) {
                    case BatteryChargerBq24295::CHARGER_STATE_UNKNOWN:
                        printf("Charger state is not (yet) known.\n");
                        break;
                    case BatteryChargerBq24295::CHARGER_STATE_DISABLED:
                        printf("Charger state: disabled.\n");
                        break;
                    case BatteryChargerBq24295::CHARGER_STATE_NO_EXTERNAL_POWER:
                        printf("Charger state: no external power.\n");
                        break;
                    case BatteryChargerBq24295::CHARGER_STATE_NOT_CHARGING:
                        printf("Charger state: not charging.\n");
                        break;
                    case BatteryChargerBq24295::CHARGER_STATE_PRECHARGE:
                        printf("Charger state: pre-charge.\n");
                        break;
                    case BatteryChargerBq24295::CHARGER_STATE_FAST_CHARGE:
                        printf("Charger state: fast charge.\n");
                        break;
                    case BatteryChargerBq24295::CHARGER_STATE_COMPLETE:
                        printf("Charger state: charging complete.\n");
                        break;
                    default:
                        printf("Unknown charger state (%d).\n", state);
                    break;
                }
            } else {
                ledGreen = 1;
                ledRed = 0;
                if (fault & BatteryChargerBq24295::CHARGER_FAULT_THERMISTOR_TOO_HOT) {
                    printf("Charger fault: thermistor is too hot.\n");
                }
                if (fault & BatteryChargerBq24295::CHARGER_FAULT_THERMISTOR_TOO_COLD) {
                    printf("Charger fault: thermistor is too cold.\n");
                }
                if (fault & BatteryChargerBq24295::CHARGER_FAULT_BATTERY_OVER_VOLTAGE) {
                    printf("Charger fault: battery over-voltage.\n");
                }
                // The testing of the fault bit map looks slightly strange here.
                // This is because the definition of the fault bits in the
                // BQ24295 chip means that Charger Timer Expired overlaps
                // input fault and thermal shutdown.  See the definition of the
                // meaning of REG09 in the BQ24295 data sheet for more information.
                if ((fault & BatteryChargerBq24295::CHARGER_FAULT_INPUT_FAULT) &&
                    !(fault & BatteryChargerBq24295::CHARGER_FAULT_THERMAL_SHUTDOWN)) {
                    printf("Charger fault: input fault.\n");
                }
                if ((fault & BatteryChargerBq24295::CHARGER_FAULT_THERMAL_SHUTDOWN) &&
                    !(fault & BatteryChargerBq24295::CHARGER_FAULT_INPUT_FAULT)) {
                    printf("Charger fault: thermal shutdown.\n");
                }
                if ((fault & BatteryChargerBq24295::CHARGER_FAULT_CHARGE_TIMER_EXPIRED) ==
                    BatteryChargerBq24295::CHARGER_FAULT_CHARGE_TIMER_EXPIRED) {
                    printf("Charger fault: charge timer expired.\n");
                }
                if (fault & BatteryChargerBq24295::CHARGER_FAULT_OTG) {
                    printf("Charger fault: OTG charging fault.\n");
                }
                if (fault & BatteryChargerBq24295::CHARGER_FAULT_WATCHDOG_EXPIRED) {
                    printf("Charger fault: charger watchdog expired.\n");
                }
            }
            wait_ms(5000);
        }
    }
    
    ledGreen = 1;
    ledRed = 0;
    printf("Should never get here.\n");
    MBED_ASSERT(false);
}

// End Of File