#include "mbed.h"
#include "rtos.h"
#include "C12832.h"
#include "nsdl_support.h"
#include "dbg.h"
// Include various resources
#include "temperature.h"
#include "light.h"
#include "gps_res.h"
#include "relay.h"

#include "UDPSocket.h"
#include "Endpoint.h"

//------------------------------------------------------------------------------------
// You need to configure these cellular modem / SIM parameters.
// These parameters are ignored for LISA-C200 variants and can be left NULL.
//------------------------------------------------------------------------------------
#include "MDM.h"
#include "GPS.h"
//! Set your secret SIM pin here (e.g. "1234"). Check your SIM manual.
#define SIMPIN      NULL
/*! The APN of your network operator SIM, sometimes it is "internet" check your 
    contract with the network operator. You can also try to look-up your settings in 
    google: https://www.google.de/search?q=APN+list */
#define APN         NULL
//! Set the user name for your APN, or NULL if not needed
#define USERNAME    NULL
//! Set the password for your APN, or NULL if not needed
#define PASSWORD    NULL 
//------------------------------------------------------------------------------------

static C12832 lcd(D11, D13, D12, D7, D10);

// ****************************************************************************
// Configuration section

// NSP configuration
/* Change this IP address to that of your NanoService Platform installation */
static const char* NSP_ADDRESS = "nanoservice-demo.mbed.org"; /* demo NSP, web interface at http://nanoservice-demo.mbed.org */ 
static const int NSP_PORT = 5683;
char endpoint_name[16] = "mbed-cellular1";
uint8_t ep_type[] = {"mbed_device"};
uint8_t lifetime_ptr[] = {"1200"};

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

static void nsp_init()
{
    server.init();
    server.bind(NSP_PORT);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d", NSP_ADDRESS, NSP_PORT);
    NSDL_DEBUG("NSDL Demo Portal\r\n"
               "  Website:  http://%s\r\n"
               "  Username: demo\r\n"
               "  Password: demo", NSP_ADDRESS);
    
    lcd.locate(0,22);
    lcd.printf("EP name:%s\n", endpoint_name);
}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    NSDL_DEBUG("Creating resources");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Static resources
    #define _PT(str)   sizeof(str)-1, (uint8_t*)str
    #define _RSC(str)  (uint8_t*)str, sizeof(str)-1
    nsdl_create_static_resource(resource_ptr, _PT("dev/mfg"), 0, 0, _RSC("Sensinode / u-blox"));
    nsdl_create_static_resource(resource_ptr, _PT("dev/mdl"), 0, 0, _RSC("NSDL-C cellular mbed device"));

    // Dynamic resources
    create_temperature_resource(resource_ptr);
    create_light_resource(resource_ptr);
    create_gps_resource(resource_ptr);
    create_relay_resource(resource_ptr);

        /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
        printf("NSP registering failed\r\n");
    else
        printf("NSP registering OK\r\n");
    nsdl_clean_register_endpoint(&endpoint_ptr);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

// ****************************************************************************
// Program entry point
void ndslTask(void const* argument) 
{
    MDMRtos<MDMSerial> mdm;
    
    //mdm.setDebug(4); // enable this for debugging issues 
    if (!mdm.connect(SIMPIN, APN,USERNAME,PASSWORD))
        return;

    lcd.locate(0,11);

    // Initialize NSP node
    nsp_init();
    
    // Initialize NSDL stack
    nsdl_init();
    
    // Create NSDL resources
    create_resources();
    
    // Run the NSDL event loop (never returns)
    nsdl_event_loop();

    mdm.disconnect();
    mdm.powerOff();
}

void gpsTask(void const* argument) 
{
    GPSI2C gps;
    char buf[256];
    while (1) {
        int ret = gps.getMessage(buf, sizeof(buf));
        if (ret > 0) {
            int len = LENGTH(ret);
            if ((PROTOCOL(ret) == GPSParser::NMEA) && (len > 6))
            {
                if (!strncmp("$GPGLL", buf, 6)) {
                    double la = 0, lo = 0;
                    char ch;
                    if (gps.getNmeaAngle(1,buf,len,la) && 
                        gps.getNmeaAngle(3,buf,len,lo) && 
                        gps.getNmeaItem(6,buf,len,ch) && ch == 'A')
                    {
                        printf("gps update lat %.6f lon %.6f\r\n", la, lo);
                        gps_resource_set(la,lo);
                    }
                }
            }
        } else
            Thread::wait(100);
    }
}

int main()
{
    Serial pc(USBTX, USBRX);
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("mbed NanoService demo");
    NSDL_DEBUG("mbed NanoService Example App 0.1\r\n");

    Thread task1(ndslTask, NULL, osPriorityNormal, 4096);
    Thread task2(gpsTask,  NULL, osPriorityNormal, 2048);
     
    DigitalOut led(LED1);
    while(true) {
        led = !led;
        wait_ms(500);
    }
}


