#include "DeviceIntegration.h"
#include <stdio.h>
#include "ComposedRecord.h"
#include "CharValue.h"
#include "IntegerValue.h"

DeviceIntegration::DeviceIntegration(SmartRest& client, SmartRestTemplate& tpl, long& deviceId, DeviceInfo& deviceInfo) :
    _client(client),
    _tpl(tpl),
    _deviceId(deviceId),
    _deviceInfo(deviceInfo)
{
    _init = false;
}

bool DeviceIntegration::init()
{
    if (_init)
        return false;
    
    // get device by identity
    // Usage: 100,<SERIAL/NR>
    if (!_tpl.add("10,100,GET,/identity/externalIds/c8y_Serial/%%,,application/vnd.com.nsn.cumulocity.externalId+json,%%,STRING,\r\n"))
        return false;

    // get device id from identity
    // Response: 200,<DEVICE/ID>
    if (!_tpl.add("11,200,\"$.managedObject\",,\"$.id\"\r\n"))
        return false;

    // Create device
    // Usage: 101,<SERIAL/NR>
    if (!_tpl.add("10,101,POST,/inventory/managedObjects,application/vnd.com.nsn.cumulocity.managedObject+json,application/vnd.com.nsn.cumulocity.managedObject+json,%%,STRING,\"{\"\"name\"\":\"\"Mbed Test Device\"\",\"\"type\"\":\"\"com_ublox_C027_REV-A\"\",\"\"c8y_Hardware\"\":{\"\"revision\"\":\"\"1\"\",\"\"model\"\":\"\"Ublox C027\"\",\"\"serialNumber\"\":\"\"%%\"\"},\"\"c8y_SupportedMeasurements\"\":[\"\"c8y_SignalStrength\"\",\"\"c8y_TemperatureMeasurement\"\",\"\"c8y_AnalogMeasurement\"\",\"\"c8y_MotionMeasurement\"\",\"\"c8y_AnalogMeasurement\"\"],\"\"c8y_RequiredAvailability\"\":{ \"\"responseInterval\"\":15},\"\"c8y_IsDevice\"\":{}}\"\r\n"))
        return false;

    // Get device id
    // Response: 201,<DEVICE/ID>
    if (!_tpl.add("11,201,,\"$.c8y_IsDevice\",\"$.id\"\r\n"))
        return false;

    // Insert global ID
    // Usage: 102,<DEVICE/ID>,<SERIAL/NR>
    if (!_tpl.add("10,102,POST,/identity/globalIds/%%/externalIds,application/vnd.com.nsn.cumulocity.externalId+json,application/vnd.com.nsn.cumulocity.externalId+json,%%,UNSIGNED STRING,\"{\"\"type\"\":\"\"c8y_Serial\"\",\"\"externalId\"\":\"\"%%\"\"}\"\r\n"))
        return false;

    // Update IMEI, CellId and iccid
    // Usage: 103,<DEVICE/ID>,<IMEI>,<CELL/ID>,<ICCID>
    if (!_tpl.add("10,103,PUT,/inventory/managedObjects/%%,application/vnd.com.nsn.cumulocity.managedObject+json,application/vnd.com.nsn.cumulocity.managedObject+json,%%,UNSIGNED STRING STRING STRING,\"{\"\"c8y_Mobile\"\":{\"\"imei\"\":\"\"%%\"\",\"\"cellId\"\":\"\"%%\"\",\"\"iccid\"\":\"\"%%\"\"}}\"\r\n"))
        return false;

    _init = true;
    _deviceId = 0;
    return true;
}

bool DeviceIntegration::integrate()
{
    if (_deviceId != 0)
        return false;

    // template bootstrapping process
    if (_client.bootstrap(_tpl) != SMARTREST_SUCCESS) {
        puts("Template bootstrap failed.");
        return false;
    }

    if ((!deviceExisting()) && ((!createDevice()) || (!addGlobalIdentifier())))
        return false;

    if (!updateDevice())
        return false;

    return true;
}

bool DeviceIntegration::deviceExisting()
{
    ComposedRecord record;
    ParsedRecord received;

    IntegerValue msgId(100);
    CharValue imei(_deviceInfo.imei());
    if ((!record.add(msgId)) || (!record.add(imei)))
        return false;

    if (_client.send(record) != SMARTREST_SUCCESS) {
        puts("Send failed.");
        _client.stop();
        return false;
    }

    if (_client.receive(received) != SMARTREST_SUCCESS) {
        puts("No device found.");
        _client.stop();
        return false;
    }
    _client.stop();

    if (received.values() == 0) {
        puts("Received no values.");
        return false;
    }

    if (received.value(0).integerValue() == 50) {
        return false;
    }
    
    if (received.value(0).integerValue() != 200) {
        puts("Bad response.");
        return false;
    }

    _deviceId = received.value(2).integerValue();
    
    return true;
}

bool DeviceIntegration::createDevice()
{
    ComposedRecord record;
    ParsedRecord received;

    puts("Creating device...");

    IntegerValue msgId(101);
    CharValue imei(_deviceInfo.imei());
    if ((!record.add(msgId)) || (!record.add(imei)))
        return false;

    if (_client.send(record) != SMARTREST_SUCCESS) {
        puts("Send failed.");
        _client.stop();
        return 0;
    }

    if (_client.receive(received) != SMARTREST_SUCCESS) {
        puts("No device found.");
        _client.stop();
        return false;
    }
    _client.stop();

    if (received.values() != 3) {
        puts("Bad received data.");
        return false;
    }
    
    if (received.value(0).integerValue() != 201) {
        puts("Bad received data.");
        return false;
    }

    _deviceId = received.value(2).integerValue();
    return true;
}

bool DeviceIntegration::addGlobalIdentifier()
{
    ComposedRecord record;
    ParsedRecord received;

    puts("Adding global identifier...");

    IntegerValue msgId(102);
    IntegerValue deviceId(_deviceId);
    CharValue imei(_deviceInfo.imei());
    if ((!record.add(msgId)) || (!record.add(deviceId)) || (!record.add(imei)))
        return false;

    if (_client.send(record) != SMARTREST_SUCCESS) {
        puts("Sending failed.");
        _client.stop();
        return false;
    }

    if (_client.receive(received) != SMARTREST_SUCCESS) {
        puts("Failed.");
        _client.stop();
        return false;
    }
    _client.stop();

    if (received.values() != 3) {
        puts("Received bad data.");
        return false;
    }
    
    if (received.value(0).integerValue() != 200) {
        puts("Received bad data.");
        return false;
    }

    return true;
}

bool DeviceIntegration::updateDevice()
{
    ComposedRecord record;
    ParsedRecord received;

    IntegerValue msgId(103);
    IntegerValue deviceId(_deviceId);
    CharValue imei(_deviceInfo.imei());
    CharValue cellId(_deviceInfo.cellId());
    CharValue iccid(_deviceInfo.iccid());
    if ((!record.add(msgId)) || (!record.add(deviceId)) || (!record.add(imei)) || (!record.add(cellId)) || (!record.add(iccid)))
        return false;

    if (_client.send(record) != SMARTREST_SUCCESS) {
        puts("Send failed.");
        _client.stop();
        return false;
    }

    if (_client.receive(received) != SMARTREST_SUCCESS) {
        puts("Update failed.");
        _client.stop();
        return false;
    }
    _client.stop();

    if (received.values() != 3) {
        puts("Bad received data.");
        return false;
    }
    
    if (received.value(0).integerValue() != 201) {
        puts("Bad received data.");
        return false;
    }

    return true;
}
