#include "mbed.h"

//------------------------------------------------------------------------------------
/* This demo has the purpose to provide some examples about HTTP procedures.
   In details are implemented the following actions referring freely available online services:
   
   - HTTP and HTTPS HEAD command      -> www.developer.mbed.org
   - HTTP and HTTPS GET command       -> www.developer.mbed.org
   - HTTP and HTTPS DELETE command    -> www.httpbin.org
   - HTTP and HTTPS PUT command       -> www.httpbin.org
   - HTTP and HTTPS POST file command -> www.posttestserver.com
   - HTTP and HTTPS POST data command -> www.posttestserver.com
   
   P.S.: Please don't use confidential data when trying out the reported examples.
         As default, the httpInData variable is set to "Hello World" (see below).
*/
#include "MDM.h"
//------------------------------------------------------------------------------------
// You need to configure these cellular modem / SIM parameters.
// These parameters are ignored for LISA-C200 variants and can be left NULL.
//------------------------------------------------------------------------------------
//! Set your secret SIM pin here (e.g. "1234"). Check your SIM manual.
#define SIMPIN      NULL
/*! The APN of your network operator SIM, sometimes it is "internet" check your 
    contract with the network operator. You can also try to look-up your settings in 
    google: https://www.google.de/search?q=APN+list */
#define APN         NULL
//! Set the user name for your APN, or NULL if not needed
#define USERNAME    NULL
//! Set the password for your APN, or NULL if not needed
#define PASSWORD    NULL 
//------------------------------------------------------------------------------------
// You need to configure the buffer data size 
//------------------------------------------------------------------------------------
//! Set the buffer size to 2048 bytes
//#define LARGE_DATA
//! Set the buffer size to 1024 bytes
#define MEDIUM_DATA
//------------------------------------------------------------------------------------

int main(void)
{
    int ret;
#if defined(LARGE_DATA)
    printf("Set the buffer size to 2048 bytes\r\n");
    char buf[2048] = "";
#elif defined(MEDIUM_DATA)
    printf("Set the buffer size to 1024 bytes\r\n");
    char buf[1024] = "";
#else
    printf("Set the buffer size to 512 bytes\r\n");
    char buf[512] = "";
#endif
    //give time to module for powering up (ms)
    wait_ms(1000);
    // Create the modem object
    MDMSerial mdm;
    //mdm.setDebug(4); // enable this for debugging issues
    // initialize the modem 
    MDMParser::DevStatus devStatus = {};
    MDMParser::NetStatus netStatus = {};
    bool mdmOk = mdm.init(SIMPIN, &devStatus);
    mdm.dumpDevStatus(&devStatus);
    if (mdmOk) {
#if 0
        // file system API
        const char* filename = "File";
        char buf[] = "Hello World";
        printf("writeFile \"%s\"\r\n", buf);
        if (mdm.writeFile(filename, buf, sizeof(buf)))
        {
            memset(buf, 0, sizeof(buf));
            int len = mdm.readFile(filename, buf, sizeof(buf));
            if (len >= 0) 
                printf("readFile %d \"%.*s\"\r\n", len, len, buf);
            mdm.delFile(filename);
        }
#endif

        // wait until we are connected
        mdmOk = mdm.registerNet(&netStatus);
        mdm.dumpNetStatus(&netStatus);
    }
    if (mdmOk)
    {
        // http://www.geckobeach.com/cellular/secrets/gsmcodes.php
        // http://de.wikipedia.org/wiki/USSD-Codes
        const char* ussd = "*130#"; // You may get answer "UNKNOWN APPLICATION"
        printf("Ussd Send Command %s\r\n", ussd);
        ret = mdm.ussdCommand(ussd, buf);
        if (ret > 0) 
            printf("Ussd Got Answer: \"%*s\"\r\n", ret, buf);

        // join the internet connection 
        MDMParser::IP ip = mdm.join(APN,USERNAME,PASSWORD);
        if (ip != NOIP)
        {
            mdm.dumpIp(ip);
            
            printf("HTTP and HTTPS management\r\n");
            char httpInData[] = "Hello World";      //input data for some HTTP commands
            char httpInFile[48] = "http_in.txt";    //file where saving HTTP commands data
            char httpOutFile[48] = "http_out.txt";  //file where saving HTTP commands results
            
            int httpProfile = mdm.httpFindProfile();  //get the HTTP profile identifier
            if (httpProfile >= 0)
            {
                printf("Make an HTTP and HTTPS HEAD Request\r\n");
                if(mdm.httpResetProfile(httpProfile))
                {
                    if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SERVER_NAME,"developer.mbed.org"))
                    {
                        mdm.httpSetBlocking(httpProfile, 10000);  //timeout is 10000 ms: not blocking procedure
                        
                        //HTTP HEAD command
                        ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_HEAD,
                                              "/media/uploads/mbed_official/hello.txt",
                                              httpOutFile,NULL,NULL,NULL,buf,sizeof(buf));
                        if (ret > 0)
                            printf("HTTP HEAD result \"%*s\"\r\n", ret, buf);
                        
                        if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SECURE,"1"))  //HTTP Secure option enabled
                        {
                            //HTTPS HEAD command
                            ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_HEAD,
                                                  "/media/uploads/mbed_official/hello.txt",
                                                  httpOutFile,NULL,NULL,NULL,buf,sizeof(buf));
                            if (ret > 0)
                                printf("HTTPS HEAD result \"%*s\"\r\n", ret, buf);
                        } else {
                            printf("Abnormal condition during the set of HTTP secure option\r\n");
                        }
                        
                        //delete the HTTP output file
                        mdm.delFile(httpOutFile);
                    } else {
                        printf("Abnormal condition during the set of the HTTP server name\r\n");
                    }
                } else {
                    printf("Abnormal condition during the reset of HTTP profile %d\r\n", httpProfile);
                }
                
                printf("Make an HTTP and HTTPS GET Request\r\n");
                if(mdm.httpResetProfile(httpProfile))
                {
                    if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SERVER_NAME,"developer.mbed.org"))
                    {
                        mdm.httpSetBlocking(httpProfile, 10000);  //timeout is 10000 ms: not blocking procedure
                        
                        //HTTP GET command
                        ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_GET,
                                              "/media/uploads/mbed_official/hello.txt",
                                              httpOutFile,NULL,NULL,NULL,buf,sizeof(buf));
                        if (ret > 0)
                            printf("HTTP GET result \"%*s\"\r\n", ret, buf);
                        
                        if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SECURE,"1"))  //HTTP Secure option enabled
                        {
                            //HTTPS GET command
                            ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_GET,
                                                  "/media/uploads/mbed_official/hello.txt",
                                                  httpOutFile,NULL,NULL,NULL,buf,sizeof(buf));
                            if (ret > 0)
                                printf("HTTPS GET result \"%*s\"\r\n", ret, buf);
                        } else {
                            printf("Abnormal condition during the set of HTTP secure option\r\n");
                        }
                        
                        //delete the HTTP output file
                        mdm.delFile(httpOutFile);
                    } else {
                        printf("Abnormal condition during the set of the HTTP server name\r\n");
                    }
                } else {
                    printf("Abnormal condition during the reset of HTTP profile %d\r\n", httpProfile);
                }
                
                printf("Make an HTTP and HTTPS DELETE Request\r\n");
                if(mdm.httpResetProfile(httpProfile))
                {
                    if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SERVER_NAME,"httpbin.org"))
                    {
                        mdm.httpSetBlocking(httpProfile, 10000);  //timeout is 10000 ms: not blocking procedure
                        
                        //HTTP DELETE command
                        ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_DELETE,"/delete", \
                                              httpOutFile,NULL,NULL,NULL,buf,sizeof(buf));
                        if (ret > 0)
                            printf("HTTP DELETE result \"%*s\"\r\n", ret, buf);
                        
                        if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SECURE,"1"))  //HTTP Secure option enabled
                        {
                            //HTTPS DELETE command
                            ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_DELETE,"/delete", \
                                                  httpOutFile,NULL,NULL,NULL,buf,sizeof(buf));
                            if (ret > 0)
                                printf("HTTPS DELETE result \"%*s\"\r\n", ret, buf);
                        } else {
                            printf("Abnormal condition during the set of HTTP secure option\r\n");
                        }
                        
                        //delete the HTTP output file
                        mdm.delFile(httpOutFile);
                        
                    } else {
                        printf("Abnormal condition during the set of the HTTP server name\r\n");
                    }
                } else {
                    printf("Abnormal condition during the reset of HTTP profile %d\r\n", httpProfile);
                }
                
                printf("Make an HTTP and HTTPS PUT Request\r\n");
                if(mdm.httpResetProfile(httpProfile))
                {
                    if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SERVER_NAME,"httpbin.org"))
                    {
                        if(mdm.writeFile(httpInFile,httpInData,sizeof(httpInData)))
                        {
                            mdm.httpSetBlocking(httpProfile, 10000);  //timeout is 10000 ms: not blocking procedure
                            
                            //HTTP PUT command
                            ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_PUT,"/put", \
                                                  httpOutFile,httpInFile,NULL,NULL,buf,sizeof(buf));
                            if (ret > 0)
                                printf("HTTP PUT result \"%*s\"\r\n", ret, buf);
                            
                            if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SECURE,"1"))  //HTTP Secure option enabled
                            {
                                //HTTPS PUT command
                                ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_PUT,"/put", \
                                                      httpOutFile,httpInFile,NULL,NULL,buf,sizeof(buf));
                                if (ret > 0)
                                    printf("HTTPS PUT result \"%*s\"\r\n", ret, buf);
                            } else {
                                printf("Abnormal condition during the set of HTTP secure option\r\n");
                            }
                            
                            //delete the HTTP input file
                            mdm.delFile(httpInFile);
                            
                            //delete the HTTP output file
                            mdm.delFile(httpOutFile);
                        } else {
                            printf("Abnormal condition during the writing of the HTTP input file\r\n");
                        }
                    } else {
                        printf("Abnormal condition during the set of the HTTP server name\r\n");
                    }
                } else {
                    printf("Abnormal condition during the reset of HTTP profile %d\r\n", httpProfile);
                }
                
                printf("Make an HTTP and HTTPS POST file command Request\r\n");
                if(mdm.httpResetProfile(httpProfile))
                {
                    if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SERVER_NAME,"posttestserver.com"))
                    {   
                        if(mdm.writeFile(httpInFile,httpInData,sizeof(httpInData)))
                        {
                            mdm.httpSetBlocking(httpProfile, 10000);  //timeout is 10000 ms: not blocking procedure
                            
                            //HTTP POST file command
                            ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_POST_FILE,"/post.php", \
                                                  httpOutFile,httpInFile,1,NULL,buf,sizeof(buf));
                            if (ret > 0)
                                printf("HTTP POST file command result \"%*s\"\r\n", ret, buf);
                            
                            if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SECURE,"1"))  //HTTP Secure option enabled
                            {
                                //HTTPS POST file command
                                ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_POST_FILE,"/post.php", \
                                                      httpOutFile,httpInFile,1,NULL,buf,sizeof(buf));
                                if (ret > 0)
                                    printf("HTTPS POST file command result \"%*s\"\r\n", ret, buf);
                            } else {
                                printf("Abnormal condition during the set of HTTP secure option\r\n");
                            }
                            //delete the HTTP input file
                            mdm.delFile(httpInFile);
                            
                            //delete the HTTP output file
                            mdm.delFile(httpOutFile);
                        } else {
                            printf("Abnormal condition during the writing of the HTTP input file\r\n");
                        }
                    } else {
                        printf("Abnormal condition during the set of the HTTP server name\r\n");
                    }
                } else {
                    printf("Abnormal condition during the reset of HTTP profile %d\r\n", httpProfile);
                }
                
                printf("Make an HTTP and HTTPS POST data command Request\r\n");
                if(mdm.httpResetProfile(httpProfile))
                {
                    if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SERVER_NAME,"posttestserver.com"))
                    {   
                        if(mdm.writeFile(httpInFile,httpInData,sizeof(httpInData)))
                        {
                            mdm.httpSetBlocking(httpProfile, 10000);  //timeout is 10000 ms: not blocking procedure
                            
                            //HTTP POST data command
                            ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_POST_DATA,"/post.php", \
                                                  httpOutFile,httpInData,1,NULL,buf,sizeof(buf));
                            if (ret > 0)
                                printf("HTTP POST data command result \"%*s\"\r\n", ret, buf);
                            
                            if (mdm.httpSetPar(httpProfile,MDMParser::HTTP_SECURE,"1"))  //HTTP Secure option enabled
                            {
                                //HTTPS POST data command
                                ret = mdm.httpCommand(httpProfile,MDMParser::HTTP_POST_DATA,"/post.php", \
                                                      httpOutFile,httpInData,1,NULL,buf,sizeof(buf));
                                if (ret > 0)
                                    printf("HTTPS POST data command result \"%*s\"\r\n", ret, buf);
                            } else {
                                printf("Abnormal condition during the set of HTTP secure option\r\n");
                            }
                            
                            //delete the HTTP input file
                            mdm.delFile(httpInFile);
                            
                            //delete the HTTP output file
                            mdm.delFile(httpOutFile);
                        } else {
                            printf("Abnormal condition during the writing of the HTTP input file\r\n");
                        }
                    } else {
                        printf("Abnormal condition during the set of the HTTP server name\r\n");
                    }
                } else {
                    printf("Abnormal condition during the reset of HTTP profile %d\r\n", httpProfile);
                }
                
                //free current HTTP profile
                mdm.httpFreeProfile(httpProfile);
            } else {
                printf("Abnormal condition: no available HTTP profiles\r\n");    
            }
            
            // disconnect  
            mdm.disconnect();
        }
    }
    else
    {
        //in this case the mdmOk is set to false
        printf("Please verify the SIM status and the network coverage\r\n");       
    }
    printf("Final loop: start\r\n");
    int maxLoops = 3;
    const int wait = 1000;
    for (int i=0; i < maxLoops; i++)
    {
        if (mdmOk)
        {
            // check the network status
            if (mdm.checkNetStatus(&netStatus)) {
                mdm.dumpNetStatus(&netStatus, fprintf, stdout);
            }
        }
        wait_ms(wait);
    }
    printf("Final loop: end\r\n");
    mdm.powerOff();
    return 0;
}
