
#ifndef TSC2046_H
#define TSC2046_H


/**
 * Texas Instruments Touch Screen Controller (TSC2046).
 */
class TSC2046 {
public:

    typedef struct {
        int32_t x;
        int32_t y;
        int32_t z;
    } touchCoordinate_t;

    /**
     * Constructor
     *
     * @param mosi SPI MOSI pin
     * @param miso SPI MISO pin
     * @param sck SPI SCK pin
     * @param cs chip-select pin
     */
    TSC2046(PinName mosi, PinName miso, PinName sck, PinName cs);

    /**
     * Read coordinates from the touch panel.
     *
     * Before calibrate() is called this function will return uncalibrated
     * values. If there is no touch active the coordinate values will be 0.
     *
     * @param coord pointer to coordinate object. The read coordinates will be
     * written to this object.
     */
    void read(touchCoordinate_t &coord);

    /**
     * Calibrate touch screen based on three reference points and
     * three actual readings. This means that the user must be presented
     * with three points (one at a time) and asked to press on these points
     * to get an actual reading.
     */
    void calibrate(touchCoordinate_t &ref1,
            touchCoordinate_t &ref2,
            touchCoordinate_t &ref3,
            touchCoordinate_t &scr1,
            touchCoordinate_t &scr2,
            touchCoordinate_t &scr3);

    /**
     * Reset a previous calibration (in order to do a new calibration)
     */
    void uncalibrate();


private:

    typedef struct {
        int64_t x;
        int64_t y;
    } calibPoint_t;

    typedef struct {
        int64_t An;
        int64_t Bn;
        int64_t Cn;
        int64_t Dn;
        int64_t En;
        int64_t Fn;
        int64_t Divider;
    } calibMatrix_t;

    SPI _spi;
    DigitalOut _cs;
    bool _calibrated;
    bool _initialized;
    calibMatrix_t _calibMatrix;

    void init();
    void readAndFilter(touchCoordinate_t &coord);
    int32_t getFilteredValue(int cmd);
    uint16_t spiTransfer(uint8_t cmd);



    int setCalibrationMatrix( calibPoint_t * displayPtr,
            calibPoint_t * screenPtr,
            calibMatrix_t * matrixPtr);
    int getDisplayPoint( calibPoint_t * displayPtr,
            calibPoint_t * screenPtr,
            calibMatrix_t * matrixPtr );

};

#endif


